"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const cloudfront = require("@aws-cdk/aws-cloudfront");
const route53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const targets = require("../lib");
test('use CloudFrontTarget partition hosted zone id mapping', () => {
    // GIVEN
    const stack = new core_1.Stack();
    // WHEN
    targets.CloudFrontTarget.getHostedZoneId(stack);
    // THEN
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toEqual({
        Mappings: {
            AWSCloudFrontPartitionHostedZoneIdMap: {
                'aws': {
                    zoneId: 'Z2FDTNDATAQYW2',
                },
                'aws-cn': {
                    zoneId: 'Z3RFFRIM2A3IF5',
                },
            },
        },
    });
});
test('use CloudFrontTarget hosted zone id mappings in nested stacks', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const nestedStackA = new core_1.NestedStack(stack, 'nestedStackA');
    const nestedStackB = new core_1.NestedStack(stack, 'nestedStackB');
    // WHEN
    targets.CloudFrontTarget.getHostedZoneId(nestedStackA);
    targets.CloudFrontTarget.getHostedZoneId(nestedStackB);
    // THEN
    for (let nestedStack of [nestedStackA, nestedStackB]) {
        expect(assert_1.SynthUtils.toCloudFormation(nestedStack)).toEqual({
            Mappings: {
                AWSCloudFrontPartitionHostedZoneIdMap: {
                    'aws': {
                        zoneId: 'Z2FDTNDATAQYW2',
                    },
                    'aws-cn': {
                        zoneId: 'Z3RFFRIM2A3IF5',
                    },
                },
            },
        });
    }
});
test('use CloudFront as record target', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const sourceBucket = new s3.Bucket(stack, 'Bucket');
    const distribution = new cloudfront.CloudFrontWebDistribution(stack, 'MyDistribution', {
        originConfigs: [
            {
                s3OriginSource: {
                    s3BucketSource: sourceBucket,
                },
                behaviors: [{ isDefaultBehavior: true }],
            },
        ],
    });
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName: 'test.public' });
    new route53.ARecord(zone, 'Alias', {
        zone,
        recordName: '_foo',
        target: route53.RecordTarget.fromAlias(new targets.CloudFrontTarget(distribution)),
    });
    // THEN
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        AliasTarget: {
            DNSName: { 'Fn::GetAtt': ['MyDistributionCFDistributionDE147309', 'DomainName'] },
            HostedZoneId: {
                'Fn::FindInMap': [
                    'AWSCloudFrontPartitionHostedZoneIdMap',
                    {
                        Ref: 'AWS::Partition',
                    },
                    'zoneId',
                ],
            },
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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