"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DockerImageAsset = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const ecr = require("@aws-cdk/aws-ecr");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line
const assets_1 = require("@aws-cdk/assets");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
/**
 * An asset that represents a Docker image.
 *
 * The image will be created in build time and uploaded to an ECR repository.
 *
 * @stability stable
 */
class DockerImageAsset extends core_2.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e;
        super(scope, id);
        // none of the properties use tokens
        validateProps(props);
        // resolve full path
        const dir = path.resolve(props.directory);
        if (!fs.existsSync(dir)) {
            throw new Error(`Cannot find image directory at ${dir}`);
        }
        // validate the docker file exists
        const file = path.join(dir, props.file || 'Dockerfile');
        if (!fs.existsSync(file)) {
            throw new Error(`Cannot find file at ${file}`);
        }
        const defaultIgnoreMode = core_1.FeatureFlags.of(this).isEnabled(cxapi.DOCKER_IGNORE_SUPPORT)
            ? core_1.IgnoreMode.DOCKER : core_1.IgnoreMode.GLOB;
        let ignoreMode = (_b = props.ignoreMode) !== null && _b !== void 0 ? _b : defaultIgnoreMode;
        let exclude = props.exclude || [];
        const ignore = path.join(dir, '.dockerignore');
        if (fs.existsSync(ignore)) {
            const dockerIgnorePatterns = fs.readFileSync(ignore).toString().split('\n').filter(e => !!e);
            exclude = [
                ...dockerIgnorePatterns,
                ...exclude,
                // Ensure .dockerignore is included no matter what.
                '!.dockerignore',
            ];
        }
        // Ensure the Dockerfile is included no matter what.
        exclude.push('!' + path.basename(file));
        // Ensure the cdk.out folder is not included to avoid infinite loops.
        const cdkout = (_d = (_c = core_1.Stage.of(this)) === null || _c === void 0 ? void 0 : _c.outdir) !== null && _d !== void 0 ? _d : 'cdk.out';
        exclude.push(cdkout);
        if (props.repositoryName) {
            core_1.Annotations.of(this).addWarning('DockerImageAsset.repositoryName is deprecated. Override "core.Stack.addDockerImageAsset" to control asset locations');
        }
        // include build context in "extra" so it will impact the hash
        const extraHash = {};
        if (props.extraHash) {
            extraHash.user = props.extraHash;
        }
        if (props.buildArgs) {
            extraHash.buildArgs = props.buildArgs;
        }
        if (props.target) {
            extraHash.target = props.target;
        }
        if (props.file) {
            extraHash.file = props.file;
        }
        if (props.repositoryName) {
            extraHash.repositoryName = props.repositoryName;
        }
        // add "salt" to the hash in order to invalidate the image in the upgrade to
        // 1.21.0 which removes the AdoptedRepository resource (and will cause the
        // deletion of the ECR repository the app used).
        extraHash.version = '1.21.0';
        const staging = new core_1.AssetStaging(this, 'Staging', {
            ...props,
            follow: (_e = props.followSymlinks) !== null && _e !== void 0 ? _e : toSymlinkFollow(props.follow),
            exclude,
            ignoreMode,
            sourcePath: dir,
            extraHash: Object.keys(extraHash).length === 0
                ? undefined
                : JSON.stringify(extraHash),
        });
        this.sourceHash = staging.assetHash;
        this.assetHash = staging.assetHash;
        const stack = core_1.Stack.of(this);
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: staging.relativeStagedPath(stack),
            dockerBuildArgs: props.buildArgs,
            dockerBuildTarget: props.target,
            dockerFile: props.file,
            sourceHash: staging.assetHash,
        });
        this.repository = ecr.Repository.fromRepositoryName(this, 'Repository', location.repositoryName);
        this.imageUri = location.imageUri;
    }
}
exports.DockerImageAsset = DockerImageAsset;
_a = JSII_RTTI_SYMBOL_1;
DockerImageAsset[_a] = { fqn: "@aws-cdk/aws-ecr-assets.DockerImageAsset", version: "1.119.0" };
function validateProps(props) {
    for (const [key, value] of Object.entries(props)) {
        if (core_1.Token.isUnresolved(value)) {
            throw new Error(`Cannot use Token as value of '${key}': this value is used before deployment starts`);
        }
    }
    validateBuildArgs(props.buildArgs);
}
function validateBuildArgs(buildArgs) {
    for (const [key, value] of Object.entries(buildArgs || {})) {
        if (core_1.Token.isUnresolved(key) || core_1.Token.isUnresolved(value)) {
            throw new Error('Cannot use tokens in keys or values of "buildArgs" since they are needed before deployment');
        }
    }
}
function toSymlinkFollow(follow) {
    switch (follow) {
        case undefined: return undefined;
        case assets_1.FollowMode.NEVER: return core_1.SymlinkFollowMode.NEVER;
        case assets_1.FollowMode.ALWAYS: return core_1.SymlinkFollowMode.ALWAYS;
        case assets_1.FollowMode.BLOCK_EXTERNAL: return core_1.SymlinkFollowMode.BLOCK_EXTERNAL;
        case assets_1.FollowMode.EXTERNAL: return core_1.SymlinkFollowMode.EXTERNAL;
    }
}
//# sourceMappingURL=data:application/json;base64,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