# Copyright (c) 2025 e-dynamics GmbH and affiliates
#
# This source code is licensed under the MIT license found in the
# LICENSE file in the root directory of this source tree.

"""
Tests the wheels during build_wheels.yml workflow
"""

# Standard Library
# Verify from where Gloria was imported
import importlib
import pathlib

m = importlib.import_module("gloria")
print("gloria imported from:", pathlib.Path(m.__file__).resolve())

# Third Party
# Imports
import pandas as pd

# Gloria
from gloria import Gloria, cast_series_to_kind

# Data types each model requires
MODEL_TO_KIND_MAP = {
    "normal": "f",
    "binomial": "u",
    "poisson": "u",
    "negative binomial": "u",
    "gamma": "f",
    "beta": "f",
    "beta-binomial": "u",
}

# Test data for each model
# fmt: off
data = pd.DataFrame({'ds': ['2025-01-01', '2025-01-02', '2025-01-03', '2025-01-04', '2025-01-05', '2025-01-06', '2025-01-07', '2025-01-08', '2025-01-09', '2025-01-10', '2025-01-11', '2025-01-12', '2025-01-13', '2025-01-14', '2025-01-15', '2025-01-16', '2025-01-17', '2025-01-18', '2025-01-19', '2025-01-20', '2025-01-21', '2025-01-22', '2025-01-23', '2025-01-24', '2025-01-25', '2025-01-26', '2025-01-27', '2025-01-28', '2025-01-29', '2025-01-30'], 'normal': [-0.4231347083721892, 0.3357378591715205, 1.0373742443653604, -0.4803326937881257, -0.9440912599076132, -0.4565897932271078, -0.2326671266974623, -0.4564784167417445, 0.2327757927801666, 0.9204340212731724, -0.3857556845794029, -0.9789896996945228, -0.3889348300729245, -0.2739206853873592, -0.6225518992132018, 0.2473772280737896, 1.048786540483795, -0.5695307352290613, -1.0295717277138987, -0.5461901839846204, -0.3193062380777263, -0.555798218733433, 0.0958823680614643, 0.9192274693707996, -0.5681226368185596, -1.044334320878817, -0.5867664273879396, -0.3390927249594525, -0.552670213926425, 0.3295152528935312], 'binomial': [26, 52, 70, 47, 45, 61, 48, 21, 44, 80, 39, 37, 62, 49, 29, 45, 66, 36, 53, 70, 50, 33, 47, 58, 37, 47, 59, 45, 32, 46], 'poisson': [45, 30, 52, 70, 32, 49, 53, 52, 26, 58, 70, 42, 43, 58, 45, 35, 69, 68, 42, 54, 61, 50, 45, 66, 59, 38, 48, 56, 39, 37], 'negative binomial': [33, 70, 46, 37, 69, 53, 45, 37, 70, 45, 44, 46, 45, 35, 23, 56, 47, 30, 53, 28, 42, 23, 75, 47, 39, 60, 32, 51, 27, 68], 'gamma': [1.4306978678846838, 0.962551916090637, 1.4020131156023643, 1.1465415574970623, 1.05294789239388, 2.080493130679061, 1.2084560564736126, 1.4507658722569017, 1.0461475756404353, 1.259891083044627, 1.0668352356303474, 1.1894336655357935, 2.035719672488394, 1.2085277585330112, 1.5704077999487545, 1.0214119527164995, 1.3788166772509838, 1.02339674422549, 1.0301572356884137, 2.031017246632578, 1.208681676160187, 1.54227023656915, 1.0752501969833983, 1.3891687053102402, 1.0948458311194431, 1.0938020837188305, 1.956435894851119, 1.3744067523752204, 1.4981673400796176, 1.003832512981804], 'beta': [0.7443650596385081, 0.5626978555653177, 0.4634115360913539, 0.5591168551916887, 0.2385499558879268, 0.3364122912725406, 0.3878653714668964, 0.6054992096292293, 0.443980747723462, 0.4433282296912506, 0.5486485315993714, 0.3351991091716627, 0.342369102484318, 0.4250548421067758, 0.6983319364893573, 0.4243180138258659, 0.428603190497045, 0.518425002247202, 0.2704395178415581, 0.3992728276560353, 0.4174142230465483, 0.5742832854738725, 0.4220243764292088, 0.37309466064454, 0.3823114301257859, 0.3240021127085461, 0.3666112736281537, 0.3456151304468548, 0.7015915610413099, 0.4688142460076385], 'beta-binomial': [33, 73, 51, 71, 42, 54, 64, 32, 61, 61, 53, 50, 64, 55, 31, 75, 45, 47, 41, 60, 63, 41, 71, 63, 56, 49, 51, 58, 28, 73]}) # ruff: noqa
# fmt: on

# Convert to timestamps
data["ds"] = pd.to_datetime(data["ds"])

# Attempt to fit each model and collect which failed
failed_models = []
for model in MODEL_TO_KIND_MAP.keys():
    m = Gloria(model=model, metric_name=model, n_changepoints=0)
    m.add_seasonality("weekly", "7d", fourier_order=3)
    data_in = data[["ds", model]].copy()
    data_in[model] = cast_series_to_kind(
        data_in[model], MODEL_TO_KIND_MAP[model]
    )
    try:
        m.fit(data_in)
    except Exception as e:
        failed_models.append(model)
        print(f"Error for {model} was: \n", e)

# If any model failed, raise an error
if failed_models:
    failed_models_str = ", ".join(failed_models)
    raise RuntimeError(f"models {failed_models_str} failed to run.")
