from typing import List, Dict, Any
from malloryai.sdk.api.v1.decorators.sorting import validate_sorting
from malloryai.sdk.api.v1.http_client import HttpClient


class ExploitationsClient:
    """Client for managing exploitations."""

    LIST_EXPLOITATIONS_PATH = "/exploitations"
    EXPLOITATION_DETAIL_PATH = "/exploitations/{identifier}"

    def __init__(self, http_client: HttpClient):
        self.http_client = http_client

    @validate_sorting(sort_pattern="^(count|created_at|updated_at)$")
    async def list_exploitations(
        self,
        offset: int = 0,
        limit: int = 100,
        sort: str = "created_at",
        order: str = "desc",
    ) -> List[Dict[str, Any]]:
        """
        List exploitations.
        :param offset: Offset for pagination.
        :param limit: Limit for pagination.
        :param sort: Sorting field.
        :param order: Sorting order.
        :return: List of exploitations.
        """
        params = {"offset": offset, "limit": limit, "sort": sort, "order": order}
        return await self.http_client.get(self.LIST_EXPLOITATIONS_PATH, params=params)

    async def get_exploitation(self, identifier: str) -> Dict[str, Any]:
        """
        Fetch details of a specific exploitation.
        :param identifier: Identifier of the exploitation.
        :return: Details of the exploitation.
        """
        path = self.EXPLOITATION_DETAIL_PATH.format(identifier=identifier)
        return await self.http_client.get(path)

    @validate_sorting(sort_pattern="^(count|created_at|updated_at)$")
    def list_exploitations_sync(
        self,
        offset: int = 0,
        limit: int = 100,
        sort: str = "created_at",
        order: str = "desc",
    ) -> List[Dict[str, Any]]:
        """
        Synchronous version of list_exploitations.
        :param offset: Offset for pagination.
        :param limit: Limit for pagination.
        :param sort: Sorting field.
        :param order: Sorting order.
        :return: List of exploitations.
        """
        params = {"offset": offset, "limit": limit, "sort": sort, "order": order}
        return self.http_client.get_sync(self.LIST_EXPLOITATIONS_PATH, params=params)

    def get_exploitation_sync(self, identifier: str) -> Dict[str, Any]:
        """
        Synchronous version of get_exploitation.
        :param identifier: Identifier of the exploitation.
        :return: Details of the exploitation.
        """
        path = self.EXPLOITATION_DETAIL_PATH.format(identifier=identifier)
        return self.http_client.get_sync(path)
