from typing import List, Dict, Any
from malloryai.sdk.api.v1.decorators.sorting import validate_sorting
from malloryai.sdk.api.v1.http_client import HttpClient


class ExploitsClient:
    """Client for managing exploits."""

    LIST_EXPLOITS_PATH = "/exploits"
    EXPLOIT_DETAIL_PATH = "/exploits/{identifier}"

    def __init__(self, http_client: HttpClient):
        self.http_client = http_client

    @validate_sorting(sort_pattern="^(url|authors|maturity|created_at|updated_at)$")
    async def list_exploits(
        self,
        offset: int = 0,
        limit: int = 100,
        sort: str = "created_at",
        order: str = "desc",
        filter: str = "",
    ) -> List[Dict[str, Any]]:
        """
        List all exploits.
        :param offset: Offset for pagination.
        :param limit: Limit for pagination.
        :param sort: Sorting field.
        :param order: Sorting order.
        :param filter: Filtering field.
        :return: List of exploits.
        """
        params = {
            "offset": offset,
            "limit": limit,
            "sort": sort,
            "order": order,
            "filter": filter,
        }
        return await self.http_client.get(self.LIST_EXPLOITS_PATH, params=params)

    async def get_exploit(self, identifier: str) -> Dict[str, Any]:
        """
        Get an exploit by identifier.
        :param identifier: Exploit identifier.
        :return: Exploit.
        """
        path = self.EXPLOIT_DETAIL_PATH.format(identifier=identifier)
        return await self.http_client.get(path)

    def list_exploits_sync(
        self,
        offset: int = 0,
        limit: int = 100,
        sort: str = "created_at",
        order: str = "desc",
        filter: str = "",
    ) -> List[Dict[str, Any]]:
        """
        Synchronous version of list_exploits.
        :param offset: Offset for pagination.
        :param limit: Limit for pagination.
        :param sort: Sorting field.
        :param order: Sorting order.
        :param filter: Filtering field.
        :return: List of exploits.
        """
        params = {
            "offset": offset,
            "limit": limit,
            "sort": sort,
            "order": order,
            "filter": filter,
        }
        return self.http_client.get_sync(self.LIST_EXPLOITS_PATH, params=params)

    def get_exploit_sync(self, identifier: str) -> Dict[str, Any]:
        """
        Synchronous version of get_exploit.
        :param identifier: Exploit identifier.
        :return: Exploit.
        """
        path = self.EXPLOIT_DETAIL_PATH.format(identifier=identifier)
        return self.http_client.get_sync(path)
