from ._base import Badge, GitBadge, XMLFileBadge
from .mixins import RangedBadgeMixin


class TextBadge(Badge):
    """ A simple badge to display some Text. """

    def __init__(self, left_text, right_text):
        self._right_text = right_text
        self._left_text = left_text

        super().__init__()

    @property
    def name(self):
        return f"{self.left_text}_{self.right_text}"

    @property
    def left_text(self):
        return self._left_text

    @property
    def right_text(self):
        return self._right_text


class GitVersionBadge(GitBadge):
    """ A badge that retrieves the version from the last tag """

    name = "version"

    @property
    def right_text(self):
        tags = self.repo.tags

        try:
            return tags[-1].name
        except IndexError:
            return "N.D."


class CoverageBadge(XMLFileBadge, RangedBadgeMixin):
    """ This Badge is a coverage badge created parsing a coverage file
    generated by `coverage.py` in XML format
    """

    name = "coverage"

    @property
    def right_text(self):
        root = self.content.getroot()
        line_rate = float(root.attrib["line-rate"])
        self.value = line_rate

        return f"{line_rate*100:.2f}%"


class LOCBadge(XMLFileBadge):
    """ This Badge shows the line of codes of the project generated from
    a CLOC XML format.

    You can use `pygount` to generate the file. Other LOC programs should
    comply with the same format
    """

    name = "loc"

    @property
    def right_text(self):
        root = self.content.getroot()
        total = root.find(".//total")

        return f'{total.attrib["code"]} lines'
