"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SsmDocument = exports.DocumentFormat = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const os_1 = require("os");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ssm_1 = require("aws-cdk-lib/aws-ssm");
const constructs_1 = require("constructs");
const string_variable_1 = require("../interface/variables/string-variable");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const yaml = require('js-yaml');
var DocumentFormat;
(function (DocumentFormat) {
    DocumentFormat[DocumentFormat["JSON"] = 0] = "JSON";
    DocumentFormat[DocumentFormat["YAML"] = 1] = "YAML";
})(DocumentFormat = exports.DocumentFormat || (exports.DocumentFormat = {}));
class SsmDocument extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.documentName = props.documentName ?? id;
        this.description = props.description ?? this.documentName;
        this.header = props.header;
        this.assumeRole = props.assumeRole;
        this.docOutputs = props.docOutputs ?? [];
        this.docInputs = props.docInputs ?? [];
        if (this.assumeRole && this.assumeRole instanceof string_variable_1.StringVariable &&
            !this.docInputs.map(i => i.name).includes(this.assumeRole.reference)) {
            throw new Error('Assume role specified but not provided in inputs: ' + this.assumeRole);
        }
        this.props = props;
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        this.cfnDocument = new aws_ssm_1.CfnDocument(this, 'Resource', {
            ...this.props,
            ...{
                content: aws_cdk_lib_1.Lazy.any({
                    produce: () => {
                        const doc = this.buildSsmDocument();
                        return JSON.parse(JSON.stringify(doc));
                    },
                }),
                name: this.props.documentName,
                documentFormat: isYaml ? 'YAML' : 'JSON',
                documentType: this.documentType(),
                tags: [{ key: 'CdkGenerated', value: 'true' }, ...(this.props.tags || [])],
            },
        });
    }
    /**
     * The SSM Document inputs optionally specify a number of parameters including allowedValues, minItems etc.
     * This function builds an object containing the relevant (declared) input parameters.
     * The return object will be used to build the yaml/json representation of the document.
     */
    formatInputs() {
        const ssmInputs = {};
        this.docInputs.forEach(inp => {
            ssmInputs[inp.name] = inp.toSsm();
        });
        return ssmInputs;
    }
    /**
       * Synthesize before calling this function!
       * You can use this to Synthesize: cdk.SynthUtils.synthesize(stack);
       *
       * Converts the objects define in the SSM Document (including all of the steps) to an SSM document string.
       * The format is dependency on the documentFormat property provided to the class.
       * The yaml can be used as is and will behave (or at least should behave) as was simulated in the runSimulation().
       * @returns a string representation of this document as an SSM formatted yaml/json.
       */
    print() {
        const isYaml = this.props.documentFormat == DocumentFormat.YAML;
        // format as JSON first so that the JSON printing goes into effect
        const doc = JSON.parse(JSON.stringify(this.buildSsmDocument()));
        if (isYaml) {
            // Prepend the document with the header if defined
            return this.headerWithComments() + yaml.dump(doc);
        }
        else {
            // There is no way to provide the header comment in a json document
            return JSON.stringify(doc);
        }
    }
    /**
       * Prepend header line with # if it does not start with #.
       * If no header is provided, this method will return an emptry string.
       * Headers are only available in yaml format.
       */
    headerWithComments() {
        if (this.header == undefined) {
            return '';
        }
        return this.header.split(os_1.EOL).map(line => line.trim().startsWith('#') ? line : '# ' + line).join(os_1.EOL) +
            // "---" demarks the end of the header and the start of the document.
            '\n---\n';
    }
}
exports.SsmDocument = SsmDocument;
_a = JSII_RTTI_SYMBOL_1;
SsmDocument[_a] = { fqn: "@cdklabs/cdk-ssm-documents.SsmDocument", version: "0.0.34" };
//# sourceMappingURL=data:application/json;base64,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