"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiExecuteAutomationHook = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const __1 = require("../..");
/**
 * ExecuteAutomation implementation using AWS API
 */
class ApiExecuteAutomationHook {
    constructor(awsInvoker, sleepHook) {
        this.props = { awsInvoker, sleepHook };
    }
    execute(props) {
        console.log(`ExecuteAutomation: Checking execution inputs for ${props.documentName}`);
        this.parameterValidation(props);
        console.log(`ExecuteAutomation: Starting execution of ${props.documentName}`);
        const executionId = this.startExecution(props);
        console.log(`ExecuteAutomation: Waiting for ${executionId} to complete`);
        const result = this.waitForExecution(executionId);
        console.log(`ExecuteAutomation: ${executionId} finished`);
        return result;
    }
    parameterValidation(props) {
        if (props.targetLocations && (props.targetLocations.length < 1 || props.targetLocations.length > 100)) {
            throw new Error('TargetLocations must contain atleast 1 and no more than 100 items.');
        }
        if (props.targetMaps && props.targets) {
            throw new Error('TargetMaps and Targets cannot both be specified.');
        }
        if ((props.targets && !props.targetParameterName) || (!props.targets && props.targetParameterName)) {
            throw new Error('TargetParameterName and Targets must both be specified together.');
        }
    }
    startExecution(props) {
        const result = new __1.AutomationStepSimulation(new __1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'execute', {
            service: __1.AwsService.SSM,
            pascalCaseApi: 'StartAutomationExecution',
            apiParams: this.getExecuteProps(props),
            outputs: [{
                    outputType: __1.DataTypeEnum.STRING,
                    name: 'executionId',
                    selector: '$.AutomationExecutionId',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== __1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to start execution: ${result.stackTrace}`);
        }
        return result.outputs?.['execute.executionId'];
    }
    getExecuteProps(props) {
        const apiParamMap = {
            DocumentName: props.documentName,
            DocumentVersion: props.documentVersion,
            TargetParameterName: props.targetParameterName,
            MaxConcurrency: props.maxConcurrency,
            MaxErrors: props.maxErrors,
            Targets: props.targets,
            Tags: props.tags,
            TargetLocations: props.targetLocations,
        };
        const apiParams = {};
        for (const key of Object.keys(apiParamMap)) {
            const value = apiParamMap[key];
            if (value === undefined) {
                continue;
            }
            apiParams[key] = value;
        }
        if (props.runtimeParameters) {
            apiParams.Parameters = this.adaptExecuteParameters(props.runtimeParameters);
        }
        if (props.targetMaps) {
            apiParams.TargetMaps = this.adaptTargetMaps(props.targetMaps);
        }
        return apiParams;
    }
    adaptExecuteParameters(parameters) {
        const result = {};
        for (const key of Object.keys(parameters)) {
            const value = parameters[key];
            if (__1.isString(value)) {
                result[key] = [value];
                continue;
            }
            if (__1.isBoolean(value) || __1.isNumber(value)) {
                result[key] = [`${value}`];
                continue;
            }
            if (__1.isStringList(value)) {
                result[key] = value;
                continue;
            }
            if (__1.isMapList(value)) {
                result[key] = value.map(x => JSON.stringify(x));
                continue;
            }
            if (__1.isStringMap(value)) {
                result[key] = [JSON.stringify(value)];
                continue;
            }
            throw new Error(`Parameter ${key} must be a Boolean, Number, String, StringList, StringMap, or MapList.`);
        }
        return result;
    }
    adaptTargetMaps(targetMaps) {
        return targetMaps.map(this.adaptExecuteParameters);
    }
    waitForExecution(executionId) {
        let execution = this.getExecution(executionId);
        while (ApiExecuteAutomationHook.RunningStatuses.includes(execution.AutomationExecutionStatus)) {
            console.log(`Automation ${executionId} is not complete. Status: ${execution.AutomationExecutionStatus}.`);
            new __1.AutomationStepSimulation(new __1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                sleepSeconds: 2,
            }), this.props).invoke({});
            execution = this.getExecution(executionId);
        }
        if (execution.AutomationExecutionStatus !== 'Success') {
            throw new Error(`${executionId} ended in status ${execution.AutomationExecutionStatus}`);
        }
        return {
            executionId: executionId,
            status: execution.AutomationExecutionStatus,
            output: Object.values(execution.Outputs).flat(),
        };
    }
    getExecution(executionId) {
        const result = new __1.AutomationStepSimulation(new __1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'getExecution', {
            service: __1.AwsService.SSM,
            pascalCaseApi: 'GetAutomationExecution',
            apiParams: {
                AutomationExecutionId: executionId,
            },
            outputs: [{
                    outputType: __1.DataTypeEnum.STRING_MAP,
                    name: 'execution',
                    selector: '$.AutomationExecution',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== __1.ResponseCode.SUCCESS) {
            throw new Error(`Failed to get execution ${executionId} status: ${result.stackTrace}`);
        }
        return result.outputs?.['getExecution.execution'];
    }
}
exports.ApiExecuteAutomationHook = ApiExecuteAutomationHook;
_a = JSII_RTTI_SYMBOL_1;
ApiExecuteAutomationHook[_a] = { fqn: "@cdklabs/cdk-ssm-documents.ApiExecuteAutomationHook", version: "0.0.34" };
ApiExecuteAutomationHook.RunningStatuses = ['Pending', 'InProgress', 'Cancelling', 'Waiting'];
//# sourceMappingURL=data:application/json;base64,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