"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.AutomationStep = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const __1 = require("..");
const data_type_1 = require("../domain/data-type");
const on_failure_1 = require("../interface/on-failure");
const step_1 = require("./step");
/**
 * Parent class for AutomationSteps. These steps are using in an AutomationDocument.
 * You can instantiate steps using the AutomationBuilder for convenience.
 * You can use these steps to simulate their execution (mimics the SSM run) AND to build their yaml/json declaration.
 * Control flow of the subsequent step is determined by the currently executing step.
 * The flow of the execution therefore follows a chain-of-responsibility pattern.
 * The inputs received into a step AND the outputs of previous steps are merged to form inputs of subsequent steps.
 */
class AutomationStep extends step_1.Step {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.maxAttempts = props.maxAttempts ?? AutomationStep.DEFAULT_MAX_ATTEMPTS;
        this.timeoutSeconds = props.timeoutSeconds ?? AutomationStep.DEFAULT_TIMEOUT;
        this.isEnd = props.isEnd ?? false;
        this.onFailure = props.onFailure ?? new on_failure_1.Abort();
        this.onCancel = props.onCancel ?? new on_failure_1.Abort();
        this.explicitNextStep = props.explicitNextStep;
    }
    addToDocument(doc) {
        doc.addStep(this);
    }
    prepareSsmEntry(inputs) {
        // Maps maintain insertion order
        const ssmDef = new Map();
        if (this.description) {
            ssmDef.set('description', this.description);
        }
        ssmDef.set('name', this.name);
        ssmDef.set('action', this.action);
        ssmDef.set('inputs', inputs);
        if (this.listUserOutputs().length > 0) {
            const formattedOutputs = this.listUserOutputs().map(this.formatOutput);
            ssmDef.set('outputs', formattedOutputs);
        }
        if (this.isEnd) {
            ssmDef.set('isEnd', true);
        }
        if (this.maxAttempts != AutomationStep.DEFAULT_MAX_ATTEMPTS) {
            ssmDef.set('maxAttempts', this.maxAttempts);
        }
        if (this.timeoutSeconds != AutomationStep.DEFAULT_TIMEOUT) {
            ssmDef.set('timeoutSeconds', this.timeoutSeconds);
        }
        if (!(this.onCancel instanceof on_failure_1.Abort)) {
            ssmDef.set('onCancel', this.onCancel.toSsmValue());
        }
        if (!(this.onFailure instanceof on_failure_1.Abort)) {
            ssmDef.set('onFailure', this.onFailure.toSsmValue());
        }
        if (this.explicitNextStep != undefined) {
            ssmDef.set('nextStep', this.explicitNextStep.stepName);
        }
        const ssmDict = {};
        ssmDef.forEach((value, key) => {
            ssmDict[key] = value;
        });
        return ssmDict;
    }
    /**
     * Formats the provided output into an object that can be used to generate the SSM yaml/json representation.
     */
    formatOutput(output) {
        return {
            Name: output.name,
            Selector: output.selector,
            Type: new data_type_1.DataType(output.outputType).toSsmString(),
        };
    }
    variables() {
        return Object.assign({}, ...this.listOutputs()
            .map(out => ({ [out.name]: new __1.StringVariable(`${this.name}.${out.name}`) })));
    }
    /**
     * Lists the outputs defined by the user for this step.
     */
    listUserOutputs() {
        return [];
    }
    listOutputs() {
        return this.listUserOutputs();
    }
}
exports.AutomationStep = AutomationStep;
_a = JSII_RTTI_SYMBOL_1;
AutomationStep[_a] = { fqn: "@cdklabs/cdk-ssm-documents.AutomationStep", version: "0.0.34" };
AutomationStep.DEFAULT_TIMEOUT = 3600;
AutomationStep.DEFAULT_MAX_ATTEMPTS = 1;
//# sourceMappingURL=data:application/json;base64,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