"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeleteImageSimulation = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const data_type_1 = require("../../domain/data-type");
const response_code_1 = require("../../domain/response-code");
const aws_api_step_1 = require("../../parent-steps/automation/aws-api-step");
const sleep_step_1 = require("../../parent-steps/automation/sleep-step");
const automation_step_simulation_1 = require("../automation-step-simulation");
const automation_simulation_base_1 = require("./automation-simulation-base");
const aws_service_1 = require("../../domain/aws-service");
/**
 * AutomationStep implementation for aws:deleteImage
 * https://docs.aws.amazon.com/systems-manager/latest/userguide/automation-action-delete.html
 */
class DeleteImageSimulation extends automation_simulation_base_1.AutomationSimulationBase {
    constructor(step, props) {
        super(step);
        this.action = 'aws:deleteImage';
        this.deleteImageStep = step;
        this.props = props;
    }
    executeStep(inputs) {
        const imageId = this.deleteImageStep.imageId.resolveToString(inputs);
        console.log(`DeleteImage: Get snapshots of ${imageId}`);
        const snapshots = this.getSnapshots(imageId);
        console.log(`DeleteImage: Deregister ${imageId}`);
        this.deregisterImage(imageId);
        console.log(`DeleteImage: Delete snapshots for ${imageId}`);
        this.deleteSnapshotsOfImage(snapshots);
        console.log(`DeleteImage: ${imageId} is deleted`);
        return {};
    }
    getSnapshots(imageId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'describeImage', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'DescribeImages',
            apiParams: {
                ImageIds: [imageId],
            },
            outputs: [{
                    outputType: data_type_1.DataTypeEnum.STRING_MAP,
                    selector: '$.Images[0]',
                    name: 'Image',
                }],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Describe image failed for ${imageId}: ${result.stackTrace}`);
        }
        const image = result.outputs?.['describeImage.Image'];
        return image?.BlockDeviceMappings.map(mapping => mapping.Ebs.SnapshotId) ?? [];
    }
    deregisterImage(imageId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deregisterImage', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'DeregisterImage',
            apiParams: {
                ImageId: imageId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Deregister image failed for ${imageId}: ${result.stackTrace}`);
        }
    }
    deleteSnapshotsOfImage(snapshotIds) {
        for (const snapshotId of snapshotIds) {
            if (snapshotId === null || snapshotId === undefined) {
                continue;
            }
            try {
                this.deleteSnapshot(snapshotId);
            }
            catch (e) {
                if (e.message.indexOf('InvalidSnapshot.InUse') < 0) {
                    throw e;
                }
                console.log('InvalidSnapshot.InUse case found. Attempt to retry after 2 seconds.');
                new automation_step_simulation_1.AutomationStepSimulation(new sleep_step_1.SleepStep(new aws_cdk_lib_1.Stack(), 'sleep', {
                    sleepSeconds: 2,
                }), this.props).invoke({});
                this.deleteSnapshot(snapshotId);
            }
        }
    }
    deleteSnapshot(snapshotId) {
        const result = new automation_step_simulation_1.AutomationStepSimulation(new aws_api_step_1.AwsApiStep(new aws_cdk_lib_1.Stack(), 'deleteSnapshot', {
            service: aws_service_1.AwsService.EC2,
            pascalCaseApi: 'DeleteSnapshot',
            apiParams: {
                SnapshotId: snapshotId,
            },
            outputs: [],
        }), this.props).invoke({});
        if (result.responseCode !== response_code_1.ResponseCode.SUCCESS) {
            throw new Error(`Delete snapshot failed for ${snapshotId}: ${result.stackTrace}`);
        }
    }
}
exports.DeleteImageSimulation = DeleteImageSimulation;
_a = JSII_RTTI_SYMBOL_1;
DeleteImageSimulation[_a] = { fqn: "@cdklabs/cdk-ssm-documents.DeleteImageSimulation", version: "0.0.34" };
//# sourceMappingURL=data:application/json;base64,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