from copy import copy
from django.contrib import admin
from edc_action_item import action_fieldset_tuple, action_fields
from edc_model_admin import audit_fieldset_tuple, SimpleHistoryAdmin
from edc_model_admin.dashboard import ModelAdminSubjectDashboardMixin

from ..admin_site import ambition_prn_admin
from ..forms import DeathReportTmgForm
from ..models import DeathReportTmg, DeathReport


@admin.register(DeathReportTmg, site=ambition_prn_admin)
class DeathReportTmgAdmin(ModelAdminSubjectDashboardMixin, SimpleHistoryAdmin):

    form = DeathReportTmgForm

    fieldsets = (
        (None, {"fields": ("subject_identifier", "death_report", "report_datetime")}),
        (
            "Opinion of TMG",
            {
                "fields": (
                    "cause_of_death",
                    "cause_of_death_other",
                    "tb_site",
                    "cause_of_death_agreed",
                    "narrative",
                    "report_status",
                    "report_closed_datetime",
                )
            },
        ),
        action_fieldset_tuple,
        audit_fieldset_tuple,
    )

    radio_fields = {
        "cause_of_death": admin.VERTICAL,
        "cause_of_death_agreed": admin.VERTICAL,
        "tb_site": admin.VERTICAL,
        "report_status": admin.VERTICAL,
    }

    list_display = [
        "subject_identifier",
        "dashboard",
        "report_datetime",
        "cause_of_death",
        "cause_of_death_agreed",
        "status",
        "report_closed_datetime",
    ]

    list_filter = (
        "report_datetime",
        "report_status",
        "cause_of_death_agreed",
        "cause_of_death",
    )

    search_fields = [
        "subject_identifier",
        "action_identifier",
        "tracking_identifier",
        "death_report__action_identifier",
        "death_report__tracking_identifier",
    ]

    def get_readonly_fields(self, request, obj=None):
        fields = super().get_readonly_fields(request, obj)
        action_flds = copy(list(action_fields))
        action_flds.remove("action_identifier")
        fields = list(action_flds) + list(fields)
        if obj:
            fields = fields + ["death_report"]
        return fields

    def status(self, obj=None):
        return obj.report_status.title()

    def formfield_for_foreignkey(self, db_field, request, **kwargs):
        if db_field.name == "death_report":
            if request.GET.get("death_report"):
                kwargs["queryset"] = DeathReport.objects.filter(
                    id__exact=request.GET.get("death_report", 0)
                )
            else:
                kwargs["queryset"] = DeathReport.objects.none()
        return super().formfield_for_foreignkey(db_field, request, **kwargs)
