import os
import pathlib
import tempfile

import pandas as pd
import polars as pl
from loguru import logger

from capcruncher.api.filter import CCSliceFilter, TiledCSliceFilter, TriCSliceFilter
from capcruncher.api.io import parse_bam
from capcruncher.api.statistics import SliceFilterStatsList

SLICE_FILTERS = {
    "capture": CCSliceFilter,
    "tri": TriCSliceFilter,
    "tiled": TiledCSliceFilter,
}


def merge_annotations(slices: os.PathLike, annotations: os.PathLike) -> pd.DataFrame:
    """
    Merges a parquet file containing slice information with a parquet file containing
    annotation information.

    Args:
        slices (os.PathLike): Path to parquet file containing slice information
        annotations (os.PathLike): Path to parquet file containing annotation information

    Returns:
        pd.DataFrame: Merged dataframe
    """

    logger.info("Opening annotations")

    with pl.StringCache():
        df_slices = pl.scan_parquet(slices)
        df_annotations = pl.scan_parquet(annotations).rename(
            {"Chromosome": "chrom", "Start": "start", "End": "end"}
        )

        df_slices = df_slices.join(
            df_annotations, on=["slice_name", "chrom", "start"], how="inner"
        )
        df_slices = df_slices.unique(subset=["slice_name"])

        return df_slices.collect().to_pandas()


def filter(
    bam: os.PathLike,
    annotations: os.PathLike,
    custom_filtering: os.PathLike = None,
    output_prefix: os.PathLike = "reporters",
    statistics: os.PathLike = "",
    method: str = "capture",
    sample_name: str = "",
    read_type: str = "",
    fragments: bool = True,
):
    """
    Removes unwanted aligned slices and identifies reporters.

    Parses a BAM file and merges this with a supplied annotation to identify unwanted slices.
    Filtering can be tuned for Capture-C, Tri-C and Tiled-C data to ensure optimal filtering.

    Common filters include:

    - Removal of unmapped slices
    - Removal of excluded/blacklisted slices
    - Removal of non-capture fragments
    - Removal of multi-capture fragments
    - Removal of non-reporter fragments
    - Removal of fragments with duplicated coordinates.

    For specific filtering for each of the three methods see:

    - :class:`CCSliceFilter <capcruncher.tools.filter.CCSliceFilter>`
    - :class:`TriCSliceFilter <capcruncher.tools.filter.TriCSliceFilter>`
    - :class:`TiledCSliceFilter <capcruncher.tools.filter.TiledCSliceFilter>`


    In addition to outputting valid reporter fragments and slices separated by capture probe,
    this script also provides statistics on the number of read/slices filtered at each stage,
    and the number of cis/trans reporters for each probe.

    Notes:

     Whilst the script is capable of processing any annotations in tsv format, provided
     that the correct columns are present. It is highly recomended that the "annotate"
     subcomand is used to generate this file.

     Slice filtering is currently hard coded into each filtering class. This will be
     modified in a future update to enable custom filtering orders.


    \f
    Args:
     bam (os.PathLike): Input bam file to analyse
     annotations (os.PathLike): Annotations file generated by slices-annotate
     custom_filtering (os.PathLike): Allows for custom filtering to be performed. A yaml file is used to supply this ordering.
     output_prefix (os.PathLike, optional): Output file prefix. Defaults to "reporters".
     stats_prefix (os.PathLike, optional): Output stats prefix. Defaults to "".
     method (str, optional): Analysis method. Choose from (capture|tri|tiled). Defaults to "capture".
     sample_name (str, optional): Sample being processed e.g. DOX-treated_1. Defaults to "".
     read_type (str, optional): Process combined(flashed) or non-combined reads (pe) used for statistics. Defaults to "".
     gzip (bool, optional): Compress output with gzip. Defaults to False.
     fragments (bool, optional): Enables fragments to be output. Defaults to True.
     read_stats (bool, optional): Enables read level statistics to be output. Defaults to True.
     slice_stats (bool, optional): Enables slice level statistics to be output. Defaults to True.
     cis_and_trans_stats (bool, optional): Enables cis/trans statistics to be output. Defaults to True.
    """

    with logger.catch():
        with tempfile.TemporaryDirectory() as tmpdir:
            tmp = pathlib.Path(tmpdir) / "tmp.parquet"

            logger.info("Loading bam file")
            # Its faster to write to parquet and then read it back than to join both dataframes with pandas
            parse_bam(bam).to_parquet(tmp)

            # Join bam file with annotations
            logger.info("Merging bam file with annotations")
            df_alignment = merge_annotations(tmp, annotations)

            # Make sure that the blacklist column is present
            if "blacklist" not in df_alignment.columns:
                df_alignment["blacklist"] = 0

        # Initialise SliceFilter
        # If no custom filtering, will use the class default.
        slice_filter_class = SLICE_FILTERS[method]
        slice_filter = slice_filter_class(
            slices=df_alignment,
            sample_name=sample_name,
            read_type=read_type,
            filter_stages=custom_filtering,
        )

        # Filter slices using the slice_filter
        logger.info(f"Filtering slices with method: {method}")
        slice_filter.filter_slices()

        # Extract statistics
        logger.info("Extracting statistics")
        stats_list = SliceFilterStatsList.from_list(slice_filter.filtering_stats)
        with open(statistics, "w") as f:
            f.write(stats_list.model_dump_json())

        # Write output
        df_slices = slice_filter.slices
        df_slices_with_viewpoint = slice_filter.slices_with_viewpoint
        df_capture = slice_filter.captures

        if fragments:
            logger.info("Writing reporters at the fragment level")
            df_fragments = (
                slice_filter_class(df_slices)
                .fragments.join(
                    df_capture["capture"], lsuffix="_slices", rsuffix="_capture"
                )
                .rename(
                    columns={
                        "capture_slices": "capture",
                        "capture_capture": "viewpoint",
                    }
                )
            )

            df_fragments.to_parquet(
                f"{output_prefix}.fragments.parquet",
                compression="snappy",
                engine="pyarrow",
            )

        logger.info("Writing reporters slices")

        # Enforce dtype for parent_id
        df_slices_with_viewpoint = df_slices_with_viewpoint.assign(
            parent_id=lambda df: df["parent_id"].astype("int64")
        ).drop_duplicates("slice_id")

        # Convert objects to category
        to_convert = df_slices_with_viewpoint.select_dtypes(include="object").columns
        df_slices_with_viewpoint[to_convert] = df_slices_with_viewpoint[
            to_convert
        ].astype("category")

        df_slices_with_viewpoint.to_parquet(
            f"{output_prefix}.slices.parquet",
            compression="snappy",
            engine="pyarrow",
        )

        logger.info("Completed analysis of BAM file")
