"""
Metric roll-up classes.
"""


from collections import defaultdict
import random
import time


class Metric(object):
    """
    A base metric class that accepts points, slices them into time intervals
    and performs roll-ups within those intervals.
    """

    def add_point(self, value):
        """ Add a point to the given metric. """
        raise NotImplementedError()

    def flush(self, timestamp):
        """ Flush all metrics up to the given timestamp. """
        raise NotImplementedError()


class Gauge(Metric):
    """ A gauge metric. """

    stats_tag = 'g'

    def __init__(self, name, tags):
        self.name = name
        self.tags = tags
        self.value = None

    def add_point(self, value):
        self.value = value

    def flush(self, timestamp):
        return [(timestamp, self.value, self.name, self.tags)]

class Counter(Metric):
    """ A counter metric. """

    stats_tag = 'c'

    def __init__(self, name, tags):
        self.name = name
        self.tags = tags
        self.count = 0

    def add_point(self, value):
        self.count += value

    def flush(self, timestamp):
        return [(timestamp, self.count, self.name, self.tags)]


class Histogram(Metric):
    """ A histogram metric. """

    stats_tag = 'h'

    def __init__(self, name, tags):
        self.name = name
        self.tags = tags
        self.max = float("-inf")
        self.min = float("inf")
        self.sum = 0
        self.count = 0
        self.sample_size = 1000
        self.samples = []
        self.percentiles = [0.75, 0.85, 0.95, 0.99]

    def add_point(self, value):
        self.max = self.max if self.max > value else value
        self.min = self.min if self.min < value else value
        self.sum += value
        if self.count < self.sample_size:
            self.samples.append(value)
        else:
            self.samples[random.randrange(0, self.sample_size)] = value
        self.count += 1

    def flush(self, timestamp):
        if not self.count:
            return []
        metrics = [
            (timestamp, self.min,       '%s.min'   % self.name, self.tags),
            (timestamp, self.max,       '%s.max'   % self.name, self.tags),
            (timestamp, self.count,     '%s.count' % self.name, self.tags),
            (timestamp, self.average(), '%s.avg'   % self.name, self.tags)
        ]
        length = len(self.samples)
        self.samples.sort()
        for p in self.percentiles:
            val = self.samples[int(round(p * length - 1))]
            name = '%s.%spercentile' % (self.name, int(p * 100))
            metrics.append((timestamp, val, name, self.tags))
        return metrics

    def average(self):
        return float(self.sum) / self.count


class MetricsAggregator(object):
    """
    A small class to handle the roll-ups of multiple metrics at once.
    """

    def __init__(self, roll_up_interval=10):
        self._metrics = defaultdict(lambda: {})
        self._roll_up_interval = roll_up_interval

    def add_point(self, metric, tags, timestamp, value, metric_class, sample_rate=1):
        # The sample rate is currently ignored for in process stuff
        interval = timestamp - timestamp % self._roll_up_interval
        key = (metric, tuple(sorted(tags)) if tags else tags)
        if key not in self._metrics[interval]:
            self._metrics[interval][key] = metric_class(metric, tags)
        self._metrics[interval][key].add_point(value)

    def flush(self, timestamp):
        """ Flush all metrics up to the given timestamp. """
        interval = timestamp - timestamp % self._roll_up_interval
        past_intervals = [i for i in self._metrics if i < interval]
        metrics = []
        for i in past_intervals:
            for m in list(self._metrics.pop(i).values()):
                metrics += m.flush(i)
        return metrics

