# Copyright 2019 Gradient Health Inc. All Rights Reserved.
# Author: Marcelo Lerendegui <marcelo@gradienthealth.io>
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# ============================================================================

"""Use decode_dicom ops in python."""

import numpy as np

import tensorflow as tf
from tensorflow_io.python.ops import core_ops


def decode_dicom_image(
    contents,
    color_dim=False,
    on_error="skip",
    scale="preserve",
    dtype=tf.uint16,
    name=None,
):
    """Getting DICOM Image Data.

    This package has two operations which wrap `DCMTK` functions.
    `decode_dicom_image` decodes the pixel data from DICOM files, and
    `decode_dicom_data` decodes tag information.
    `dicom_tags` contains useful DICOM tags such as `dicom_tags.PatientsName`.
    We borrow the same tag notation from the
    [`pydicom`](https://pydicom.github.io/) dicom package.

    The detailed usage of DICOM is available in
    [tutorial](https://www.tensorflow.org/io/tutorials/dicom).

    If this package helped, please kindly cite the below:
    ```
    @misc{marcelo_lerendegui_2019_3337331,
        author       = {Marcelo Lerendegui and Ouwen Huang},
        title        = {Tensorflow Dicom Decoder},
        month        = jul,
        year         = 2019,
        doi          = {10.5281/zenodo.3337331},
        url          = {https://doi.org/10.5281/zenodo.3337331}
    }
    ```

    Args:
        contents: A Tensor of type string. 0-D. The byte string encoded DICOM file.
        color_dim: An optional `bool`. Defaults to `False`. If `True`, a third
        channel will be appended to all images forming a 3-D tensor.
        A 1024 x 1024 grayscale image will be 1024 x 1024 x 1.
        on_error: Defaults to `skip`. This attribute establishes the behavior in
        case an error occurs on opening the image or if the output type cannot
        accomodate all the possible input values. For example if the user sets
        the output dtype to `tf.uint8`, but a dicom image stores a `tf.uint16`
        type. `strict` throws an error. `skip` returns a 1-D empty tensor.
        `lossy` continues with the operation scaling the value via the `scale`
        attribute.
        scale:  Defaults to `preserve`. This attribute establishes what to do with
        the scale of the input values. `auto` will autoscale the input values,
        if the output type is integer, `auto` will use the maximum output scale
        for example a `uint8` which stores values from [0, 255] can be linearly
        stretched to fill a `uint16` that is [0,65535]. If the output is float,
        `auto` will scale to [0,1]. `preserve` keeps the values as they are, an
        input value greater than the maximum possible output will be clipped.
        dtype: An optional `tf.DType` from: `tf.uint8`, `tf.uint16`, `tf.uint32`,
        `tf.uint64`, `tf.float16`, `tf.float32`, `tf.float64`. Defaults to
        `tf.uint16`.
        name: A name for the operation (optional).

    Returns:
        A `Tensor` of type `dtype` and the shape is determined by the DICOM file.
    """
    return core_ops.io_decode_dicom_image(
        contents=contents,
        color_dim=color_dim,
        on_error=on_error,
        scale=scale,
        dtype=dtype,
        name=name,
    )


def decode_dicom_data(contents, tags=None, name=None):
    """Getting DICOM Tag Data.

    This package has two operations which wrap `DCMTK` functions.
    `decode_dicom_image` decodes the pixel data from DICOM files, and
    `decode_dicom_data` decodes tag information.
    `dicom_tags` contains useful DICOM tags such as `dicom_tags.PatientsName`.
    We borrow the same tag notation from the
    [`pydicom`](https://pydicom.github.io/) dicom package.

    The detailed usage of DICOM is available in
    [tutorial](https://www.tensorflow.org/io/tutorials/dicom).

    If this package helped, please kindly cite the below:
    ```
    @misc{marcelo_lerendegui_2019_3337331,
        author       = {Marcelo Lerendegui and Ouwen Huang},
        title        = {Tensorflow Dicom Decoder},
        month        = jul,
        year         = 2019,
        doi          = {10.5281/zenodo.3337331},
        url          = {https://doi.org/10.5281/zenodo.3337331}
    }
    ```

    Args:
        contents: A Tensor of type string. 0-D. The byte string encoded DICOM file.
        tags: A Tensor of type `tf.uint32` of any dimension.
        These `uint32` numbers map directly to DICOM tags.
        name: A name for the operation (optional).

    Returns:
        A `Tensor` of type `tf.string` and same shape as `tags`. If a dicom tag is
        a list of strings, they are combined into one string and seperated by a
        double backslash `\\`. There is a bug in
        [DCMTK](https://support.dcmtk.org/docs/) if the tag is a list of numbers,
        only the zeroth element will be returned as a string.
    """
    return core_ops.io_decode_dicom_data(contents=contents, tags=tags, name=name)


class dicom_tags:  # pylint: disable=invalid-name
    """dicom_tags"""

    def __init__(self):
        pass

    # pylint: disable=invalid-name
    FileMetaInformationGroupLength = np.uint32(int("00020000", 16))
    FileMetaInformationVersion = np.uint32(int("00020001", 16))
    MediaStorageSOPClassUID = np.uint32(int("00020002", 16))
    MediaStorageSOPInstanceUID = np.uint32(int("00020003", 16))
    TransferSyntaxUID = np.uint32(int("00020010", 16))
    ImplementationClassUID = np.uint32(int("00020012", 16))
    ImplementationVersionName = np.uint32(int("00020013", 16))
    SourceApplicationEntityTitle = np.uint32(int("00020016", 16))
    SendingApplicationEntityTitle = np.uint32(int("00020017", 16))
    ReceivingApplicationEntityTitle = np.uint32(int("00020018", 16))
    PrivateInformationCreatorUID = np.uint32(int("00020100", 16))
    PrivateInformation = np.uint32(int("00020102", 16))
    FilesetID = np.uint32(int("00041130", 16))
    FilesetDescriptorFileID = np.uint32(int("00041141", 16))
    SpecificCharacterSetofFilesetDescriptorFile = np.uint32(int("00041142", 16))
    OffsetoftheFirstDirectoryRecordoftheRootDirectoryEntity = np.uint32(
        int("00041200", 16)
    )
    OffsetoftheLastDirectoryRecordoftheRootDirectoryEntity = np.uint32(
        int("00041202", 16)
    )
    FilesetConsistencyFlag = np.uint32(int("00041212", 16))
    DirectoryRecordSequence = np.uint32(int("00041220", 16))
    OffsetoftheNextDirectoryRecord = np.uint32(int("00041400", 16))
    RecordInuseFlag = np.uint32(int("00041410", 16))
    OffsetofReferencedLowerLevelDirectoryEntity = np.uint32(int("00041420", 16))
    DirectoryRecordType = np.uint32(int("00041430", 16))
    PrivateRecordUID = np.uint32(int("00041432", 16))
    ReferencedFileID = np.uint32(int("00041500", 16))
    MRDRDirectoryRecordOffset = np.uint32(int("00041504", 16))
    ReferencedSOPClassUIDinFile = np.uint32(int("00041510", 16))
    ReferencedSOPInstanceUIDinFile = np.uint32(int("00041511", 16))
    ReferencedTransferSyntaxUIDinFile = np.uint32(int("00041512", 16))
    ReferencedRelatedGeneralSOPClassUIDinFile = np.uint32(int("0004151A", 16))
    NumberofReferences = np.uint32(int("00041600", 16))
    LengthtoEnd = np.uint32(int("00080001", 16))
    SpecificCharacterSet = np.uint32(int("00080005", 16))
    LanguageCodeSequence = np.uint32(int("00080006", 16))
    ImageType = np.uint32(int("00080008", 16))
    RecognitionCode = np.uint32(int("00080010", 16))
    InstanceCreationDate = np.uint32(int("00080012", 16))
    InstanceCreationTime = np.uint32(int("00080013", 16))
    InstanceCreatorUID = np.uint32(int("00080014", 16))
    InstanceCoercionDateTime = np.uint32(int("00080015", 16))
    SOPClassUID = np.uint32(int("00080016", 16))
    SOPInstanceUID = np.uint32(int("00080018", 16))
    RelatedGeneralSOPClassUID = np.uint32(int("0008001A", 16))
    OriginalSpecializedSOPClassUID = np.uint32(int("0008001B", 16))
    StudyDate = np.uint32(int("00080020", 16))
    SeriesDate = np.uint32(int("00080021", 16))
    AcquisitionDate = np.uint32(int("00080022", 16))
    ContentDate = np.uint32(int("00080023", 16))
    OverlayDate = np.uint32(int("00080024", 16))
    CurveDate = np.uint32(int("00080025", 16))
    AcquisitionDateTime = np.uint32(int("0008002A", 16))
    StudyTime = np.uint32(int("00080030", 16))
    SeriesTime = np.uint32(int("00080031", 16))
    AcquisitionTime = np.uint32(int("00080032", 16))
    ContentTime = np.uint32(int("00080033", 16))
    OverlayTime = np.uint32(int("00080034", 16))
    CurveTime = np.uint32(int("00080035", 16))
    DataSetType = np.uint32(int("00080040", 16))
    DataSetSubtype = np.uint32(int("00080041", 16))
    NuclearMedicineSeriesType = np.uint32(int("00080042", 16))
    AccessionNumber = np.uint32(int("00080050", 16))
    IssuerofAccessionNumberSequence = np.uint32(int("00080051", 16))
    QueryRetrieveLevel = np.uint32(int("00080052", 16))
    QueryRetrieveView = np.uint32(int("00080053", 16))
    RetrieveAETitle = np.uint32(int("00080054", 16))
    StationAETitle = np.uint32(int("00080055", 16))
    InstanceAvailability = np.uint32(int("00080056", 16))
    FailedSOPInstanceUIDList = np.uint32(int("00080058", 16))
    Modality = np.uint32(int("00080060", 16))
    ModalitiesinStudy = np.uint32(int("00080061", 16))
    SOPClassesinStudy = np.uint32(int("00080062", 16))
    ConversionType = np.uint32(int("00080064", 16))
    PresentationIntentType = np.uint32(int("00080068", 16))
    Manufacturer = np.uint32(int("00080070", 16))
    InstitutionName = np.uint32(int("00080080", 16))
    InstitutionAddress = np.uint32(int("00080081", 16))
    InstitutionCodeSequence = np.uint32(int("00080082", 16))
    ReferringPhysiciansName = np.uint32(int("00080090", 16))
    ReferringPhysiciansAddress = np.uint32(int("00080092", 16))
    ReferringPhysiciansTelephoneNumbers = np.uint32(int("00080094", 16))
    ReferringPhysicianIdentificationSequence = np.uint32(int("00080096", 16))
    ConsultingPhysiciansName = np.uint32(int("0008009C", 16))
    ConsultingPhysicianIdentificationSequence = np.uint32(int("0008009D", 16))
    CodeValue = np.uint32(int("00080100", 16))
    ExtendedCodeValue = np.uint32(int("00080101", 16))
    CodingSchemeDesignator = np.uint32(int("00080102", 16))
    CodingSchemeVersion = np.uint32(int("00080103", 16))
    CodeMeaning = np.uint32(int("00080104", 16))
    MappingResource = np.uint32(int("00080105", 16))
    ContextGroupVersion = np.uint32(int("00080106", 16))
    ContextGroupLocalVersion = np.uint32(int("00080107", 16))
    ExtendedCodeMeaning = np.uint32(int("00080108", 16))
    ContextGroupExtensionFlag = np.uint32(int("0008010B", 16))
    CodingSchemeUID = np.uint32(int("0008010C", 16))
    ContextGroupExtensionCreatorUID = np.uint32(int("0008010D", 16))
    ContextIdentifier = np.uint32(int("0008010F", 16))
    CodingSchemeIdentificationSequence = np.uint32(int("00080110", 16))
    CodingSchemeRegistry = np.uint32(int("00080112", 16))
    CodingSchemeExternalID = np.uint32(int("00080114", 16))
    CodingSchemeName = np.uint32(int("00080115", 16))
    CodingSchemeResponsibleOrganization = np.uint32(int("00080116", 16))
    ContextUID = np.uint32(int("00080117", 16))
    MappingResourceUID = np.uint32(int("00080118", 16))
    LongCodeValue = np.uint32(int("00080119", 16))
    URNCodeValue = np.uint32(int("00080120", 16))
    EquivalentCodeSequence = np.uint32(int("00080121", 16))
    MappingResourceName = np.uint32(int("00080122", 16))
    ContextGroupIdentificationSequence = np.uint32(int("00080123", 16))
    MappingResourceIdentificationSequence = np.uint32(int("00080124", 16))
    TimezoneOffsetFromUTC = np.uint32(int("00080201", 16))
    PrivateDataElementCharacteristicsSequence = np.uint32(int("00080300", 16))
    PrivateGroupReference = np.uint32(int("00080301", 16))
    PrivateCreatorReference = np.uint32(int("00080302", 16))
    BlockIdentifyingInformationStatus = np.uint32(int("00080303", 16))
    NonidentifyingPrivateElements = np.uint32(int("00080304", 16))
    IdentifyingPrivateElements = np.uint32(int("00080306", 16))
    DeidentificationActionSequence = np.uint32(int("00080305", 16))
    DeidentificationAction = np.uint32(int("00080307", 16))
    NetworkID = np.uint32(int("00081000", 16))
    StationName = np.uint32(int("00081010", 16))
    StudyDescription = np.uint32(int("00081030", 16))
    ProcedureCodeSequence = np.uint32(int("00081032", 16))
    SeriesDescription = np.uint32(int("0008103E", 16))
    SeriesDescriptionCodeSequence = np.uint32(int("0008103F", 16))
    InstitutionalDepartmentName = np.uint32(int("00081040", 16))
    PhysiciansofRecord = np.uint32(int("00081048", 16))
    PhysiciansofRecordIdentificationSequence = np.uint32(int("00081049", 16))
    PerformingPhysiciansName = np.uint32(int("00081050", 16))
    PerformingPhysicianIdentificationSequence = np.uint32(int("00081052", 16))
    NameofPhysiciansReadingStudy = np.uint32(int("00081060", 16))
    PhysiciansReadingStudyIdentificationSequence = np.uint32(int("00081062", 16))
    OperatorsName = np.uint32(int("00081070", 16))
    OperatorIdentificationSequence = np.uint32(int("00081072", 16))
    AdmittingDiagnosesDescription = np.uint32(int("00081080", 16))
    AdmittingDiagnosesCodeSequence = np.uint32(int("00081084", 16))
    ManufacturersModelName = np.uint32(int("00081090", 16))
    ReferencedResultsSequence = np.uint32(int("00081100", 16))
    ReferencedStudySequence = np.uint32(int("00081110", 16))
    ReferencedPerformedProcedureStepSequence = np.uint32(int("00081111", 16))
    ReferencedSeriesSequence = np.uint32(int("00081115", 16))
    ReferencedPatientSequence = np.uint32(int("00081120", 16))
    ReferencedVisitSequence = np.uint32(int("00081125", 16))
    ReferencedOverlaySequence = np.uint32(int("00081130", 16))
    ReferencedStereometricInstanceSequence = np.uint32(int("00081134", 16))
    ReferencedWaveformSequence = np.uint32(int("0008113A", 16))
    ReferencedImageSequence = np.uint32(int("00081140", 16))
    ReferencedCurveSequence = np.uint32(int("00081145", 16))
    ReferencedInstanceSequence = np.uint32(int("0008114A", 16))
    ReferencedRealWorldValueMappingInstanceSequence = np.uint32(int("0008114B", 16))
    ReferencedSOPClassUID = np.uint32(int("00081150", 16))
    ReferencedSOPInstanceUID = np.uint32(int("00081155", 16))
    SOPClassesSupported = np.uint32(int("0008115A", 16))
    ReferencedFrameNumber = np.uint32(int("00081160", 16))
    SimpleFrameList = np.uint32(int("00081161", 16))
    CalculatedFrameList = np.uint32(int("00081162", 16))
    TimeRange = np.uint32(int("00081163", 16))
    FrameExtractionSequence = np.uint32(int("00081164", 16))
    MultiframeSourceSOPInstanceUID = np.uint32(int("00081167", 16))
    RetrieveURL = np.uint32(int("00081190", 16))
    TransactionUID = np.uint32(int("00081195", 16))
    WarningReason = np.uint32(int("00081196", 16))
    FailureReason = np.uint32(int("00081197", 16))
    FailedSOPSequence = np.uint32(int("00081198", 16))
    ReferencedSOPSequence = np.uint32(int("00081199", 16))
    OtherFailuresSequence = np.uint32(int("0008119A", 16))
    StudiesContainingOtherReferencedInstancesSequence = np.uint32(int("00081200", 16))
    RelatedSeriesSequence = np.uint32(int("00081250", 16))
    LossyImageCompressionRetired = np.uint32(int("00082110", 16))
    DerivationDescription = np.uint32(int("00082111", 16))
    SourceImageSequence = np.uint32(int("00082112", 16))
    StageName = np.uint32(int("00082120", 16))
    StageNumber = np.uint32(int("00082122", 16))
    NumberofStages = np.uint32(int("00082124", 16))
    ViewName = np.uint32(int("00082127", 16))
    ViewNumber = np.uint32(int("00082128", 16))
    NumberofEventTimers = np.uint32(int("00082129", 16))
    NumberofViewsinStage = np.uint32(int("0008212A", 16))
    EventElapsedTimes = np.uint32(int("00082130", 16))
    EventTimerNames = np.uint32(int("00082132", 16))
    EventTimerSequence = np.uint32(int("00082133", 16))
    EventTimeOffset = np.uint32(int("00082134", 16))
    EventCodeSequence = np.uint32(int("00082135", 16))
    StartTrim = np.uint32(int("00082142", 16))
    StopTrim = np.uint32(int("00082143", 16))
    RecommendedDisplayFrameRate = np.uint32(int("00082144", 16))
    TransducerPosition = np.uint32(int("00082200", 16))
    TransducerOrientation = np.uint32(int("00082204", 16))
    AnatomicStructure = np.uint32(int("00082208", 16))
    AnatomicRegionSequence = np.uint32(int("00082218", 16))
    AnatomicRegionModifierSequence = np.uint32(int("00082220", 16))
    PrimaryAnatomicStructureSequence = np.uint32(int("00082228", 16))
    AnatomicStructureSpaceorRegionSequence = np.uint32(int("00082229", 16))
    PrimaryAnatomicStructureModifierSequence = np.uint32(int("00082230", 16))
    TransducerPositionSequence = np.uint32(int("00082240", 16))
    TransducerPositionModifierSequence = np.uint32(int("00082242", 16))
    TransducerOrientationSequence = np.uint32(int("00082244", 16))
    TransducerOrientationModifierSequence = np.uint32(int("00082246", 16))
    AnatomicStructureSpaceOrRegionCodeSequenceTrial = np.uint32(int("00082251", 16))
    AnatomicPortalOfEntranceCodeSequenceTrial = np.uint32(int("00082253", 16))
    AnatomicApproachDirectionCodeSequenceTrial = np.uint32(int("00082255", 16))
    AnatomicPerspectiveDescriptionTrial = np.uint32(int("00082256", 16))
    AnatomicPerspectiveCodeSequenceTrial = np.uint32(int("00082257", 16))
    AnatomicLocationOfExaminingInstrumentDescriptionTrial = np.uint32(
        int("00082258", 16)
    )
    AnatomicLocationOfExaminingInstrumentCodeSequenceTrial = np.uint32(
        int("00082259", 16)
    )
    AnatomicStructureSpaceOrRegionModifierCodeSequenceTrial = np.uint32(
        int("0008225A", 16)
    )
    OnAxisBackgroundAnatomicStructureCodeSequenceTrial = np.uint32(int("0008225C", 16))
    AlternateRepresentationSequence = np.uint32(int("00083001", 16))
    IrradiationEventUID = np.uint32(int("00083010", 16))
    SourceIrradiationEventSequence = np.uint32(int("00083011", 16))
    RadiopharmaceuticalAdministrationEventUID = np.uint32(int("00083012", 16))
    IdentifyingComments = np.uint32(int("00084000", 16))
    FrameType = np.uint32(int("00089007", 16))
    ReferencedImageEvidenceSequence = np.uint32(int("00089092", 16))
    ReferencedRawDataSequence = np.uint32(int("00089121", 16))
    CreatorVersionUID = np.uint32(int("00089123", 16))
    DerivationImageSequence = np.uint32(int("00089124", 16))
    SourceImageEvidenceSequence = np.uint32(int("00089154", 16))
    PixelPresentation = np.uint32(int("00089205", 16))
    VolumetricProperties = np.uint32(int("00089206", 16))
    VolumeBasedCalculationTechnique = np.uint32(int("00089207", 16))
    ComplexImageComponent = np.uint32(int("00089208", 16))
    AcquisitionContrast = np.uint32(int("00089209", 16))
    DerivationCodeSequence = np.uint32(int("00089215", 16))
    ReferencedPresentationStateSequence = np.uint32(int("00089237", 16))
    ReferencedOtherPlaneSequence = np.uint32(int("00089410", 16))
    FrameDisplaySequence = np.uint32(int("00089458", 16))
    RecommendedDisplayFrameRateinFloat = np.uint32(int("00089459", 16))
    SkipFrameRangeFlag = np.uint32(int("00089460", 16))
    PatientsName = np.uint32(int("00100010", 16))
    PatientID = np.uint32(int("00100020", 16))
    IssuerofPatientID = np.uint32(int("00100021", 16))
    TypeofPatientID = np.uint32(int("00100022", 16))
    IssuerofPatientIDQualifiersSequence = np.uint32(int("00100024", 16))
    SourcePatientGroupIdentificationSequence = np.uint32(int("00100026", 16))
    GroupofPatientsIdentificationSequence = np.uint32(int("00100027", 16))
    SubjectRelativePositioninImage = np.uint32(int("00100028", 16))
    PatientsBirthDate = np.uint32(int("00100030", 16))
    PatientsBirthTime = np.uint32(int("00100032", 16))
    PatientsBirthDateinAlternativeCalendar = np.uint32(int("00100033", 16))
    PatientsDeathDateinAlternativeCalendar = np.uint32(int("00100034", 16))
    PatientsAlternativeCalendar = np.uint32(int("00100035", 16))
    PatientsSex = np.uint32(int("00100040", 16))
    PatientsInsurancePlanCodeSequence = np.uint32(int("00100050", 16))
    PatientsPrimaryLanguageCodeSequence = np.uint32(int("00100101", 16))
    PatientsPrimaryLanguageModifierCodeSequence = np.uint32(int("00100102", 16))
    QualityControlSubject = np.uint32(int("00100200", 16))
    QualityControlSubjectTypeCodeSequence = np.uint32(int("00100201", 16))
    StrainDescription = np.uint32(int("00100212", 16))
    StrainNomenclature = np.uint32(int("00100213", 16))
    StrainStockNumber = np.uint32(int("00100214", 16))
    StrainSourceRegistryCodeSequence = np.uint32(int("00100215", 16))
    StrainStockSequence = np.uint32(int("00100216", 16))
    StrainSource = np.uint32(int("00100217", 16))
    StrainAdditionalInformation = np.uint32(int("00100218", 16))
    StrainCodeSequence = np.uint32(int("00100219", 16))
    OtherPatientIDs = np.uint32(int("00101000", 16))
    OtherPatientNames = np.uint32(int("00101001", 16))
    OtherPatientIDsSequence = np.uint32(int("00101002", 16))
    PatientsBirthName = np.uint32(int("00101005", 16))
    PatientsAge = np.uint32(int("00101010", 16))
    PatientsSize = np.uint32(int("00101020", 16))
    PatientsSizeCodeSequence = np.uint32(int("00101021", 16))
    PatientsWeight = np.uint32(int("00101030", 16))
    PatientsAddress = np.uint32(int("00101040", 16))
    InsurancePlanIdentification = np.uint32(int("00101050", 16))
    PatientsMothersBirthName = np.uint32(int("00101060", 16))
    MilitaryRank = np.uint32(int("00101080", 16))
    BranchofService = np.uint32(int("00101081", 16))
    MedicalRecordLocator = np.uint32(int("00101090", 16))
    ReferencedPatientPhotoSequence = np.uint32(int("00101100", 16))
    MedicalAlerts = np.uint32(int("00102000", 16))
    Allergies = np.uint32(int("00102110", 16))
    CountryofResidence = np.uint32(int("00102150", 16))
    RegionofResidence = np.uint32(int("00102152", 16))
    PatientsTelephoneNumbers = np.uint32(int("00102154", 16))
    PatientsTelecomInformation = np.uint32(int("00102155", 16))
    EthnicGroup = np.uint32(int("00102160", 16))
    Occupation = np.uint32(int("00102180", 16))
    SmokingStatus = np.uint32(int("001021A0", 16))
    AdditionalPatientHistory = np.uint32(int("001021B0", 16))
    PregnancyStatus = np.uint32(int("001021C0", 16))
    LastMenstrualDate = np.uint32(int("001021D0", 16))
    PatientsReligiousPreference = np.uint32(int("001021F0", 16))
    PatientSpeciesDescription = np.uint32(int("00102201", 16))
    PatientSpeciesCodeSequence = np.uint32(int("00102202", 16))
    PatientsSexNeutered = np.uint32(int("00102203", 16))
    AnatomicalOrientationType = np.uint32(int("00102210", 16))
    PatientBreedDescription = np.uint32(int("00102292", 16))
    PatientBreedCodeSequence = np.uint32(int("00102293", 16))
    BreedRegistrationSequence = np.uint32(int("00102294", 16))
    BreedRegistrationNumber = np.uint32(int("00102295", 16))
    BreedRegistryCodeSequence = np.uint32(int("00102296", 16))
    ResponsiblePerson = np.uint32(int("00102297", 16))
    ResponsiblePersonRole = np.uint32(int("00102298", 16))
    ResponsibleOrganization = np.uint32(int("00102299", 16))
    PatientComments = np.uint32(int("00104000", 16))
    ExaminedBodyThickness = np.uint32(int("00109431", 16))
    ClinicalTrialSponsorName = np.uint32(int("00120010", 16))
    ClinicalTrialProtocolID = np.uint32(int("00120020", 16))
    ClinicalTrialProtocolName = np.uint32(int("00120021", 16))
    ClinicalTrialSiteID = np.uint32(int("00120030", 16))
    ClinicalTrialSiteName = np.uint32(int("00120031", 16))
    ClinicalTrialSubjectID = np.uint32(int("00120040", 16))
    ClinicalTrialSubjectReadingID = np.uint32(int("00120042", 16))
    ClinicalTrialTimePointID = np.uint32(int("00120050", 16))
    ClinicalTrialTimePointDescription = np.uint32(int("00120051", 16))
    ClinicalTrialCoordinatingCenterName = np.uint32(int("00120060", 16))
    PatientIdentityRemoved = np.uint32(int("00120062", 16))
    DeidentificationMethod = np.uint32(int("00120063", 16))
    DeidentificationMethodCodeSequence = np.uint32(int("00120064", 16))
    ClinicalTrialSeriesID = np.uint32(int("00120071", 16))
    ClinicalTrialSeriesDescription = np.uint32(int("00120072", 16))
    ClinicalTrialProtocolEthicsCommitteeName = np.uint32(int("00120081", 16))
    ClinicalTrialProtocolEthicsCommitteeApprovalNumber = np.uint32(int("00120082", 16))
    ConsentforClinicalTrialUseSequence = np.uint32(int("00120083", 16))
    DistributionType = np.uint32(int("00120084", 16))
    ConsentforDistributionFlag = np.uint32(int("00120085", 16))
    CADFileFormat = np.uint32(int("00140023", 16))
    ComponentReferenceSystem = np.uint32(int("00140024", 16))
    ComponentManufacturingProcedure = np.uint32(int("00140025", 16))
    ComponentManufacturer = np.uint32(int("00140028", 16))
    MaterialThickness = np.uint32(int("00140030", 16))
    MaterialPipeDiameter = np.uint32(int("00140032", 16))
    MaterialIsolationDiameter = np.uint32(int("00140034", 16))
    MaterialGrade = np.uint32(int("00140042", 16))
    MaterialPropertiesDescription = np.uint32(int("00140044", 16))
    MaterialPropertiesFileFormatRetired = np.uint32(int("00140045", 16))
    MaterialNotes = np.uint32(int("00140046", 16))
    ComponentShape = np.uint32(int("00140050", 16))
    CurvatureType = np.uint32(int("00140052", 16))
    OuterDiameter = np.uint32(int("00140054", 16))
    InnerDiameter = np.uint32(int("00140056", 16))
    ComponentWelderIDs = np.uint32(int("00140100", 16))
    SecondaryApprovalStatus = np.uint32(int("00140101", 16))
    SecondaryReviewDate = np.uint32(int("00140102", 16))
    SecondaryReviewTime = np.uint32(int("00140103", 16))
    SecondaryReviewerName = np.uint32(int("00140104", 16))
    RepairID = np.uint32(int("00140105", 16))
    MultipleComponentApprovalSequence = np.uint32(int("00140106", 16))
    OtherApprovalStatus = np.uint32(int("00140107", 16))
    OtherSecondaryApprovalStatus = np.uint32(int("00140108", 16))
    ActualEnvironmentalConditions = np.uint32(int("00141010", 16))
    ExpiryDate = np.uint32(int("00141020", 16))
    EnvironmentalConditions = np.uint32(int("00141040", 16))
    EvaluatorSequence = np.uint32(int("00142002", 16))
    EvaluatorNumber = np.uint32(int("00142004", 16))
    EvaluatorName = np.uint32(int("00142006", 16))
    EvaluationAttempt = np.uint32(int("00142008", 16))
    IndicationSequence = np.uint32(int("00142012", 16))
    IndicationNumber = np.uint32(int("00142014", 16))
    IndicationLabel = np.uint32(int("00142016", 16))
    IndicationDescription = np.uint32(int("00142018", 16))
    IndicationType = np.uint32(int("0014201A", 16))
    IndicationDisposition = np.uint32(int("0014201C", 16))
    IndicationROISequence = np.uint32(int("0014201E", 16))
    IndicationPhysicalPropertySequence = np.uint32(int("00142030", 16))
    PropertyLabel = np.uint32(int("00142032", 16))
    CoordinateSystemNumberofAxes = np.uint32(int("00142202", 16))
    CoordinateSystemAxesSequence = np.uint32(int("00142204", 16))
    CoordinateSystemAxisDescription = np.uint32(int("00142206", 16))
    CoordinateSystemDataSetMapping = np.uint32(int("00142208", 16))
    CoordinateSystemAxisNumber = np.uint32(int("0014220A", 16))
    CoordinateSystemAxisType = np.uint32(int("0014220C", 16))
    CoordinateSystemAxisUnits = np.uint32(int("0014220E", 16))
    CoordinateSystemAxisValues = np.uint32(int("00142210", 16))
    CoordinateSystemTransformSequence = np.uint32(int("00142220", 16))
    TransformDescription = np.uint32(int("00142222", 16))
    TransformNumberofAxes = np.uint32(int("00142224", 16))
    TransformOrderofAxes = np.uint32(int("00142226", 16))
    TransformedAxisUnits = np.uint32(int("00142228", 16))
    CoordinateSystemTransformRotationandScaleMatrix = np.uint32(int("0014222A", 16))
    CoordinateSystemTransformTranslationMatrix = np.uint32(int("0014222C", 16))
    InternalDetectorFrameTime = np.uint32(int("00143011", 16))
    NumberofFramesIntegrated = np.uint32(int("00143012", 16))
    DetectorTemperatureSequence = np.uint32(int("00143020", 16))
    SensorName = np.uint32(int("00143022", 16))
    HorizontalOffsetofSensor = np.uint32(int("00143024", 16))
    VerticalOffsetofSensor = np.uint32(int("00143026", 16))
    SensorTemperature = np.uint32(int("00143028", 16))
    DarkCurrentSequence = np.uint32(int("00143040", 16))
    DarkCurrentCounts = np.uint32(int("00143050", 16))
    GainCorrectionReferenceSequence = np.uint32(int("00143060", 16))
    AirCounts = np.uint32(int("00143070", 16))
    KVUsedinGainCalibration = np.uint32(int("00143071", 16))
    MAUsedinGainCalibration = np.uint32(int("00143072", 16))
    NumberofFramesUsedforIntegration = np.uint32(int("00143073", 16))
    FilterMaterialUsedinGainCalibration = np.uint32(int("00143074", 16))
    FilterThicknessUsedinGainCalibration = np.uint32(int("00143075", 16))
    DateofGainCalibration = np.uint32(int("00143076", 16))
    TimeofGainCalibration = np.uint32(int("00143077", 16))
    BadPixelImage = np.uint32(int("00143080", 16))
    CalibrationNotes = np.uint32(int("00143099", 16))
    PulserEquipmentSequence = np.uint32(int("00144002", 16))
    PulserType = np.uint32(int("00144004", 16))
    PulserNotes = np.uint32(int("00144006", 16))
    ReceiverEquipmentSequence = np.uint32(int("00144008", 16))
    AmplifierType = np.uint32(int("0014400A", 16))
    ReceiverNotes = np.uint32(int("0014400C", 16))
    PreAmplifierEquipmentSequence = np.uint32(int("0014400E", 16))
    PreAmplifierNotes = np.uint32(int("0014400F", 16))
    TransmitTransducerSequence = np.uint32(int("00144010", 16))
    ReceiveTransducerSequence = np.uint32(int("00144011", 16))
    NumberofElements = np.uint32(int("00144012", 16))
    ElementShape = np.uint32(int("00144013", 16))
    ElementDimensionA = np.uint32(int("00144014", 16))
    ElementDimensionB = np.uint32(int("00144015", 16))
    ElementPitchA = np.uint32(int("00144016", 16))
    MeasuredBeamDimensionA = np.uint32(int("00144017", 16))
    MeasuredBeamDimensionB = np.uint32(int("00144018", 16))
    LocationofMeasuredBeamDiameter = np.uint32(int("00144019", 16))
    NominalFrequency = np.uint32(int("0014401A", 16))
    MeasuredCenterFrequency = np.uint32(int("0014401B", 16))
    MeasuredBandwidth = np.uint32(int("0014401C", 16))
    ElementPitchB = np.uint32(int("0014401D", 16))
    PulserSettingsSequence = np.uint32(int("00144020", 16))
    PulseWidth = np.uint32(int("00144022", 16))
    ExcitationFrequency = np.uint32(int("00144024", 16))
    ModulationType = np.uint32(int("00144026", 16))
    Damping = np.uint32(int("00144028", 16))
    ReceiverSettingsSequence = np.uint32(int("00144030", 16))
    AcquiredSoundpathLength = np.uint32(int("00144031", 16))
    AcquisitionCompressionType = np.uint32(int("00144032", 16))
    AcquisitionSampleSize = np.uint32(int("00144033", 16))
    RectifierSmoothing = np.uint32(int("00144034", 16))
    DACSequence = np.uint32(int("00144035", 16))
    DACType = np.uint32(int("00144036", 16))
    DACGainPoints = np.uint32(int("00144038", 16))
    DACTimePoints = np.uint32(int("0014403A", 16))
    DACAmplitude = np.uint32(int("0014403C", 16))
    PreAmplifierSettingsSequence = np.uint32(int("00144040", 16))
    TransmitTransducerSettingsSequence = np.uint32(int("00144050", 16))
    ReceiveTransducerSettingsSequence = np.uint32(int("00144051", 16))
    IncidentAngle = np.uint32(int("00144052", 16))
    CouplingTechnique = np.uint32(int("00144054", 16))
    CouplingMedium = np.uint32(int("00144056", 16))
    CouplingVelocity = np.uint32(int("00144057", 16))
    ProbeCenterLocationX = np.uint32(int("00144058", 16))
    ProbeCenterLocationZ = np.uint32(int("00144059", 16))
    SoundPathLength = np.uint32(int("0014405A", 16))
    DelayLawIdentifier = np.uint32(int("0014405C", 16))
    GateSettingsSequence = np.uint32(int("00144060", 16))
    GateThreshold = np.uint32(int("00144062", 16))
    VelocityofSound = np.uint32(int("00144064", 16))
    CalibrationSettingsSequence = np.uint32(int("00144070", 16))
    CalibrationProcedure = np.uint32(int("00144072", 16))
    ProcedureVersion = np.uint32(int("00144074", 16))
    ProcedureCreationDate = np.uint32(int("00144076", 16))
    ProcedureExpirationDate = np.uint32(int("00144078", 16))
    ProcedureLastModifiedDate = np.uint32(int("0014407A", 16))
    CalibrationTime = np.uint32(int("0014407C", 16))
    CalibrationDate = np.uint32(int("0014407E", 16))
    ProbeDriveEquipmentSequence = np.uint32(int("00144080", 16))
    DriveType = np.uint32(int("00144081", 16))
    ProbeDriveNotes = np.uint32(int("00144082", 16))
    DriveProbeSequence = np.uint32(int("00144083", 16))
    ProbeInductance = np.uint32(int("00144084", 16))
    ProbeResistance = np.uint32(int("00144085", 16))
    ReceiveProbeSequence = np.uint32(int("00144086", 16))
    ProbeDriveSettingsSequence = np.uint32(int("00144087", 16))
    BridgeResistors = np.uint32(int("00144088", 16))
    ProbeOrientationAngle = np.uint32(int("00144089", 16))
    UserSelectedGainY = np.uint32(int("0014408B", 16))
    UserSelectedPhase = np.uint32(int("0014408C", 16))
    UserSelectedOffsetX = np.uint32(int("0014408D", 16))
    UserSelectedOffsetY = np.uint32(int("0014408E", 16))
    ChannelSettingsSequence = np.uint32(int("00144091", 16))
    ChannelThreshold = np.uint32(int("00144092", 16))
    ScannerSettingsSequence = np.uint32(int("0014409A", 16))
    ScanProcedure = np.uint32(int("0014409B", 16))
    TranslationRateX = np.uint32(int("0014409C", 16))
    TranslationRateY = np.uint32(int("0014409D", 16))
    ChannelOverlap = np.uint32(int("0014409F", 16))
    ImageQualityIndicatorType = np.uint32(int("001440A0", 16))
    ImageQualityIndicatorMaterial = np.uint32(int("001440A1", 16))
    ImageQualityIndicatorSize = np.uint32(int("001440A2", 16))
    LINACEnergy = np.uint32(int("00145002", 16))
    LINACOutput = np.uint32(int("00145004", 16))
    ActiveAperture = np.uint32(int("00145100", 16))
    TotalAperture = np.uint32(int("00145101", 16))
    ApertureElevation = np.uint32(int("00145102", 16))
    MainLobeAngle = np.uint32(int("00145103", 16))
    MainRoofAngle = np.uint32(int("00145104", 16))
    ConnectorType = np.uint32(int("00145105", 16))
    WedgeModelNumber = np.uint32(int("00145106", 16))
    WedgeAngleFloat = np.uint32(int("00145107", 16))
    WedgeRoofAngle = np.uint32(int("00145108", 16))
    WedgeElement1Position = np.uint32(int("00145109", 16))
    WedgeMaterialVelocity = np.uint32(int("0014510A", 16))
    WedgeMaterial = np.uint32(int("0014510B", 16))
    WedgeOffsetZ = np.uint32(int("0014510C", 16))
    WedgeOriginOffsetX = np.uint32(int("0014510D", 16))
    WedgeTimeDelay = np.uint32(int("0014510E", 16))
    WedgeName = np.uint32(int("0014510F", 16))
    WedgeManufacturerName = np.uint32(int("00145110", 16))
    WedgeDescription = np.uint32(int("00145111", 16))
    NominalBeamAngle = np.uint32(int("00145112", 16))
    WedgeOffsetX = np.uint32(int("00145113", 16))
    WedgeOffsetY = np.uint32(int("00145114", 16))
    WedgeTotalLength = np.uint32(int("00145115", 16))
    WedgeInContactLength = np.uint32(int("00145116", 16))
    WedgeFrontGap = np.uint32(int("00145117", 16))
    WedgeTotalHeight = np.uint32(int("00145118", 16))
    WedgeFrontHeight = np.uint32(int("00145119", 16))
    WedgeRearHeight = np.uint32(int("0014511A", 16))
    WedgeTotalWidth = np.uint32(int("0014511B", 16))
    WedgeInContactWidth = np.uint32(int("0014511C", 16))
    WedgeChamferHeight = np.uint32(int("0014511D", 16))
    WedgeCurve = np.uint32(int("0014511E", 16))
    RadiusAlongtheWedge = np.uint32(int("0014511F", 16))
    ContrastBolusAgent = np.uint32(int("00180010", 16))
    ContrastBolusAgentSequence = np.uint32(int("00180012", 16))
    ContrastBolusT1Relaxivity = np.uint32(int("00180013", 16))
    ContrastBolusAdministrationRouteSequence = np.uint32(int("00180014", 16))
    BodyPartExamined = np.uint32(int("00180015", 16))
    ScanningSequence = np.uint32(int("00180020", 16))
    SequenceVariant = np.uint32(int("00180021", 16))
    ScanOptions = np.uint32(int("00180022", 16))
    MRAcquisitionType = np.uint32(int("00180023", 16))
    SequenceName = np.uint32(int("00180024", 16))
    AngioFlag = np.uint32(int("00180025", 16))
    InterventionDrugInformationSequence = np.uint32(int("00180026", 16))
    InterventionDrugStopTime = np.uint32(int("00180027", 16))
    InterventionDrugDose = np.uint32(int("00180028", 16))
    InterventionDrugCodeSequence = np.uint32(int("00180029", 16))
    AdditionalDrugSequence = np.uint32(int("0018002A", 16))
    Radionuclide = np.uint32(int("00180030", 16))
    Radiopharmaceutical = np.uint32(int("00180031", 16))
    EnergyWindowCenterline = np.uint32(int("00180032", 16))
    EnergyWindowTotalWidth = np.uint32(int("00180033", 16))
    InterventionDrugName = np.uint32(int("00180034", 16))
    InterventionDrugStartTime = np.uint32(int("00180035", 16))
    InterventionSequence = np.uint32(int("00180036", 16))
    TherapyType = np.uint32(int("00180037", 16))
    InterventionStatus = np.uint32(int("00180038", 16))
    TherapyDescription = np.uint32(int("00180039", 16))
    InterventionDescription = np.uint32(int("0018003A", 16))
    CineRate = np.uint32(int("00180040", 16))
    InitialCineRunState = np.uint32(int("00180042", 16))
    SliceThickness = np.uint32(int("00180050", 16))
    KVP = np.uint32(int("00180060", 16))
    CountsAccumulated = np.uint32(int("00180070", 16))
    AcquisitionTerminationCondition = np.uint32(int("00180071", 16))
    EffectiveDuration = np.uint32(int("00180072", 16))
    AcquisitionStartCondition = np.uint32(int("00180073", 16))
    AcquisitionStartConditionData = np.uint32(int("00180074", 16))
    AcquisitionTerminationConditionData = np.uint32(int("00180075", 16))
    RepetitionTime = np.uint32(int("00180080", 16))
    EchoTime = np.uint32(int("00180081", 16))
    InversionTime = np.uint32(int("00180082", 16))
    NumberofAverages = np.uint32(int("00180083", 16))
    ImagingFrequency = np.uint32(int("00180084", 16))
    ImagedNucleus = np.uint32(int("00180085", 16))
    EchoNumbers = np.uint32(int("00180086", 16))
    MagneticFieldStrength = np.uint32(int("00180087", 16))
    SpacingBetweenSlices = np.uint32(int("00180088", 16))
    NumberofPhaseEncodingSteps = np.uint32(int("00180089", 16))
    DataCollectionDiameter = np.uint32(int("00180090", 16))
    EchoTrainLength = np.uint32(int("00180091", 16))
    PercentSampling = np.uint32(int("00180093", 16))
    PercentPhaseFieldofView = np.uint32(int("00180094", 16))
    PixelBandwidth = np.uint32(int("00180095", 16))
    DeviceSerialNumber = np.uint32(int("00181000", 16))
    DeviceUID = np.uint32(int("00181002", 16))
    DeviceID = np.uint32(int("00181003", 16))
    PlateID = np.uint32(int("00181004", 16))
    GeneratorID = np.uint32(int("00181005", 16))
    GridID = np.uint32(int("00181006", 16))
    CassetteID = np.uint32(int("00181007", 16))
    GantryID = np.uint32(int("00181008", 16))
    SecondaryCaptureDeviceID = np.uint32(int("00181010", 16))
    HardcopyCreationDeviceID = np.uint32(int("00181011", 16))
    DateofSecondaryCapture = np.uint32(int("00181012", 16))
    TimeofSecondaryCapture = np.uint32(int("00181014", 16))
    SecondaryCaptureDeviceManufacturer = np.uint32(int("00181016", 16))
    HardcopyDeviceManufacturer = np.uint32(int("00181017", 16))
    SecondaryCaptureDeviceManufacturersModelName = np.uint32(int("00181018", 16))
    SecondaryCaptureDeviceSoftwareVersions = np.uint32(int("00181019", 16))
    HardcopyDeviceSoftwareVersion = np.uint32(int("0018101A", 16))
    HardcopyDeviceManufacturersModelName = np.uint32(int("0018101B", 16))
    SoftwareVersions = np.uint32(int("00181020", 16))
    VideoImageFormatAcquired = np.uint32(int("00181022", 16))
    DigitalImageFormatAcquired = np.uint32(int("00181023", 16))
    ProtocolName = np.uint32(int("00181030", 16))
    ContrastBolusRoute = np.uint32(int("00181040", 16))
    ContrastBolusVolume = np.uint32(int("00181041", 16))
    ContrastBolusStartTime = np.uint32(int("00181042", 16))
    ContrastBolusStopTime = np.uint32(int("00181043", 16))
    ContrastBolusTotalDose = np.uint32(int("00181044", 16))
    SyringeCounts = np.uint32(int("00181045", 16))
    ContrastFlowRate = np.uint32(int("00181046", 16))
    ContrastFlowDuration = np.uint32(int("00181047", 16))
    ContrastBolusIngredient = np.uint32(int("00181048", 16))
    ContrastBolusIngredientConcentration = np.uint32(int("00181049", 16))
    SpatialResolution = np.uint32(int("00181050", 16))
    TriggerTime = np.uint32(int("00181060", 16))
    TriggerSourceorType = np.uint32(int("00181061", 16))
    NominalInterval = np.uint32(int("00181062", 16))
    FrameTime = np.uint32(int("00181063", 16))
    CardiacFramingType = np.uint32(int("00181064", 16))
    FrameTimeVector = np.uint32(int("00181065", 16))
    FrameDelay = np.uint32(int("00181066", 16))
    ImageTriggerDelay = np.uint32(int("00181067", 16))
    MultiplexGroupTimeOffset = np.uint32(int("00181068", 16))
    TriggerTimeOffset = np.uint32(int("00181069", 16))
    SynchronizationTrigger = np.uint32(int("0018106A", 16))
    SynchronizationChannel = np.uint32(int("0018106C", 16))
    TriggerSamplePosition = np.uint32(int("0018106E", 16))
    RadiopharmaceuticalRoute = np.uint32(int("00181070", 16))
    RadiopharmaceuticalVolume = np.uint32(int("00181071", 16))
    RadiopharmaceuticalStartTime = np.uint32(int("00181072", 16))
    RadiopharmaceuticalStopTime = np.uint32(int("00181073", 16))
    RadionuclideTotalDose = np.uint32(int("00181074", 16))
    RadionuclideHalfLife = np.uint32(int("00181075", 16))
    RadionuclidePositronFraction = np.uint32(int("00181076", 16))
    RadiopharmaceuticalSpecificActivity = np.uint32(int("00181077", 16))
    RadiopharmaceuticalStartDateTime = np.uint32(int("00181078", 16))
    RadiopharmaceuticalStopDateTime = np.uint32(int("00181079", 16))
    BeatRejectionFlag = np.uint32(int("00181080", 16))
    LowRRValue = np.uint32(int("00181081", 16))
    HighRRValue = np.uint32(int("00181082", 16))
    IntervalsAcquired = np.uint32(int("00181083", 16))
    IntervalsRejected = np.uint32(int("00181084", 16))
    PVCRejection = np.uint32(int("00181085", 16))
    SkipBeats = np.uint32(int("00181086", 16))
    HeartRate = np.uint32(int("00181088", 16))
    CardiacNumberofImages = np.uint32(int("00181090", 16))
    TriggerWindow = np.uint32(int("00181094", 16))
    ReconstructionDiameter = np.uint32(int("00181100", 16))
    DistanceSourcetoDetector = np.uint32(int("00181110", 16))
    DistanceSourcetoPatient = np.uint32(int("00181111", 16))
    EstimatedRadiographicMagnificationFactor = np.uint32(int("00181114", 16))
    GantryDetectorTilt = np.uint32(int("00181120", 16))
    GantryDetectorSlew = np.uint32(int("00181121", 16))
    TableHeight = np.uint32(int("00181130", 16))
    TableTraverse = np.uint32(int("00181131", 16))
    TableMotion = np.uint32(int("00181134", 16))
    TableVerticalIncrement = np.uint32(int("00181135", 16))
    TableLateralIncrement = np.uint32(int("00181136", 16))
    TableLongitudinalIncrement = np.uint32(int("00181137", 16))
    TableAngle = np.uint32(int("00181138", 16))
    TableType = np.uint32(int("0018113A", 16))
    RotationDirection = np.uint32(int("00181140", 16))
    AngularPosition = np.uint32(int("00181141", 16))
    RadialPosition = np.uint32(int("00181142", 16))
    ScanArc = np.uint32(int("00181143", 16))
    AngularStep = np.uint32(int("00181144", 16))
    CenterofRotationOffset = np.uint32(int("00181145", 16))
    RotationOffset = np.uint32(int("00181146", 16))
    FieldofViewShape = np.uint32(int("00181147", 16))
    FieldofViewDimensions = np.uint32(int("00181149", 16))
    ExposureTime = np.uint32(int("00181150", 16))
    XRayTubeCurrent = np.uint32(int("00181151", 16))
    Exposure = np.uint32(int("00181152", 16))
    ExposureinAs = np.uint32(int("00181153", 16))
    AveragePulseWidth = np.uint32(int("00181154", 16))
    RadiationSetting = np.uint32(int("00181155", 16))
    RectificationType = np.uint32(int("00181156", 16))
    RadiationMode = np.uint32(int("0018115A", 16))
    ImageandFluoroscopyAreaDoseProduct = np.uint32(int("0018115E", 16))
    FilterType = np.uint32(int("00181160", 16))
    TypeofFilters = np.uint32(int("00181161", 16))
    IntensifierSize = np.uint32(int("00181162", 16))
    ImagerPixelSpacing = np.uint32(int("00181164", 16))
    Grid = np.uint32(int("00181166", 16))
    GeneratorPower = np.uint32(int("00181170", 16))
    CollimatorgridName = np.uint32(int("00181180", 16))
    CollimatorType = np.uint32(int("00181181", 16))
    FocalDistance = np.uint32(int("00181182", 16))
    XFocusCenter = np.uint32(int("00181183", 16))
    YFocusCenter = np.uint32(int("00181184", 16))
    FocalSpots = np.uint32(int("00181190", 16))
    AnodeTargetMaterial = np.uint32(int("00181191", 16))
    BodyPartThickness = np.uint32(int("001811A0", 16))
    CompressionForce = np.uint32(int("001811A2", 16))
    PaddleDescription = np.uint32(int("001811A4", 16))
    DateofLastCalibration = np.uint32(int("00181200", 16))
    TimeofLastCalibration = np.uint32(int("00181201", 16))
    DateTimeofLastCalibration = np.uint32(int("00181202", 16))
    ConvolutionKernel = np.uint32(int("00181210", 16))
    UpperLowerPixelValues = np.uint32(int("00181240", 16))
    ActualFrameDuration = np.uint32(int("00181242", 16))
    CountRate = np.uint32(int("00181243", 16))
    PreferredPlaybackSequencing = np.uint32(int("00181244", 16))
    ReceiveCoilName = np.uint32(int("00181250", 16))
    TransmitCoilName = np.uint32(int("00181251", 16))
    PlateType = np.uint32(int("00181260", 16))
    PhosphorType = np.uint32(int("00181261", 16))
    WaterEquivalentDiameter = np.uint32(int("00181271", 16))
    WaterEquivalentDiameterCalculationMethodCodeSequence = np.uint32(
        int("00181272", 16)
    )
    ScanVelocity = np.uint32(int("00181300", 16))
    WholeBodyTechnique = np.uint32(int("00181301", 16))
    ScanLength = np.uint32(int("00181302", 16))
    AcquisitionMatrix = np.uint32(int("00181310", 16))
    InplanePhaseEncodingDirection = np.uint32(int("00181312", 16))
    FlipAngle = np.uint32(int("00181314", 16))
    VariableFlipAngleFlag = np.uint32(int("00181315", 16))
    SAR = np.uint32(int("00181316", 16))
    dBdt = np.uint32(int("00181318", 16))
    B1rms = np.uint32(int("00181320", 16))
    AcquisitionDeviceProcessingDescription = np.uint32(int("00181400", 16))
    AcquisitionDeviceProcessingCode = np.uint32(int("00181401", 16))
    CassetteOrientation = np.uint32(int("00181402", 16))
    CassetteSize = np.uint32(int("00181403", 16))
    ExposuresonPlate = np.uint32(int("00181404", 16))
    RelativeXRayExposure = np.uint32(int("00181405", 16))
    ExposureIndex = np.uint32(int("00181411", 16))
    TargetExposureIndex = np.uint32(int("00181412", 16))
    DeviationIndex = np.uint32(int("00181413", 16))
    ColumnAngulation = np.uint32(int("00181450", 16))
    TomoLayerHeight = np.uint32(int("00181460", 16))
    TomoAngle = np.uint32(int("00181470", 16))
    TomoTime = np.uint32(int("00181480", 16))
    TomoType = np.uint32(int("00181490", 16))
    TomoClass = np.uint32(int("00181491", 16))
    NumberofTomosynthesisSourceImages = np.uint32(int("00181495", 16))
    PositionerMotion = np.uint32(int("00181500", 16))
    PositionerType = np.uint32(int("00181508", 16))
    PositionerPrimaryAngle = np.uint32(int("00181510", 16))
    PositionerSecondaryAngle = np.uint32(int("00181511", 16))
    PositionerPrimaryAngleIncrement = np.uint32(int("00181520", 16))
    PositionerSecondaryAngleIncrement = np.uint32(int("00181521", 16))
    DetectorPrimaryAngle = np.uint32(int("00181530", 16))
    DetectorSecondaryAngle = np.uint32(int("00181531", 16))
    ShutterShape = np.uint32(int("00181600", 16))
    ShutterLeftVerticalEdge = np.uint32(int("00181602", 16))
    ShutterRightVerticalEdge = np.uint32(int("00181604", 16))
    ShutterUpperHorizontalEdge = np.uint32(int("00181606", 16))
    ShutterLowerHorizontalEdge = np.uint32(int("00181608", 16))
    CenterofCircularShutter = np.uint32(int("00181610", 16))
    RadiusofCircularShutter = np.uint32(int("00181612", 16))
    VerticesofthePolygonalShutter = np.uint32(int("00181620", 16))
    ShutterPresentationValue = np.uint32(int("00181622", 16))
    ShutterOverlayGroup = np.uint32(int("00181623", 16))
    ShutterPresentationColorCIELabValue = np.uint32(int("00181624", 16))
    CollimatorShape = np.uint32(int("00181700", 16))
    CollimatorLeftVerticalEdge = np.uint32(int("00181702", 16))
    CollimatorRightVerticalEdge = np.uint32(int("00181704", 16))
    CollimatorUpperHorizontalEdge = np.uint32(int("00181706", 16))
    CollimatorLowerHorizontalEdge = np.uint32(int("00181708", 16))
    CenterofCircularCollimator = np.uint32(int("00181710", 16))
    RadiusofCircularCollimator = np.uint32(int("00181712", 16))
    VerticesofthePolygonalCollimator = np.uint32(int("00181720", 16))
    AcquisitionTimeSynchronized = np.uint32(int("00181800", 16))
    TimeSource = np.uint32(int("00181801", 16))
    TimeDistributionProtocol = np.uint32(int("00181802", 16))
    NTPSourceAddress = np.uint32(int("00181803", 16))
    PageNumberVector = np.uint32(int("00182001", 16))
    FrameLabelVector = np.uint32(int("00182002", 16))
    FramePrimaryAngleVector = np.uint32(int("00182003", 16))
    FrameSecondaryAngleVector = np.uint32(int("00182004", 16))
    SliceLocationVector = np.uint32(int("00182005", 16))
    DisplayWindowLabelVector = np.uint32(int("00182006", 16))
    NominalScannedPixelSpacing = np.uint32(int("00182010", 16))
    DigitizingDeviceTransportDirection = np.uint32(int("00182020", 16))
    RotationofScannedFilm = np.uint32(int("00182030", 16))
    BiopsyTargetSequence = np.uint32(int("00182041", 16))
    TargetUID = np.uint32(int("00182042", 16))
    LocalizingCursorPosition = np.uint32(int("00182043", 16))
    CalculatedTargetPosition = np.uint32(int("00182044", 16))
    TargetLabel = np.uint32(int("00182045", 16))
    DisplayedZValue = np.uint32(int("00182046", 16))
    IVUSAcquisition = np.uint32(int("00183100", 16))
    IVUSPullbackRate = np.uint32(int("00183101", 16))
    IVUSGatedRate = np.uint32(int("00183102", 16))
    IVUSPullbackStartFrameNumber = np.uint32(int("00183103", 16))
    IVUSPullbackStopFrameNumber = np.uint32(int("00183104", 16))
    LesionNumber = np.uint32(int("00183105", 16))
    AcquisitionComments = np.uint32(int("00184000", 16))
    OutputPower = np.uint32(int("00185000", 16))
    TransducerData = np.uint32(int("00185010", 16))
    FocusDepth = np.uint32(int("00185012", 16))
    ProcessingFunction = np.uint32(int("00185020", 16))
    PostprocessingFunction = np.uint32(int("00185021", 16))
    MechanicalIndex = np.uint32(int("00185022", 16))
    BoneThermalIndex = np.uint32(int("00185024", 16))
    CranialThermalIndex = np.uint32(int("00185026", 16))
    SoftTissueThermalIndex = np.uint32(int("00185027", 16))
    SoftTissuefocusThermalIndex = np.uint32(int("00185028", 16))
    SoftTissuesurfaceThermalIndex = np.uint32(int("00185029", 16))
    DynamicRange = np.uint32(int("00185030", 16))
    TotalGain = np.uint32(int("00185040", 16))
    DepthofScanField = np.uint32(int("00185050", 16))
    PatientPosition = np.uint32(int("00185100", 16))
    ViewPosition = np.uint32(int("00185101", 16))
    ProjectionEponymousNameCodeSequence = np.uint32(int("00185104", 16))
    ImageTransformationMatrix = np.uint32(int("00185210", 16))
    ImageTranslationVector = np.uint32(int("00185212", 16))
    Sensitivity = np.uint32(int("00186000", 16))
    SequenceofUltrasoundRegions = np.uint32(int("00186011", 16))
    RegionSpatialFormat = np.uint32(int("00186012", 16))
    RegionDataType = np.uint32(int("00186014", 16))
    RegionFlags = np.uint32(int("00186016", 16))
    RegionLocationMinX0 = np.uint32(int("00186018", 16))
    RegionLocationMinY0 = np.uint32(int("0018601A", 16))
    RegionLocationMaxX1 = np.uint32(int("0018601C", 16))
    RegionLocationMaxY1 = np.uint32(int("0018601E", 16))
    ReferencePixelX0 = np.uint32(int("00186020", 16))
    ReferencePixelY0 = np.uint32(int("00186022", 16))
    PhysicalUnitsXDirection = np.uint32(int("00186024", 16))
    PhysicalUnitsYDirection = np.uint32(int("00186026", 16))
    ReferencePixelPhysicalValueX = np.uint32(int("00186028", 16))
    ReferencePixelPhysicalValueY = np.uint32(int("0018602A", 16))
    PhysicalDeltaX = np.uint32(int("0018602C", 16))
    PhysicalDeltaY = np.uint32(int("0018602E", 16))
    TransducerFrequency = np.uint32(int("00186030", 16))
    TransducerType = np.uint32(int("00186031", 16))
    PulseRepetitionFrequency = np.uint32(int("00186032", 16))
    DopplerCorrectionAngle = np.uint32(int("00186034", 16))
    SteeringAngle = np.uint32(int("00186036", 16))
    DopplerSampleVolumeXPositionRetired = np.uint32(int("00186038", 16))
    DopplerSampleVolumeXPosition = np.uint32(int("00186039", 16))
    DopplerSampleVolumeYPositionRetired = np.uint32(int("0018603A", 16))
    DopplerSampleVolumeYPosition = np.uint32(int("0018603B", 16))
    TMLinePositionX0Retired = np.uint32(int("0018603C", 16))
    TMLinePositionX0 = np.uint32(int("0018603D", 16))
    TMLinePositionY0Retired = np.uint32(int("0018603E", 16))
    TMLinePositionY0 = np.uint32(int("0018603F", 16))
    TMLinePositionX1Retired = np.uint32(int("00186040", 16))
    TMLinePositionX1 = np.uint32(int("00186041", 16))
    TMLinePositionY1Retired = np.uint32(int("00186042", 16))
    TMLinePositionY1 = np.uint32(int("00186043", 16))
    PixelComponentOrganization = np.uint32(int("00186044", 16))
    PixelComponentMask = np.uint32(int("00186046", 16))
    PixelComponentRangeStart = np.uint32(int("00186048", 16))
    PixelComponentRangeStop = np.uint32(int("0018604A", 16))
    PixelComponentPhysicalUnits = np.uint32(int("0018604C", 16))
    PixelComponentDataType = np.uint32(int("0018604E", 16))
    NumberofTableBreakPoints = np.uint32(int("00186050", 16))
    TableofXBreakPoints = np.uint32(int("00186052", 16))
    TableofYBreakPoints = np.uint32(int("00186054", 16))
    NumberofTableEntries = np.uint32(int("00186056", 16))
    TableofPixelValues = np.uint32(int("00186058", 16))
    TableofParameterValues = np.uint32(int("0018605A", 16))
    RWaveTimeVector = np.uint32(int("00186060", 16))
    DetectorConditionsNominalFlag = np.uint32(int("00187000", 16))
    DetectorTemperature = np.uint32(int("00187001", 16))
    DetectorType = np.uint32(int("00187004", 16))
    DetectorConfiguration = np.uint32(int("00187005", 16))
    DetectorDescription = np.uint32(int("00187006", 16))
    DetectorMode = np.uint32(int("00187008", 16))
    DetectorID = np.uint32(int("0018700A", 16))
    DateofLastDetectorCalibration = np.uint32(int("0018700C", 16))
    TimeofLastDetectorCalibration = np.uint32(int("0018700E", 16))
    ExposuresonDetectorSinceLastCalibration = np.uint32(int("00187010", 16))
    ExposuresonDetectorSinceManufactured = np.uint32(int("00187011", 16))
    DetectorTimeSinceLastExposure = np.uint32(int("00187012", 16))
    DetectorActiveTime = np.uint32(int("00187014", 16))
    DetectorActivationOffsetFromExposure = np.uint32(int("00187016", 16))
    DetectorBinning = np.uint32(int("0018701A", 16))
    DetectorElementPhysicalSize = np.uint32(int("00187020", 16))
    DetectorElementSpacing = np.uint32(int("00187022", 16))
    DetectorActiveShape = np.uint32(int("00187024", 16))
    DetectorActiveDimensions = np.uint32(int("00187026", 16))
    DetectorActiveOrigin = np.uint32(int("00187028", 16))
    DetectorManufacturerName = np.uint32(int("0018702A", 16))
    DetectorManufacturersModelName = np.uint32(int("0018702B", 16))
    FieldofViewOrigin = np.uint32(int("00187030", 16))
    FieldofViewRotation = np.uint32(int("00187032", 16))
    FieldofViewHorizontalFlip = np.uint32(int("00187034", 16))
    PixelDataAreaOriginRelativeToFOV = np.uint32(int("00187036", 16))
    PixelDataAreaRotationAngleRelativeToFOV = np.uint32(int("00187038", 16))
    GridAbsorbingMaterial = np.uint32(int("00187040", 16))
    GridSpacingMaterial = np.uint32(int("00187041", 16))
    GridThickness = np.uint32(int("00187042", 16))
    GridPitch = np.uint32(int("00187044", 16))
    GridAspectRatio = np.uint32(int("00187046", 16))
    GridPeriod = np.uint32(int("00187048", 16))
    GridFocalDistance = np.uint32(int("0018704C", 16))
    FilterMaterial = np.uint32(int("00187050", 16))
    FilterThicknessMinimum = np.uint32(int("00187052", 16))
    FilterThicknessMaximum = np.uint32(int("00187054", 16))
    FilterBeamPathLengthMinimum = np.uint32(int("00187056", 16))
    FilterBeamPathLengthMaximum = np.uint32(int("00187058", 16))
    ExposureControlMode = np.uint32(int("00187060", 16))
    ExposureControlModeDescription = np.uint32(int("00187062", 16))
    ExposureStatus = np.uint32(int("00187064", 16))
    PhototimerSetting = np.uint32(int("00187065", 16))
    ExposureTimeinS = np.uint32(int("00188150", 16))
    XRayTubeCurrentinA = np.uint32(int("00188151", 16))
    ContentQualification = np.uint32(int("00189004", 16))
    PulseSequenceName = np.uint32(int("00189005", 16))
    MRImagingModifierSequence = np.uint32(int("00189006", 16))
    EchoPulseSequence = np.uint32(int("00189008", 16))
    InversionRecovery = np.uint32(int("00189009", 16))
    FlowCompensation = np.uint32(int("00189010", 16))
    MultipleSpinEcho = np.uint32(int("00189011", 16))
    MultiplanarExcitation = np.uint32(int("00189012", 16))
    PhaseContrast = np.uint32(int("00189014", 16))
    TimeofFlightContrast = np.uint32(int("00189015", 16))
    Spoiling = np.uint32(int("00189016", 16))
    SteadyStatePulseSequence = np.uint32(int("00189017", 16))
    EchoPlanarPulseSequence = np.uint32(int("00189018", 16))
    TagAngleFirstAxis = np.uint32(int("00189019", 16))
    MagnetizationTransfer = np.uint32(int("00189020", 16))
    T2Preparation = np.uint32(int("00189021", 16))
    BloodSignalNulling = np.uint32(int("00189022", 16))
    SaturationRecovery = np.uint32(int("00189024", 16))
    SpectrallySelectedSuppression = np.uint32(int("00189025", 16))
    SpectrallySelectedExcitation = np.uint32(int("00189026", 16))
    SpatialPresaturation = np.uint32(int("00189027", 16))
    Tagging = np.uint32(int("00189028", 16))
    OversamplingPhase = np.uint32(int("00189029", 16))
    TagSpacingFirstDimension = np.uint32(int("00189030", 16))
    GeometryofkSpaceTraversal = np.uint32(int("00189032", 16))
    SegmentedkSpaceTraversal = np.uint32(int("00189033", 16))
    RectilinearPhaseEncodeReordering = np.uint32(int("00189034", 16))
    TagThickness = np.uint32(int("00189035", 16))
    PartialFourierDirection = np.uint32(int("00189036", 16))
    CardiacSynchronizationTechnique = np.uint32(int("00189037", 16))
    ReceiveCoilManufacturerName = np.uint32(int("00189041", 16))
    MRReceiveCoilSequence = np.uint32(int("00189042", 16))
    ReceiveCoilType = np.uint32(int("00189043", 16))
    QuadratureReceiveCoil = np.uint32(int("00189044", 16))
    MultiCoilDefinitionSequence = np.uint32(int("00189045", 16))
    MultiCoilConfiguration = np.uint32(int("00189046", 16))
    MultiCoilElementName = np.uint32(int("00189047", 16))
    MultiCoilElementUsed = np.uint32(int("00189048", 16))
    MRTransmitCoilSequence = np.uint32(int("00189049", 16))
    TransmitCoilManufacturerName = np.uint32(int("00189050", 16))
    TransmitCoilType = np.uint32(int("00189051", 16))
    SpectralWidth = np.uint32(int("00189052", 16))
    ChemicalShiftReference = np.uint32(int("00189053", 16))
    VolumeLocalizationTechnique = np.uint32(int("00189054", 16))
    MRAcquisitionFrequencyEncodingSteps = np.uint32(int("00189058", 16))
    Decoupling = np.uint32(int("00189059", 16))
    DecoupledNucleus = np.uint32(int("00189060", 16))
    DecouplingFrequency = np.uint32(int("00189061", 16))
    DecouplingMethod = np.uint32(int("00189062", 16))
    DecouplingChemicalShiftReference = np.uint32(int("00189063", 16))
    kspaceFiltering = np.uint32(int("00189064", 16))
    TimeDomainFiltering = np.uint32(int("00189065", 16))
    NumberofZeroFills = np.uint32(int("00189066", 16))
    BaselineCorrection = np.uint32(int("00189067", 16))
    ParallelReductionFactorInplane = np.uint32(int("00189069", 16))
    CardiacRRIntervalSpecified = np.uint32(int("00189070", 16))
    AcquisitionDuration = np.uint32(int("00189073", 16))
    FrameAcquisitionDateTime = np.uint32(int("00189074", 16))
    DiffusionDirectionality = np.uint32(int("00189075", 16))
    DiffusionGradientDirectionSequence = np.uint32(int("00189076", 16))
    ParallelAcquisition = np.uint32(int("00189077", 16))
    ParallelAcquisitionTechnique = np.uint32(int("00189078", 16))
    InversionTimes = np.uint32(int("00189079", 16))
    MetaboliteMapDescription = np.uint32(int("00189080", 16))
    PartialFourier = np.uint32(int("00189081", 16))
    EffectiveEchoTime = np.uint32(int("00189082", 16))
    MetaboliteMapCodeSequence = np.uint32(int("00189083", 16))
    ChemicalShiftSequence = np.uint32(int("00189084", 16))
    CardiacSignalSource = np.uint32(int("00189085", 16))
    Diffusionbvalue = np.uint32(int("00189087", 16))
    DiffusionGradientOrientation = np.uint32(int("00189089", 16))
    VelocityEncodingDirection = np.uint32(int("00189090", 16))
    VelocityEncodingMinimumValue = np.uint32(int("00189091", 16))
    VelocityEncodingAcquisitionSequence = np.uint32(int("00189092", 16))
    NumberofkSpaceTrajectories = np.uint32(int("00189093", 16))
    CoverageofkSpace = np.uint32(int("00189094", 16))
    SpectroscopyAcquisitionPhaseRows = np.uint32(int("00189095", 16))
    ParallelReductionFactorInplaneRetired = np.uint32(int("00189096", 16))
    TransmitterFrequency = np.uint32(int("00189098", 16))
    ResonantNucleus = np.uint32(int("00189100", 16))
    FrequencyCorrection = np.uint32(int("00189101", 16))
    MRSpectroscopyFOVGeometrySequence = np.uint32(int("00189103", 16))
    SlabThickness = np.uint32(int("00189104", 16))
    SlabOrientation = np.uint32(int("00189105", 16))
    MidSlabPosition = np.uint32(int("00189106", 16))
    MRSpatialSaturationSequence = np.uint32(int("00189107", 16))
    MRTimingandRelatedParametersSequence = np.uint32(int("00189112", 16))
    MREchoSequence = np.uint32(int("00189114", 16))
    MRModifierSequence = np.uint32(int("00189115", 16))
    MRDiffusionSequence = np.uint32(int("00189117", 16))
    CardiacSynchronizationSequence = np.uint32(int("00189118", 16))
    MRAveragesSequence = np.uint32(int("00189119", 16))
    MRFOVGeometrySequence = np.uint32(int("00189125", 16))
    VolumeLocalizationSequence = np.uint32(int("00189126", 16))
    SpectroscopyAcquisitionDataColumns = np.uint32(int("00189127", 16))
    DiffusionAnisotropyType = np.uint32(int("00189147", 16))
    FrameReferenceDateTime = np.uint32(int("00189151", 16))
    MRMetaboliteMapSequence = np.uint32(int("00189152", 16))
    ParallelReductionFactoroutofplane = np.uint32(int("00189155", 16))
    SpectroscopyAcquisitionOutofplanePhaseSteps = np.uint32(int("00189159", 16))
    BulkMotionStatus = np.uint32(int("00189166", 16))
    ParallelReductionFactorSecondInplane = np.uint32(int("00189168", 16))
    CardiacBeatRejectionTechnique = np.uint32(int("00189169", 16))
    RespiratoryMotionCompensationTechnique = np.uint32(int("00189170", 16))
    RespiratorySignalSource = np.uint32(int("00189171", 16))
    BulkMotionCompensationTechnique = np.uint32(int("00189172", 16))
    BulkMotionSignalSource = np.uint32(int("00189173", 16))
    ApplicableSafetyStandardAgency = np.uint32(int("00189174", 16))
    ApplicableSafetyStandardDescription = np.uint32(int("00189175", 16))
    OperatingModeSequence = np.uint32(int("00189176", 16))
    OperatingModeType = np.uint32(int("00189177", 16))
    OperatingMode = np.uint32(int("00189178", 16))
    SpecificAbsorptionRateDefinition = np.uint32(int("00189179", 16))
    GradientOutputType = np.uint32(int("00189180", 16))
    SpecificAbsorptionRateValue = np.uint32(int("00189181", 16))
    GradientOutput = np.uint32(int("00189182", 16))
    FlowCompensationDirection = np.uint32(int("00189183", 16))
    TaggingDelay = np.uint32(int("00189184", 16))
    RespiratoryMotionCompensationTechniqueDescription = np.uint32(int("00189185", 16))
    RespiratorySignalSourceID = np.uint32(int("00189186", 16))
    ChemicalShiftMinimumIntegrationLimitinHz = np.uint32(int("00189195", 16))
    ChemicalShiftMaximumIntegrationLimitinHz = np.uint32(int("00189196", 16))
    MRVelocityEncodingSequence = np.uint32(int("00189197", 16))
    FirstOrderPhaseCorrection = np.uint32(int("00189198", 16))
    WaterReferencedPhaseCorrection = np.uint32(int("00189199", 16))
    MRSpectroscopyAcquisitionType = np.uint32(int("00189200", 16))
    RespiratoryCyclePosition = np.uint32(int("00189214", 16))
    VelocityEncodingMaximumValue = np.uint32(int("00189217", 16))
    TagSpacingSecondDimension = np.uint32(int("00189218", 16))
    TagAngleSecondAxis = np.uint32(int("00189219", 16))
    FrameAcquisitionDuration = np.uint32(int("00189220", 16))
    MRImageFrameTypeSequence = np.uint32(int("00189226", 16))
    MRSpectroscopyFrameTypeSequence = np.uint32(int("00189227", 16))
    MRAcquisitionPhaseEncodingStepsinplane = np.uint32(int("00189231", 16))
    MRAcquisitionPhaseEncodingStepsoutofplane = np.uint32(int("00189232", 16))
    SpectroscopyAcquisitionPhaseColumns = np.uint32(int("00189234", 16))
    CardiacCyclePosition = np.uint32(int("00189236", 16))
    SpecificAbsorptionRateSequence = np.uint32(int("00189239", 16))
    RFEchoTrainLength = np.uint32(int("00189240", 16))
    GradientEchoTrainLength = np.uint32(int("00189241", 16))
    ArterialSpinLabelingContrast = np.uint32(int("00189250", 16))
    MRArterialSpinLabelingSequence = np.uint32(int("00189251", 16))
    ASLTechniqueDescription = np.uint32(int("00189252", 16))
    ASLSlabNumber = np.uint32(int("00189253", 16))
    ASLSlabThickness = np.uint32(int("00189254", 16))
    ASLSlabOrientation = np.uint32(int("00189255", 16))
    ASLMidSlabPosition = np.uint32(int("00189256", 16))
    ASLContext = np.uint32(int("00189257", 16))
    ASLPulseTrainDuration = np.uint32(int("00189258", 16))
    ASLCrusherFlag = np.uint32(int("00189259", 16))
    ASLCrusherFlowLimit = np.uint32(int("0018925A", 16))
    ASLCrusherDescription = np.uint32(int("0018925B", 16))
    ASLBolusCutoffFlag = np.uint32(int("0018925C", 16))
    ASLBolusCutoffTimingSequence = np.uint32(int("0018925D", 16))
    ASLBolusCutoffTechnique = np.uint32(int("0018925E", 16))
    ASLBolusCutoffDelayTime = np.uint32(int("0018925F", 16))
    ASLSlabSequence = np.uint32(int("00189260", 16))
    ChemicalShiftMinimumIntegrationLimitinppm = np.uint32(int("00189295", 16))
    ChemicalShiftMaximumIntegrationLimitinppm = np.uint32(int("00189296", 16))
    WaterReferenceAcquisition = np.uint32(int("00189297", 16))
    EchoPeakPosition = np.uint32(int("00189298", 16))
    CTAcquisitionTypeSequence = np.uint32(int("00189301", 16))
    AcquisitionType = np.uint32(int("00189302", 16))
    TubeAngle = np.uint32(int("00189303", 16))
    CTAcquisitionDetailsSequence = np.uint32(int("00189304", 16))
    RevolutionTime = np.uint32(int("00189305", 16))
    SingleCollimationWidth = np.uint32(int("00189306", 16))
    TotalCollimationWidth = np.uint32(int("00189307", 16))
    CTTableDynamicsSequence = np.uint32(int("00189308", 16))
    TableSpeed = np.uint32(int("00189309", 16))
    TableFeedperRotation = np.uint32(int("00189310", 16))
    SpiralPitchFactor = np.uint32(int("00189311", 16))
    CTGeometrySequence = np.uint32(int("00189312", 16))
    DataCollectionCenterPatient = np.uint32(int("00189313", 16))
    CTReconstructionSequence = np.uint32(int("00189314", 16))
    ReconstructionAlgorithm = np.uint32(int("00189315", 16))
    ConvolutionKernelGroup = np.uint32(int("00189316", 16))
    ReconstructionFieldofView = np.uint32(int("00189317", 16))
    ReconstructionTargetCenterPatient = np.uint32(int("00189318", 16))
    ReconstructionAngle = np.uint32(int("00189319", 16))
    ImageFilter = np.uint32(int("00189320", 16))
    CTExposureSequence = np.uint32(int("00189321", 16))
    ReconstructionPixelSpacing = np.uint32(int("00189322", 16))
    ExposureModulationType = np.uint32(int("00189323", 16))
    EstimatedDoseSaving = np.uint32(int("00189324", 16))
    CTXRayDetailsSequence = np.uint32(int("00189325", 16))
    CTPositionSequence = np.uint32(int("00189326", 16))
    TablePosition = np.uint32(int("00189327", 16))
    ExposureTimeinms = np.uint32(int("00189328", 16))
    CTImageFrameTypeSequence = np.uint32(int("00189329", 16))
    XRayTubeCurrentinmA = np.uint32(int("00189330", 16))
    ExposureinmAs = np.uint32(int("00189332", 16))
    ConstantVolumeFlag = np.uint32(int("00189333", 16))
    FluoroscopyFlag = np.uint32(int("00189334", 16))
    DistanceSourcetoDataCollectionCenter = np.uint32(int("00189335", 16))
    ContrastBolusAgentNumber = np.uint32(int("00189337", 16))
    ContrastBolusIngredientCodeSequence = np.uint32(int("00189338", 16))
    ContrastAdministrationProfileSequence = np.uint32(int("00189340", 16))
    ContrastBolusUsageSequence = np.uint32(int("00189341", 16))
    ContrastBolusAgentAdministered = np.uint32(int("00189342", 16))
    ContrastBolusAgentDetected = np.uint32(int("00189343", 16))
    ContrastBolusAgentPhase = np.uint32(int("00189344", 16))
    CTDIvol = np.uint32(int("00189345", 16))
    CTDIPhantomTypeCodeSequence = np.uint32(int("00189346", 16))
    CalciumScoringMassFactorPatient = np.uint32(int("00189351", 16))
    CalciumScoringMassFactorDevice = np.uint32(int("00189352", 16))
    EnergyWeightingFactor = np.uint32(int("00189353", 16))
    CTAdditionalXRaySourceSequence = np.uint32(int("00189360", 16))
    ProjectionPixelCalibrationSequence = np.uint32(int("00189401", 16))
    DistanceSourcetoIsocenter = np.uint32(int("00189402", 16))
    DistanceObjecttoTableTop = np.uint32(int("00189403", 16))
    ObjectPixelSpacinginCenterofBeam = np.uint32(int("00189404", 16))
    PositionerPositionSequence = np.uint32(int("00189405", 16))
    TablePositionSequence = np.uint32(int("00189406", 16))
    CollimatorShapeSequence = np.uint32(int("00189407", 16))
    PlanesinAcquisition = np.uint32(int("00189410", 16))
    XAXRFFrameCharacteristicsSequence = np.uint32(int("00189412", 16))
    FrameAcquisitionSequence = np.uint32(int("00189417", 16))
    XRayReceptorType = np.uint32(int("00189420", 16))
    AcquisitionProtocolName = np.uint32(int("00189423", 16))
    AcquisitionProtocolDescription = np.uint32(int("00189424", 16))
    ContrastBolusIngredientOpaque = np.uint32(int("00189425", 16))
    DistanceReceptorPlanetoDetectorHousing = np.uint32(int("00189426", 16))
    IntensifierActiveShape = np.uint32(int("00189427", 16))
    IntensifierActiveDimensions = np.uint32(int("00189428", 16))
    PhysicalDetectorSize = np.uint32(int("00189429", 16))
    PositionofIsocenterProjection = np.uint32(int("00189430", 16))
    FieldofViewSequence = np.uint32(int("00189432", 16))
    FieldofViewDescription = np.uint32(int("00189433", 16))
    ExposureControlSensingRegionsSequence = np.uint32(int("00189434", 16))
    ExposureControlSensingRegionShape = np.uint32(int("00189435", 16))
    ExposureControlSensingRegionLeftVerticalEdge = np.uint32(int("00189436", 16))
    ExposureControlSensingRegionRightVerticalEdge = np.uint32(int("00189437", 16))
    ExposureControlSensingRegionUpperHorizontalEdge = np.uint32(int("00189438", 16))
    ExposureControlSensingRegionLowerHorizontalEdge = np.uint32(int("00189439", 16))
    CenterofCircularExposureControlSensingRegion = np.uint32(int("00189440", 16))
    RadiusofCircularExposureControlSensingRegion = np.uint32(int("00189441", 16))
    VerticesofthePolygonalExposureControlSensingRegion = np.uint32(int("00189442", 16))
    ColumnAngulationPatient = np.uint32(int("00189447", 16))
    BeamAngle = np.uint32(int("00189449", 16))
    FrameDetectorParametersSequence = np.uint32(int("00189451", 16))
    CalculatedAnatomyThickness = np.uint32(int("00189452", 16))
    CalibrationSequence = np.uint32(int("00189455", 16))
    ObjectThicknessSequence = np.uint32(int("00189456", 16))
    PlaneIdentification = np.uint32(int("00189457", 16))
    FieldofViewDimensionsinFloat = np.uint32(int("00189461", 16))
    IsocenterReferenceSystemSequence = np.uint32(int("00189462", 16))
    PositionerIsocenterPrimaryAngle = np.uint32(int("00189463", 16))
    PositionerIsocenterSecondaryAngle = np.uint32(int("00189464", 16))
    PositionerIsocenterDetectorRotationAngle = np.uint32(int("00189465", 16))
    TableXPositiontoIsocenter = np.uint32(int("00189466", 16))
    TableYPositiontoIsocenter = np.uint32(int("00189467", 16))
    TableZPositiontoIsocenter = np.uint32(int("00189468", 16))
    TableHorizontalRotationAngle = np.uint32(int("00189469", 16))
    TableHeadTiltAngle = np.uint32(int("00189470", 16))
    TableCradleTiltAngle = np.uint32(int("00189471", 16))
    FrameDisplayShutterSequence = np.uint32(int("00189472", 16))
    AcquiredImageAreaDoseProduct = np.uint32(int("00189473", 16))
    CarmPositionerTabletopRelationship = np.uint32(int("00189474", 16))
    XRayGeometrySequence = np.uint32(int("00189476", 16))
    IrradiationEventIdentificationSequence = np.uint32(int("00189477", 16))
    XRay3DFrameTypeSequence = np.uint32(int("00189504", 16))
    ContributingSourcesSequence = np.uint32(int("00189506", 16))
    XRay3DAcquisitionSequence = np.uint32(int("00189507", 16))
    PrimaryPositionerScanArc = np.uint32(int("00189508", 16))
    SecondaryPositionerScanArc = np.uint32(int("00189509", 16))
    PrimaryPositionerScanStartAngle = np.uint32(int("00189510", 16))
    SecondaryPositionerScanStartAngle = np.uint32(int("00189511", 16))
    PrimaryPositionerIncrement = np.uint32(int("00189514", 16))
    SecondaryPositionerIncrement = np.uint32(int("00189515", 16))
    StartAcquisitionDateTime = np.uint32(int("00189516", 16))
    EndAcquisitionDateTime = np.uint32(int("00189517", 16))
    PrimaryPositionerIncrementSign = np.uint32(int("00189518", 16))
    SecondaryPositionerIncrementSign = np.uint32(int("00189519", 16))
    ApplicationName = np.uint32(int("00189524", 16))
    ApplicationVersion = np.uint32(int("00189525", 16))
    ApplicationManufacturer = np.uint32(int("00189526", 16))
    AlgorithmType = np.uint32(int("00189527", 16))
    AlgorithmDescription = np.uint32(int("00189528", 16))
    XRay3DReconstructionSequence = np.uint32(int("00189530", 16))
    ReconstructionDescription = np.uint32(int("00189531", 16))
    PerProjectionAcquisitionSequence = np.uint32(int("00189538", 16))
    DetectorPositionSequence = np.uint32(int("00189541", 16))
    XRayAcquisitionDoseSequence = np.uint32(int("00189542", 16))
    XRaySourceIsocenterPrimaryAngle = np.uint32(int("00189543", 16))
    XRaySourceIsocenterSecondaryAngle = np.uint32(int("00189544", 16))
    BreastSupportIsocenterPrimaryAngle = np.uint32(int("00189545", 16))
    BreastSupportIsocenterSecondaryAngle = np.uint32(int("00189546", 16))
    BreastSupportXPositiontoIsocenter = np.uint32(int("00189547", 16))
    BreastSupportYPositiontoIsocenter = np.uint32(int("00189548", 16))
    BreastSupportZPositiontoIsocenter = np.uint32(int("00189549", 16))
    DetectorIsocenterPrimaryAngle = np.uint32(int("00189550", 16))
    DetectorIsocenterSecondaryAngle = np.uint32(int("00189551", 16))
    DetectorXPositiontoIsocenter = np.uint32(int("00189552", 16))
    DetectorYPositiontoIsocenter = np.uint32(int("00189553", 16))
    DetectorZPositiontoIsocenter = np.uint32(int("00189554", 16))
    XRayGridSequence = np.uint32(int("00189555", 16))
    XRayFilterSequence = np.uint32(int("00189556", 16))
    DetectorActiveAreaTLHCPosition = np.uint32(int("00189557", 16))
    DetectorActiveAreaOrientation = np.uint32(int("00189558", 16))
    PositionerPrimaryAngleDirection = np.uint32(int("00189559", 16))
    DiffusionbmatrixSequence = np.uint32(int("00189601", 16))
    DiffusionbvalueXX = np.uint32(int("00189602", 16))
    DiffusionbvalueXY = np.uint32(int("00189603", 16))
    DiffusionbvalueXZ = np.uint32(int("00189604", 16))
    DiffusionbvalueYY = np.uint32(int("00189605", 16))
    DiffusionbvalueYZ = np.uint32(int("00189606", 16))
    DiffusionbvalueZZ = np.uint32(int("00189607", 16))
    FunctionalMRSequence = np.uint32(int("00189621", 16))
    FunctionalSettlingPhaseFramesPresent = np.uint32(int("00189622", 16))
    FunctionalSyncPulse = np.uint32(int("00189623", 16))
    SettlingPhaseFrame = np.uint32(int("00189624", 16))
    DecayCorrectionDateTime = np.uint32(int("00189701", 16))
    StartDensityThreshold = np.uint32(int("00189715", 16))
    StartRelativeDensityDifferenceThreshold = np.uint32(int("00189716", 16))
    StartCardiacTriggerCountThreshold = np.uint32(int("00189717", 16))
    StartRespiratoryTriggerCountThreshold = np.uint32(int("00189718", 16))
    TerminationCountsThreshold = np.uint32(int("00189719", 16))
    TerminationDensityThreshold = np.uint32(int("00189720", 16))
    TerminationRelativeDensityThreshold = np.uint32(int("00189721", 16))
    TerminationTimeThreshold = np.uint32(int("00189722", 16))
    TerminationCardiacTriggerCountThreshold = np.uint32(int("00189723", 16))
    TerminationRespiratoryTriggerCountThreshold = np.uint32(int("00189724", 16))
    DetectorGeometry = np.uint32(int("00189725", 16))
    TransverseDetectorSeparation = np.uint32(int("00189726", 16))
    AxialDetectorDimension = np.uint32(int("00189727", 16))
    RadiopharmaceuticalAgentNumber = np.uint32(int("00189729", 16))
    PETFrameAcquisitionSequence = np.uint32(int("00189732", 16))
    PETDetectorMotionDetailsSequence = np.uint32(int("00189733", 16))
    PETTableDynamicsSequence = np.uint32(int("00189734", 16))
    PETPositionSequence = np.uint32(int("00189735", 16))
    PETFrameCorrectionFactorsSequence = np.uint32(int("00189736", 16))
    RadiopharmaceuticalUsageSequence = np.uint32(int("00189737", 16))
    AttenuationCorrectionSource = np.uint32(int("00189738", 16))
    NumberofIterations = np.uint32(int("00189739", 16))
    NumberofSubsets = np.uint32(int("00189740", 16))
    PETReconstructionSequence = np.uint32(int("00189749", 16))
    PETFrameTypeSequence = np.uint32(int("00189751", 16))
    TimeofFlightInformationUsed = np.uint32(int("00189755", 16))
    ReconstructionType = np.uint32(int("00189756", 16))
    DecayCorrected = np.uint32(int("00189758", 16))
    AttenuationCorrected = np.uint32(int("00189759", 16))
    ScatterCorrected = np.uint32(int("00189760", 16))
    DeadTimeCorrected = np.uint32(int("00189761", 16))
    GantryMotionCorrected = np.uint32(int("00189762", 16))
    PatientMotionCorrected = np.uint32(int("00189763", 16))
    CountLossNormalizationCorrected = np.uint32(int("00189764", 16))
    RandomsCorrected = np.uint32(int("00189765", 16))
    NonuniformRadialSamplingCorrected = np.uint32(int("00189766", 16))
    SensitivityCalibrated = np.uint32(int("00189767", 16))
    DetectorNormalizationCorrection = np.uint32(int("00189768", 16))
    IterativeReconstructionMethod = np.uint32(int("00189769", 16))
    AttenuationCorrectionTemporalRelationship = np.uint32(int("00189770", 16))
    PatientPhysiologicalStateSequence = np.uint32(int("00189771", 16))
    PatientPhysiologicalStateCodeSequence = np.uint32(int("00189772", 16))
    DepthsofFocus = np.uint32(int("00189801", 16))
    ExcludedIntervalsSequence = np.uint32(int("00189803", 16))
    ExclusionStartDateTime = np.uint32(int("00189804", 16))
    ExclusionDuration = np.uint32(int("00189805", 16))
    USImageDescriptionSequence = np.uint32(int("00189806", 16))
    ImageDataTypeSequence = np.uint32(int("00189807", 16))
    DataType = np.uint32(int("00189808", 16))
    TransducerScanPatternCodeSequence = np.uint32(int("00189809", 16))
    AliasedDataType = np.uint32(int("0018980B", 16))
    PositionMeasuringDeviceUsed = np.uint32(int("0018980C", 16))
    TransducerGeometryCodeSequence = np.uint32(int("0018980D", 16))
    TransducerBeamSteeringCodeSequence = np.uint32(int("0018980E", 16))
    TransducerApplicationCodeSequence = np.uint32(int("0018980F", 16))
    ZeroVelocityPixelValue = np.uint32(int("00189810", 16))
    ContributingEquipmentSequence = np.uint32(int("0018A001", 16))
    ContributionDateTime = np.uint32(int("0018A002", 16))
    ContributionDescription = np.uint32(int("0018A003", 16))
    StudyInstanceUID = np.uint32(int("0020000D", 16))
    SeriesInstanceUID = np.uint32(int("0020000E", 16))
    StudyID = np.uint32(int("00200010", 16))
    SeriesNumber = np.uint32(int("00200011", 16))
    AcquisitionNumber = np.uint32(int("00200012", 16))
    InstanceNumber = np.uint32(int("00200013", 16))
    IsotopeNumber = np.uint32(int("00200014", 16))
    PhaseNumber = np.uint32(int("00200015", 16))
    IntervalNumber = np.uint32(int("00200016", 16))
    TimeSlotNumber = np.uint32(int("00200017", 16))
    AngleNumber = np.uint32(int("00200018", 16))
    ItemNumber = np.uint32(int("00200019", 16))
    PatientOrientation = np.uint32(int("00200020", 16))
    OverlayNumber = np.uint32(int("00200022", 16))
    CurveNumber = np.uint32(int("00200024", 16))
    LUTNumber = np.uint32(int("00200026", 16))
    ImagePosition = np.uint32(int("00200030", 16))
    ImagePositionPatient = np.uint32(int("00200032", 16))
    ImageOrientation = np.uint32(int("00200035", 16))
    ImageOrientationPatient = np.uint32(int("00200037", 16))
    Location = np.uint32(int("00200050", 16))
    FrameofReferenceUID = np.uint32(int("00200052", 16))
    Laterality = np.uint32(int("00200060", 16))
    ImageLaterality = np.uint32(int("00200062", 16))
    ImageGeometryType = np.uint32(int("00200070", 16))
    MaskingImage = np.uint32(int("00200080", 16))
    ReportNumber = np.uint32(int("002000AA", 16))
    TemporalPositionIdentifier = np.uint32(int("00200100", 16))
    NumberofTemporalPositions = np.uint32(int("00200105", 16))
    TemporalResolution = np.uint32(int("00200110", 16))
    SynchronizationFrameofReferenceUID = np.uint32(int("00200200", 16))
    SOPInstanceUIDofConcatenationSource = np.uint32(int("00200242", 16))
    SeriesinStudy = np.uint32(int("00201000", 16))
    AcquisitionsinSeries = np.uint32(int("00201001", 16))
    ImagesinAcquisition = np.uint32(int("00201002", 16))
    ImagesinSeries = np.uint32(int("00201003", 16))
    AcquisitionsinStudy = np.uint32(int("00201004", 16))
    ImagesinStudy = np.uint32(int("00201005", 16))
    Reference = np.uint32(int("00201020", 16))
    PositionReferenceIndicator = np.uint32(int("00201040", 16))
    SliceLocation = np.uint32(int("00201041", 16))
    OtherStudyNumbers = np.uint32(int("00201070", 16))
    NumberofPatientRelatedStudies = np.uint32(int("00201200", 16))
    NumberofPatientRelatedSeries = np.uint32(int("00201202", 16))
    NumberofPatientRelatedInstances = np.uint32(int("00201204", 16))
    NumberofStudyRelatedSeries = np.uint32(int("00201206", 16))
    NumberofStudyRelatedInstances = np.uint32(int("00201208", 16))
    NumberofSeriesRelatedInstances = np.uint32(int("00201209", 16))
    ModifyingDeviceID = np.uint32(int("00203401", 16))
    ModifiedImageID = np.uint32(int("00203402", 16))
    ModifiedImageDate = np.uint32(int("00203403", 16))
    ModifyingDeviceManufacturer = np.uint32(int("00203404", 16))
    ModifiedImageTime = np.uint32(int("00203405", 16))
    ModifiedImageDescription = np.uint32(int("00203406", 16))
    ImageComments = np.uint32(int("00204000", 16))
    OriginalImageIdentification = np.uint32(int("00205000", 16))
    OriginalImageIdentificationNomenclature = np.uint32(int("00205002", 16))
    StackID = np.uint32(int("00209056", 16))
    InStackPositionNumber = np.uint32(int("00209057", 16))
    FrameAnatomySequence = np.uint32(int("00209071", 16))
    FrameLaterality = np.uint32(int("00209072", 16))
    FrameContentSequence = np.uint32(int("00209111", 16))
    PlanePositionSequence = np.uint32(int("00209113", 16))
    PlaneOrientationSequence = np.uint32(int("00209116", 16))
    TemporalPositionIndex = np.uint32(int("00209128", 16))
    NominalCardiacTriggerDelayTime = np.uint32(int("00209153", 16))
    NominalCardiacTriggerTimePriorToRPeak = np.uint32(int("00209154", 16))
    ActualCardiacTriggerTimePriorToRPeak = np.uint32(int("00209155", 16))
    FrameAcquisitionNumber = np.uint32(int("00209156", 16))
    DimensionIndexValues = np.uint32(int("00209157", 16))
    FrameComments = np.uint32(int("00209158", 16))
    ConcatenationUID = np.uint32(int("00209161", 16))
    InconcatenationNumber = np.uint32(int("00209162", 16))
    InconcatenationTotalNumber = np.uint32(int("00209163", 16))
    DimensionOrganizationUID = np.uint32(int("00209164", 16))
    DimensionIndexPointer = np.uint32(int("00209165", 16))
    FunctionalGroupPointer = np.uint32(int("00209167", 16))
    UnassignedSharedConvertedAttributesSequence = np.uint32(int("00209170", 16))
    UnassignedPerFrameConvertedAttributesSequence = np.uint32(int("00209171", 16))
    ConversionSourceAttributesSequence = np.uint32(int("00209172", 16))
    DimensionIndexPrivateCreator = np.uint32(int("00209213", 16))
    DimensionOrganizationSequence = np.uint32(int("00209221", 16))
    DimensionIndexSequence = np.uint32(int("00209222", 16))
    ConcatenationFrameOffsetNumber = np.uint32(int("00209228", 16))
    FunctionalGroupPrivateCreator = np.uint32(int("00209238", 16))
    NominalPercentageofCardiacPhase = np.uint32(int("00209241", 16))
    NominalPercentageofRespiratoryPhase = np.uint32(int("00209245", 16))
    StartingRespiratoryAmplitude = np.uint32(int("00209246", 16))
    StartingRespiratoryPhase = np.uint32(int("00209247", 16))
    EndingRespiratoryAmplitude = np.uint32(int("00209248", 16))
    EndingRespiratoryPhase = np.uint32(int("00209249", 16))
    RespiratoryTriggerType = np.uint32(int("00209250", 16))
    RRIntervalTimeNominal = np.uint32(int("00209251", 16))
    ActualCardiacTriggerDelayTime = np.uint32(int("00209252", 16))
    RespiratorySynchronizationSequence = np.uint32(int("00209253", 16))
    RespiratoryIntervalTime = np.uint32(int("00209254", 16))
    NominalRespiratoryTriggerDelayTime = np.uint32(int("00209255", 16))
    RespiratoryTriggerDelayThreshold = np.uint32(int("00209256", 16))
    ActualRespiratoryTriggerDelayTime = np.uint32(int("00209257", 16))
    ImagePositionVolume = np.uint32(int("00209301", 16))
    ImageOrientationVolume = np.uint32(int("00209302", 16))
    UltrasoundAcquisitionGeometry = np.uint32(int("00209307", 16))
    ApexPosition = np.uint32(int("00209308", 16))
    VolumetoTransducerMappingMatrix = np.uint32(int("00209309", 16))
    VolumetoTableMappingMatrix = np.uint32(int("0020930A", 16))
    VolumetoTransducerRelationship = np.uint32(int("0020930B", 16))
    PatientFrameofReferenceSource = np.uint32(int("0020930C", 16))
    TemporalPositionTimeOffset = np.uint32(int("0020930D", 16))
    PlanePositionVolumeSequence = np.uint32(int("0020930E", 16))
    PlaneOrientationVolumeSequence = np.uint32(int("0020930F", 16))
    TemporalPositionSequence = np.uint32(int("00209310", 16))
    DimensionOrganizationType = np.uint32(int("00209311", 16))
    VolumeFrameofReferenceUID = np.uint32(int("00209312", 16))
    TableFrameofReferenceUID = np.uint32(int("00209313", 16))
    DimensionDescriptionLabel = np.uint32(int("00209421", 16))
    PatientOrientationinFrameSequence = np.uint32(int("00209450", 16))
    FrameLabel = np.uint32(int("00209453", 16))
    AcquisitionIndex = np.uint32(int("00209518", 16))
    ContributingSOPInstancesReferenceSequence = np.uint32(int("00209529", 16))
    ReconstructionIndex = np.uint32(int("00209536", 16))
    LightPathFilterPassThroughWavelength = np.uint32(int("00220001", 16))
    LightPathFilterPassBand = np.uint32(int("00220002", 16))
    ImagePathFilterPassThroughWavelength = np.uint32(int("00220003", 16))
    ImagePathFilterPassBand = np.uint32(int("00220004", 16))
    PatientEyeMovementCommanded = np.uint32(int("00220005", 16))
    PatientEyeMovementCommandCodeSequence = np.uint32(int("00220006", 16))
    SphericalLensPower = np.uint32(int("00220007", 16))
    CylinderLensPower = np.uint32(int("00220008", 16))
    CylinderAxis = np.uint32(int("00220009", 16))
    EmmetropicMagnification = np.uint32(int("0022000A", 16))
    IntraOcularPressure = np.uint32(int("0022000B", 16))
    HorizontalFieldofView = np.uint32(int("0022000C", 16))
    PupilDilated = np.uint32(int("0022000D", 16))
    DegreeofDilation = np.uint32(int("0022000E", 16))
    StereoBaselineAngle = np.uint32(int("00220010", 16))
    StereoBaselineDisplacement = np.uint32(int("00220011", 16))
    StereoHorizontalPixelOffset = np.uint32(int("00220012", 16))
    StereoVerticalPixelOffset = np.uint32(int("00220013", 16))
    StereoRotation = np.uint32(int("00220014", 16))
    AcquisitionDeviceTypeCodeSequence = np.uint32(int("00220015", 16))
    IlluminationTypeCodeSequence = np.uint32(int("00220016", 16))
    LightPathFilterTypeStackCodeSequence = np.uint32(int("00220017", 16))
    ImagePathFilterTypeStackCodeSequence = np.uint32(int("00220018", 16))
    LensesCodeSequence = np.uint32(int("00220019", 16))
    ChannelDescriptionCodeSequence = np.uint32(int("0022001A", 16))
    RefractiveStateSequence = np.uint32(int("0022001B", 16))
    MydriaticAgentCodeSequence = np.uint32(int("0022001C", 16))
    RelativeImagePositionCodeSequence = np.uint32(int("0022001D", 16))
    CameraAngleofView = np.uint32(int("0022001E", 16))
    StereoPairsSequence = np.uint32(int("00220020", 16))
    LeftImageSequence = np.uint32(int("00220021", 16))
    RightImageSequence = np.uint32(int("00220022", 16))
    StereoPairsPresent = np.uint32(int("00220028", 16))
    AxialLengthoftheEye = np.uint32(int("00220030", 16))
    OphthalmicFrameLocationSequence = np.uint32(int("00220031", 16))
    ReferenceCoordinates = np.uint32(int("00220032", 16))
    DepthSpatialResolution = np.uint32(int("00220035", 16))
    MaximumDepthDistortion = np.uint32(int("00220036", 16))
    AlongscanSpatialResolution = np.uint32(int("00220037", 16))
    MaximumAlongscanDistortion = np.uint32(int("00220038", 16))
    OphthalmicImageOrientation = np.uint32(int("00220039", 16))
    DepthofTransverseImage = np.uint32(int("00220041", 16))
    MydriaticAgentConcentrationUnitsSequence = np.uint32(int("00220042", 16))
    AcrossscanSpatialResolution = np.uint32(int("00220048", 16))
    MaximumAcrossscanDistortion = np.uint32(int("00220049", 16))
    MydriaticAgentConcentration = np.uint32(int("0022004E", 16))
    IlluminationWaveLength = np.uint32(int("00220055", 16))
    IlluminationPower = np.uint32(int("00220056", 16))
    IlluminationBandwidth = np.uint32(int("00220057", 16))
    MydriaticAgentSequence = np.uint32(int("00220058", 16))
    OphthalmicAxialMeasurementsRightEyeSequence = np.uint32(int("00221007", 16))
    OphthalmicAxialMeasurementsLeftEyeSequence = np.uint32(int("00221008", 16))
    OphthalmicAxialMeasurementsDeviceType = np.uint32(int("00221009", 16))
    OphthalmicAxialLengthMeasurementsType = np.uint32(int("00221010", 16))
    OphthalmicAxialLengthSequence = np.uint32(int("00221012", 16))
    OphthalmicAxialLength = np.uint32(int("00221019", 16))
    LensStatusCodeSequence = np.uint32(int("00221024", 16))
    VitreousStatusCodeSequence = np.uint32(int("00221025", 16))
    IOLFormulaCodeSequence = np.uint32(int("00221028", 16))
    IOLFormulaDetail = np.uint32(int("00221029", 16))
    KeratometerIndex = np.uint32(int("00221033", 16))
    SourceofOphthalmicAxialLengthCodeSequence = np.uint32(int("00221035", 16))
    TargetRefraction = np.uint32(int("00221037", 16))
    RefractiveProcedureOccurred = np.uint32(int("00221039", 16))
    RefractiveSurgeryTypeCodeSequence = np.uint32(int("00221040", 16))
    OphthalmicUltrasoundMethodCodeSequence = np.uint32(int("00221044", 16))
    OphthalmicAxialLengthMeasurementsSequence = np.uint32(int("00221050", 16))
    IOLPower = np.uint32(int("00221053", 16))
    PredictedRefractiveError = np.uint32(int("00221054", 16))
    OphthalmicAxialLengthVelocity = np.uint32(int("00221059", 16))
    LensStatusDescription = np.uint32(int("00221065", 16))
    VitreousStatusDescription = np.uint32(int("00221066", 16))
    IOLPowerSequence = np.uint32(int("00221090", 16))
    LensConstantSequence = np.uint32(int("00221092", 16))
    IOLManufacturer = np.uint32(int("00221093", 16))
    LensConstantDescription = np.uint32(int("00221094", 16))
    ImplantName = np.uint32(int("00221095", 16))
    KeratometryMeasurementTypeCodeSequence = np.uint32(int("00221096", 16))
    ImplantPartNumber = np.uint32(int("00221097", 16))
    ReferencedOphthalmicAxialMeasurementsSequence = np.uint32(int("00221100", 16))
    OphthalmicAxialLengthMeasurementsSegmentNameCodeSequence = np.uint32(
        int("00221101", 16)
    )
    RefractiveErrorBeforeRefractiveSurgeryCodeSequence = np.uint32(int("00221103", 16))
    IOLPowerForExactEmmetropia = np.uint32(int("00221121", 16))
    IOLPowerForExactTargetRefraction = np.uint32(int("00221122", 16))
    AnteriorChamberDepthDefinitionCodeSequence = np.uint32(int("00221125", 16))
    LensThicknessSequence = np.uint32(int("00221127", 16))
    AnteriorChamberDepthSequence = np.uint32(int("00221128", 16))
    LensThickness = np.uint32(int("00221130", 16))
    AnteriorChamberDepth = np.uint32(int("00221131", 16))
    SourceofLensThicknessDataCodeSequence = np.uint32(int("00221132", 16))
    SourceofAnteriorChamberDepthDataCodeSequence = np.uint32(int("00221133", 16))
    SourceofRefractiveMeasurementsSequence = np.uint32(int("00221134", 16))
    SourceofRefractiveMeasurementsCodeSequence = np.uint32(int("00221135", 16))
    OphthalmicAxialLengthMeasurementModified = np.uint32(int("00221140", 16))
    OphthalmicAxialLengthDataSourceCodeSequence = np.uint32(int("00221150", 16))
    OphthalmicAxialLengthAcquisitionMethodCodeSequence = np.uint32(int("00221153", 16))
    SignaltoNoiseRatio = np.uint32(int("00221155", 16))
    OphthalmicAxialLengthDataSourceDescription = np.uint32(int("00221159", 16))
    OphthalmicAxialLengthMeasurementsTotalLengthSequence = np.uint32(
        int("00221210", 16)
    )
    OphthalmicAxialLengthMeasurementsSegmentalLengthSequence = np.uint32(
        int("00221211", 16)
    )
    OphthalmicAxialLengthMeasurementsLengthSummationSequence = np.uint32(
        int("00221212", 16)
    )
    UltrasoundOphthalmicAxialLengthMeasurementsSequence = np.uint32(int("00221220", 16))
    OpticalOphthalmicAxialLengthMeasurementsSequence = np.uint32(int("00221225", 16))
    UltrasoundSelectedOphthalmicAxialLengthSequence = np.uint32(int("00221230", 16))
    OphthalmicAxialLengthSelectionMethodCodeSequence = np.uint32(int("00221250", 16))
    OpticalSelectedOphthalmicAxialLengthSequence = np.uint32(int("00221255", 16))
    SelectedSegmentalOphthalmicAxialLengthSequence = np.uint32(int("00221257", 16))
    SelectedTotalOphthalmicAxialLengthSequence = np.uint32(int("00221260", 16))
    OphthalmicAxialLengthQualityMetricSequence = np.uint32(int("00221262", 16))
    OphthalmicAxialLengthQualityMetricTypeCodeSequence = np.uint32(int("00221265", 16))
    OphthalmicAxialLengthQualityMetricTypeDescription = np.uint32(int("00221273", 16))
    IntraocularLensCalculationsRightEyeSequence = np.uint32(int("00221300", 16))
    IntraocularLensCalculationsLeftEyeSequence = np.uint32(int("00221310", 16))
    ReferencedOphthalmicAxialLengthMeasurementQCImageSequence = np.uint32(
        int("00221330", 16)
    )
    OphthalmicMappingDeviceType = np.uint32(int("00221415", 16))
    AcquisitionMethodCodeSequence = np.uint32(int("00221420", 16))
    AcquisitionMethodAlgorithmSequence = np.uint32(int("00221423", 16))
    OphthalmicThicknessMapTypeCodeSequence = np.uint32(int("00221436", 16))
    OphthalmicThicknessMappingNormalsSequence = np.uint32(int("00221443", 16))
    RetinalThicknessDefinitionCodeSequence = np.uint32(int("00221445", 16))
    PixelValueMappingtoCodedConceptSequence = np.uint32(int("00221450", 16))
    MappedPixelValue = np.uint32(int("00221452", 16))
    PixelValueMappingExplanation = np.uint32(int("00221454", 16))
    OphthalmicThicknessMapQualityThresholdSequence = np.uint32(int("00221458", 16))
    OphthalmicThicknessMapThresholdQualityRating = np.uint32(int("00221460", 16))
    AnatomicStructureReferencePoint = np.uint32(int("00221463", 16))
    RegistrationtoLocalizerSequence = np.uint32(int("00221465", 16))
    RegisteredLocalizerUnits = np.uint32(int("00221466", 16))
    RegisteredLocalizerTopLeftHandCorner = np.uint32(int("00221467", 16))
    RegisteredLocalizerBottomRightHandCorner = np.uint32(int("00221468", 16))
    OphthalmicThicknessMapQualityRatingSequence = np.uint32(int("00221470", 16))
    RelevantOPTAttributesSequence = np.uint32(int("00221472", 16))
    TransformationMethodCodeSequence = np.uint32(int("00221512", 16))
    TransformationAlgorithmSequence = np.uint32(int("00221513", 16))
    OphthalmicAxialLengthMethod = np.uint32(int("00221515", 16))
    OphthalmicFOV = np.uint32(int("00221517", 16))
    TwoDimensionaltoThreeDimensionalMapSequence = np.uint32(int("00221518", 16))
    WideFieldOphthalmicPhotographyQualityRatingSequence = np.uint32(int("00221525", 16))
    WideFieldOphthalmicPhotographyQualityThresholdSequence = np.uint32(
        int("00221526", 16)
    )
    WideFieldOphthalmicPhotographyThresholdQualityRating = np.uint32(
        int("00221527", 16)
    )
    XCoordinatesCenterPixelViewAngle = np.uint32(int("00221528", 16))
    YCoordinatesCenterPixelViewAngle = np.uint32(int("00221529", 16))
    NumberofMapPoints = np.uint32(int("00221530", 16))
    TwoDimensionaltoThreeDimensionalMapData = np.uint32(int("00221531", 16))
    VisualFieldHorizontalExtent = np.uint32(int("00240010", 16))
    VisualFieldVerticalExtent = np.uint32(int("00240011", 16))
    VisualFieldShape = np.uint32(int("00240012", 16))
    ScreeningTestModeCodeSequence = np.uint32(int("00240016", 16))
    MaximumStimulusLuminance = np.uint32(int("00240018", 16))
    BackgroundLuminance = np.uint32(int("00240020", 16))
    StimulusColorCodeSequence = np.uint32(int("00240021", 16))
    BackgroundIlluminationColorCodeSequence = np.uint32(int("00240024", 16))
    StimulusArea = np.uint32(int("00240025", 16))
    StimulusPresentationTime = np.uint32(int("00240028", 16))
    FixationSequence = np.uint32(int("00240032", 16))
    FixationMonitoringCodeSequence = np.uint32(int("00240033", 16))
    VisualFieldCatchTrialSequence = np.uint32(int("00240034", 16))
    FixationCheckedQuantity = np.uint32(int("00240035", 16))
    PatientNotProperlyFixatedQuantity = np.uint32(int("00240036", 16))
    PresentedVisualStimuliDataFlag = np.uint32(int("00240037", 16))
    NumberofVisualStimuli = np.uint32(int("00240038", 16))
    ExcessiveFixationLossesDataFlag = np.uint32(int("00240039", 16))
    ExcessiveFixationLosses = np.uint32(int("00240040", 16))
    StimuliRetestingQuantity = np.uint32(int("00240042", 16))
    CommentsonPatientsPerformanceofVisualField = np.uint32(int("00240044", 16))
    FalseNegativesEstimateFlag = np.uint32(int("00240045", 16))
    FalseNegativesEstimate = np.uint32(int("00240046", 16))
    NegativeCatchTrialsQuantity = np.uint32(int("00240048", 16))
    FalseNegativesQuantity = np.uint32(int("00240050", 16))
    ExcessiveFalseNegativesDataFlag = np.uint32(int("00240051", 16))
    ExcessiveFalseNegatives = np.uint32(int("00240052", 16))
    FalsePositivesEstimateFlag = np.uint32(int("00240053", 16))
    FalsePositivesEstimate = np.uint32(int("00240054", 16))
    CatchTrialsDataFlag = np.uint32(int("00240055", 16))
    PositiveCatchTrialsQuantity = np.uint32(int("00240056", 16))
    TestPointNormalsDataFlag = np.uint32(int("00240057", 16))
    TestPointNormalsSequence = np.uint32(int("00240058", 16))
    GlobalDeviationProbabilityNormalsFlag = np.uint32(int("00240059", 16))
    FalsePositivesQuantity = np.uint32(int("00240060", 16))
    ExcessiveFalsePositivesDataFlag = np.uint32(int("00240061", 16))
    ExcessiveFalsePositives = np.uint32(int("00240062", 16))
    VisualFieldTestNormalsFlag = np.uint32(int("00240063", 16))
    ResultsNormalsSequence = np.uint32(int("00240064", 16))
    AgeCorrectedSensitivityDeviationAlgorithmSequence = np.uint32(int("00240065", 16))
    GlobalDeviationFromNormal = np.uint32(int("00240066", 16))
    GeneralizedDefectSensitivityDeviationAlgorithmSequence = np.uint32(
        int("00240067", 16)
    )
    LocalizedDeviationFromNormal = np.uint32(int("00240068", 16))
    PatientReliabilityIndicator = np.uint32(int("00240069", 16))
    VisualFieldMeanSensitivity = np.uint32(int("00240070", 16))
    GlobalDeviationProbability = np.uint32(int("00240071", 16))
    LocalDeviationProbabilityNormalsFlag = np.uint32(int("00240072", 16))
    LocalizedDeviationProbability = np.uint32(int("00240073", 16))
    ShortTermFluctuationCalculated = np.uint32(int("00240074", 16))
    ShortTermFluctuation = np.uint32(int("00240075", 16))
    ShortTermFluctuationProbabilityCalculated = np.uint32(int("00240076", 16))
    ShortTermFluctuationProbability = np.uint32(int("00240077", 16))
    CorrectedLocalizedDeviationFromNormalCalculated = np.uint32(int("00240078", 16))
    CorrectedLocalizedDeviationFromNormal = np.uint32(int("00240079", 16))
    CorrectedLocalizedDeviationFromNormalProbabilityCalculated = np.uint32(
        int("00240080", 16)
    )
    CorrectedLocalizedDeviationFromNormalProbability = np.uint32(int("00240081", 16))
    GlobalDeviationProbabilitySequence = np.uint32(int("00240083", 16))
    LocalizedDeviationProbabilitySequence = np.uint32(int("00240085", 16))
    FovealSensitivityMeasured = np.uint32(int("00240086", 16))
    FovealSensitivity = np.uint32(int("00240087", 16))
    VisualFieldTestDuration = np.uint32(int("00240088", 16))
    VisualFieldTestPointSequence = np.uint32(int("00240089", 16))
    VisualFieldTestPointXCoordinate = np.uint32(int("00240090", 16))
    VisualFieldTestPointYCoordinate = np.uint32(int("00240091", 16))
    AgeCorrectedSensitivityDeviationValue = np.uint32(int("00240092", 16))
    StimulusResults = np.uint32(int("00240093", 16))
    SensitivityValue = np.uint32(int("00240094", 16))
    RetestStimulusSeen = np.uint32(int("00240095", 16))
    RetestSensitivityValue = np.uint32(int("00240096", 16))
    VisualFieldTestPointNormalsSequence = np.uint32(int("00240097", 16))
    QuantifiedDefect = np.uint32(int("00240098", 16))
    AgeCorrectedSensitivityDeviationProbabilityValue = np.uint32(int("00240100", 16))
    GeneralizedDefectCorrectedSensitivityDeviationFlag = np.uint32(int("00240102", 16))
    GeneralizedDefectCorrectedSensitivityDeviationValue = np.uint32(int("00240103", 16))
    GeneralizedDefectCorrectedSensitivityDeviationProbabilityValue = np.uint32(
        int("00240104", 16)
    )
    MinimumSensitivityValue = np.uint32(int("00240105", 16))
    BlindSpotLocalized = np.uint32(int("00240106", 16))
    BlindSpotXCoordinate = np.uint32(int("00240107", 16))
    BlindSpotYCoordinate = np.uint32(int("00240108", 16))
    VisualAcuityMeasurementSequence = np.uint32(int("00240110", 16))
    RefractiveParametersUsedonPatientSequence = np.uint32(int("00240112", 16))
    MeasurementLaterality = np.uint32(int("00240113", 16))
    OphthalmicPatientClinicalInformationLeftEyeSequence = np.uint32(int("00240114", 16))
    OphthalmicPatientClinicalInformationRightEyeSequence = np.uint32(
        int("00240115", 16)
    )
    FovealPointNormativeDataFlag = np.uint32(int("00240117", 16))
    FovealPointProbabilityValue = np.uint32(int("00240118", 16))
    ScreeningBaselineMeasured = np.uint32(int("00240120", 16))
    ScreeningBaselineMeasuredSequence = np.uint32(int("00240122", 16))
    ScreeningBaselineType = np.uint32(int("00240124", 16))
    ScreeningBaselineValue = np.uint32(int("00240126", 16))
    AlgorithmSource = np.uint32(int("00240202", 16))
    DataSetName = np.uint32(int("00240306", 16))
    DataSetVersion = np.uint32(int("00240307", 16))
    DataSetSource = np.uint32(int("00240308", 16))
    DataSetDescription = np.uint32(int("00240309", 16))
    VisualFieldTestReliabilityGlobalIndexSequence = np.uint32(int("00240317", 16))
    VisualFieldGlobalResultsIndexSequence = np.uint32(int("00240320", 16))
    DataObservationSequence = np.uint32(int("00240325", 16))
    IndexNormalsFlag = np.uint32(int("00240338", 16))
    IndexProbability = np.uint32(int("00240341", 16))
    IndexProbabilitySequence = np.uint32(int("00240344", 16))
    SamplesperPixel = np.uint32(int("00280002", 16))
    SamplesperPixelUsed = np.uint32(int("00280003", 16))
    PhotometricInterpretation = np.uint32(int("00280004", 16))
    ImageDimensions = np.uint32(int("00280005", 16))
    PlanarConfiguration = np.uint32(int("00280006", 16))
    NumberofFrames = np.uint32(int("00280008", 16))
    FrameIncrementPointer = np.uint32(int("00280009", 16))
    FrameDimensionPointer = np.uint32(int("0028000A", 16))
    Rows = np.uint32(int("00280010", 16))
    Columns = np.uint32(int("00280011", 16))
    Planes = np.uint32(int("00280012", 16))
    UltrasoundColorDataPresent = np.uint32(int("00280014", 16))
    PixelSpacing = np.uint32(int("00280030", 16))
    ZoomFactor = np.uint32(int("00280031", 16))
    ZoomCenter = np.uint32(int("00280032", 16))
    PixelAspectRatio = np.uint32(int("00280034", 16))
    ImageFormat = np.uint32(int("00280040", 16))
    ManipulatedImage = np.uint32(int("00280050", 16))
    CorrectedImage = np.uint32(int("00280051", 16))
    CompressionRecognitionCode = np.uint32(int("0028005F", 16))
    CompressionCode = np.uint32(int("00280060", 16))
    CompressionOriginator = np.uint32(int("00280061", 16))
    CompressionLabel = np.uint32(int("00280062", 16))
    CompressionDescription = np.uint32(int("00280063", 16))
    CompressionSequence = np.uint32(int("00280065", 16))
    CompressionStepPointers = np.uint32(int("00280066", 16))
    RepeatInterval = np.uint32(int("00280068", 16))
    BitsGrouped = np.uint32(int("00280069", 16))
    PerimeterTable = np.uint32(int("00280070", 16))
    PerimeterValue = np.uint32(int("00280071", 16))
    PredictorRows = np.uint32(int("00280080", 16))
    PredictorColumns = np.uint32(int("00280081", 16))
    PredictorConstants = np.uint32(int("00280082", 16))
    BlockedPixels = np.uint32(int("00280090", 16))
    BlockRows = np.uint32(int("00280091", 16))
    BlockColumns = np.uint32(int("00280092", 16))
    RowOverlap = np.uint32(int("00280093", 16))
    ColumnOverlap = np.uint32(int("00280094", 16))
    BitsAllocated = np.uint32(int("00280100", 16))
    BitsStored = np.uint32(int("00280101", 16))
    HighBit = np.uint32(int("00280102", 16))
    PixelRepresentation = np.uint32(int("00280103", 16))
    SmallestValidPixelValue = np.uint32(int("00280104", 16))
    LargestValidPixelValue = np.uint32(int("00280105", 16))
    SmallestImagePixelValue = np.uint32(int("00280106", 16))
    LargestImagePixelValue = np.uint32(int("00280107", 16))
    SmallestPixelValueinSeries = np.uint32(int("00280108", 16))
    LargestPixelValueinSeries = np.uint32(int("00280109", 16))
    SmallestImagePixelValueinPlane = np.uint32(int("00280110", 16))
    LargestImagePixelValueinPlane = np.uint32(int("00280111", 16))
    PixelPaddingValue = np.uint32(int("00280120", 16))
    PixelPaddingRangeLimit = np.uint32(int("00280121", 16))
    FloatPixelPaddingValue = np.uint32(int("00280122", 16))
    DoubleFloatPixelPaddingValue = np.uint32(int("00280123", 16))
    FloatPixelPaddingRangeLimit = np.uint32(int("00280124", 16))
    DoubleFloatPixelPaddingRangeLimit = np.uint32(int("00280125", 16))
    ImageLocation = np.uint32(int("00280200", 16))
    QualityControlImage = np.uint32(int("00280300", 16))
    BurnedInAnnotation = np.uint32(int("00280301", 16))
    RecognizableVisualFeatures = np.uint32(int("00280302", 16))
    LongitudinalTemporalInformationModified = np.uint32(int("00280303", 16))
    ReferencedColorPaletteInstanceUID = np.uint32(int("00280304", 16))
    TransformLabel = np.uint32(int("00280400", 16))
    TransformVersionNumber = np.uint32(int("00280401", 16))
    NumberofTransformSteps = np.uint32(int("00280402", 16))
    SequenceofCompressedData = np.uint32(int("00280403", 16))
    DetailsofCoefficients = np.uint32(int("00280404", 16))
    DCTLabel = np.uint32(int("00280700", 16))
    DataBlockDescription = np.uint32(int("00280701", 16))
    DataBlock = np.uint32(int("00280702", 16))
    NormalizationFactorFormat = np.uint32(int("00280710", 16))
    ZonalMapNumberFormat = np.uint32(int("00280720", 16))
    ZonalMapLocation = np.uint32(int("00280721", 16))
    ZonalMapFormat = np.uint32(int("00280722", 16))
    AdaptiveMapFormat = np.uint32(int("00280730", 16))
    CodeNumberFormat = np.uint32(int("00280740", 16))
    PixelSpacingCalibrationType = np.uint32(int("00280A02", 16))
    PixelSpacingCalibrationDescription = np.uint32(int("00280A04", 16))
    PixelIntensityRelationship = np.uint32(int("00281040", 16))
    PixelIntensityRelationshipSign = np.uint32(int("00281041", 16))
    WindowCenter = np.uint32(int("00281050", 16))
    WindowWidth = np.uint32(int("00281051", 16))
    RescaleIntercept = np.uint32(int("00281052", 16))
    RescaleSlope = np.uint32(int("00281053", 16))
    RescaleType = np.uint32(int("00281054", 16))
    WindowCenterWidthExplanation = np.uint32(int("00281055", 16))
    VOILUTFunction = np.uint32(int("00281056", 16))
    GrayScale = np.uint32(int("00281080", 16))
    RecommendedViewingMode = np.uint32(int("00281090", 16))
    GrayLookupTableDescriptor = np.uint32(int("00281100", 16))
    RedPaletteColorLookupTableDescriptor = np.uint32(int("00281101", 16))
    GreenPaletteColorLookupTableDescriptor = np.uint32(int("00281102", 16))
    BluePaletteColorLookupTableDescriptor = np.uint32(int("00281103", 16))
    AlphaPaletteColorLookupTableDescriptor = np.uint32(int("00281104", 16))
    LargeRedPaletteColorLookupTableDescriptor = np.uint32(int("00281111", 16))
    LargeGreenPaletteColorLookupTableDescriptor = np.uint32(int("00281112", 16))
    LargeBluePaletteColorLookupTableDescriptor = np.uint32(int("00281113", 16))
    PaletteColorLookupTableUID = np.uint32(int("00281199", 16))
    GrayLookupTableData = np.uint32(int("00281200", 16))
    RedPaletteColorLookupTableData = np.uint32(int("00281201", 16))
    GreenPaletteColorLookupTableData = np.uint32(int("00281202", 16))
    BluePaletteColorLookupTableData = np.uint32(int("00281203", 16))
    AlphaPaletteColorLookupTableData = np.uint32(int("00281204", 16))
    LargeRedPaletteColorLookupTableData = np.uint32(int("00281211", 16))
    LargeGreenPaletteColorLookupTableData = np.uint32(int("00281212", 16))
    LargeBluePaletteColorLookupTableData = np.uint32(int("00281213", 16))
    LargePaletteColorLookupTableUID = np.uint32(int("00281214", 16))
    SegmentedRedPaletteColorLookupTableData = np.uint32(int("00281221", 16))
    SegmentedGreenPaletteColorLookupTableData = np.uint32(int("00281222", 16))
    SegmentedBluePaletteColorLookupTableData = np.uint32(int("00281223", 16))
    SegmentedAlphaPaletteColorLookupTableData = np.uint32(int("00281224", 16))
    BreastImplantPresent = np.uint32(int("00281300", 16))
    PartialView = np.uint32(int("00281350", 16))
    PartialViewDescription = np.uint32(int("00281351", 16))
    PartialViewCodeSequence = np.uint32(int("00281352", 16))
    SpatialLocationsPreserved = np.uint32(int("0028135A", 16))
    DataFrameAssignmentSequence = np.uint32(int("00281401", 16))
    DataPathAssignment = np.uint32(int("00281402", 16))
    BitsMappedtoColorLookupTable = np.uint32(int("00281403", 16))
    BlendingLUT1Sequence = np.uint32(int("00281404", 16))
    BlendingLUT1TransferFunction = np.uint32(int("00281405", 16))
    BlendingWeightConstant = np.uint32(int("00281406", 16))
    BlendingLookupTableDescriptor = np.uint32(int("00281407", 16))
    BlendingLookupTableData = np.uint32(int("00281408", 16))
    EnhancedPaletteColorLookupTableSequence = np.uint32(int("0028140B", 16))
    BlendingLUT2Sequence = np.uint32(int("0028140C", 16))
    BlendingLUT2TransferFunction = np.uint32(int("0028140D", 16))
    DataPathID = np.uint32(int("0028140E", 16))
    RGBLUTTransferFunction = np.uint32(int("0028140F", 16))
    AlphaLUTTransferFunction = np.uint32(int("00281410", 16))
    ICCProfile = np.uint32(int("00282000", 16))
    ColorSpace = np.uint32(int("00282002", 16))
    LossyImageCompression = np.uint32(int("00282110", 16))
    LossyImageCompressionRatio = np.uint32(int("00282112", 16))
    LossyImageCompressionMethod = np.uint32(int("00282114", 16))
    ModalityLUTSequence = np.uint32(int("00283000", 16))
    LUTDescriptor = np.uint32(int("00283002", 16))
    LUTExplanation = np.uint32(int("00283003", 16))
    ModalityLUTType = np.uint32(int("00283004", 16))
    LUTData = np.uint32(int("00283006", 16))
    VOILUTSequence = np.uint32(int("00283010", 16))
    SoftcopyVOILUTSequence = np.uint32(int("00283110", 16))
    ImagePresentationComments = np.uint32(int("00284000", 16))
    BiPlaneAcquisitionSequence = np.uint32(int("00285000", 16))
    RepresentativeFrameNumber = np.uint32(int("00286010", 16))
    FrameNumbersofInterestFOI = np.uint32(int("00286020", 16))
    FrameofInterestDescription = np.uint32(int("00286022", 16))
    FrameofInterestType = np.uint32(int("00286023", 16))
    MaskPointers = np.uint32(int("00286030", 16))
    RWavePointer = np.uint32(int("00286040", 16))
    MaskSubtractionSequence = np.uint32(int("00286100", 16))
    MaskOperation = np.uint32(int("00286101", 16))
    ApplicableFrameRange = np.uint32(int("00286102", 16))
    MaskFrameNumbers = np.uint32(int("00286110", 16))
    ContrastFrameAveraging = np.uint32(int("00286112", 16))
    MaskSubpixelShift = np.uint32(int("00286114", 16))
    TIDOffset = np.uint32(int("00286120", 16))
    MaskOperationExplanation = np.uint32(int("00286190", 16))
    EquipmentAdministratorSequence = np.uint32(int("00287000", 16))
    NumberofDisplaySubsystems = np.uint32(int("00287001", 16))
    CurrentConfigurationID = np.uint32(int("00287002", 16))
    DisplaySubsystemID = np.uint32(int("00287003", 16))
    DisplaySubsystemName = np.uint32(int("00287004", 16))
    DisplaySubsystemDescription = np.uint32(int("00287005", 16))
    SystemStatus = np.uint32(int("00287006", 16))
    SystemStatusComment = np.uint32(int("00287007", 16))
    TargetLuminanceCharacteristicsSequence = np.uint32(int("00287008", 16))
    LuminanceCharacteristicsID = np.uint32(int("00287009", 16))
    DisplaySubsystemConfigurationSequence = np.uint32(int("0028700A", 16))
    ConfigurationID = np.uint32(int("0028700B", 16))
    ConfigurationName = np.uint32(int("0028700C", 16))
    ConfigurationDescription = np.uint32(int("0028700D", 16))
    ReferencedTargetLuminanceCharacteristicsID = np.uint32(int("0028700E", 16))
    QAResultsSequence = np.uint32(int("0028700F", 16))
    DisplaySubsystemQAResultsSequence = np.uint32(int("00287010", 16))
    ConfigurationQAResultsSequence = np.uint32(int("00287011", 16))
    MeasurementEquipmentSequence = np.uint32(int("00287012", 16))
    MeasurementFunctions = np.uint32(int("00287013", 16))
    MeasurementEquipmentType = np.uint32(int("00287014", 16))
    VisualEvaluationResultSequence = np.uint32(int("00287015", 16))
    DisplayCalibrationResultSequence = np.uint32(int("00287016", 16))
    DDLValue = np.uint32(int("00287017", 16))
    CIExyWhitePoint = np.uint32(int("00287018", 16))
    DisplayFunctionType = np.uint32(int("00287019", 16))
    GammaValue = np.uint32(int("0028701A", 16))
    NumberofLuminancePoints = np.uint32(int("0028701B", 16))
    LuminanceResponseSequence = np.uint32(int("0028701C", 16))
    TargetMinimumLuminance = np.uint32(int("0028701D", 16))
    TargetMaximumLuminance = np.uint32(int("0028701E", 16))
    LuminanceValue = np.uint32(int("0028701F", 16))
    LuminanceResponseDescription = np.uint32(int("00287020", 16))
    WhitePointFlag = np.uint32(int("00287021", 16))
    DisplayDeviceTypeCodeSequence = np.uint32(int("00287022", 16))
    DisplaySubsystemSequence = np.uint32(int("00287023", 16))
    LuminanceResultSequence = np.uint32(int("00287024", 16))
    AmbientLightValueSource = np.uint32(int("00287025", 16))
    MeasuredCharacteristics = np.uint32(int("00287026", 16))
    LuminanceUniformityResultSequence = np.uint32(int("00287027", 16))
    VisualEvaluationTestSequence = np.uint32(int("00287028", 16))
    TestResult = np.uint32(int("00287029", 16))
    TestResultComment = np.uint32(int("0028702A", 16))
    TestImageValidation = np.uint32(int("0028702B", 16))
    TestPatternCodeSequence = np.uint32(int("0028702C", 16))
    MeasurementPatternCodeSequence = np.uint32(int("0028702D", 16))
    VisualEvaluationMethodCodeSequence = np.uint32(int("0028702E", 16))
    PixelDataProviderURL = np.uint32(int("00287FE0", 16))
    DataPointRows = np.uint32(int("00289001", 16))
    DataPointColumns = np.uint32(int("00289002", 16))
    SignalDomainColumns = np.uint32(int("00289003", 16))
    LargestMonochromePixelValue = np.uint32(int("00289099", 16))
    DataRepresentation = np.uint32(int("00289108", 16))
    PixelMeasuresSequence = np.uint32(int("00289110", 16))
    FrameVOILUTSequence = np.uint32(int("00289132", 16))
    PixelValueTransformationSequence = np.uint32(int("00289145", 16))
    SignalDomainRows = np.uint32(int("00289235", 16))
    DisplayFilterPercentage = np.uint32(int("00289411", 16))
    FramePixelShiftSequence = np.uint32(int("00289415", 16))
    SubtractionItemID = np.uint32(int("00289416", 16))
    PixelIntensityRelationshipLUTSequence = np.uint32(int("00289422", 16))
    FramePixelDataPropertiesSequence = np.uint32(int("00289443", 16))
    GeometricalProperties = np.uint32(int("00289444", 16))
    GeometricMaximumDistortion = np.uint32(int("00289445", 16))
    ImageProcessingApplied = np.uint32(int("00289446", 16))
    MaskSelectionMode = np.uint32(int("00289454", 16))
    LUTFunction = np.uint32(int("00289474", 16))
    MaskVisibilityPercentage = np.uint32(int("00289478", 16))
    PixelShiftSequence = np.uint32(int("00289501", 16))
    RegionPixelShiftSequence = np.uint32(int("00289502", 16))
    VerticesoftheRegion = np.uint32(int("00289503", 16))
    MultiframePresentationSequence = np.uint32(int("00289505", 16))
    PixelShiftFrameRange = np.uint32(int("00289506", 16))
    LUTFrameRange = np.uint32(int("00289507", 16))
    ImagetoEquipmentMappingMatrix = np.uint32(int("00289520", 16))
    EquipmentCoordinateSystemIdentification = np.uint32(int("00289537", 16))
    StudyStatusID = np.uint32(int("0032000A", 16))
    StudyPriorityID = np.uint32(int("0032000C", 16))
    StudyIDIssuer = np.uint32(int("00320012", 16))
    StudyVerifiedDate = np.uint32(int("00320032", 16))
    StudyVerifiedTime = np.uint32(int("00320033", 16))
    StudyReadDate = np.uint32(int("00320034", 16))
    StudyReadTime = np.uint32(int("00320035", 16))
    ScheduledStudyStartDate = np.uint32(int("00321000", 16))
    ScheduledStudyStartTime = np.uint32(int("00321001", 16))
    ScheduledStudyStopDate = np.uint32(int("00321010", 16))
    ScheduledStudyStopTime = np.uint32(int("00321011", 16))
    ScheduledStudyLocation = np.uint32(int("00321020", 16))
    ScheduledStudyLocationAETitle = np.uint32(int("00321021", 16))
    ReasonforStudy = np.uint32(int("00321030", 16))
    RequestingPhysicianIdentificationSequence = np.uint32(int("00321031", 16))
    RequestingPhysician = np.uint32(int("00321032", 16))
    RequestingService = np.uint32(int("00321033", 16))
    RequestingServiceCodeSequence = np.uint32(int("00321034", 16))
    StudyArrivalDate = np.uint32(int("00321040", 16))
    StudyArrivalTime = np.uint32(int("00321041", 16))
    StudyCompletionDate = np.uint32(int("00321050", 16))
    StudyCompletionTime = np.uint32(int("00321051", 16))
    StudyComponentStatusID = np.uint32(int("00321055", 16))
    RequestedProcedureDescription = np.uint32(int("00321060", 16))
    RequestedProcedureCodeSequence = np.uint32(int("00321064", 16))
    RequestedContrastAgent = np.uint32(int("00321070", 16))
    StudyComments = np.uint32(int("00324000", 16))
    ReferencedPatientAliasSequence = np.uint32(int("00380004", 16))
    VisitStatusID = np.uint32(int("00380008", 16))
    AdmissionID = np.uint32(int("00380010", 16))
    IssuerofAdmissionID = np.uint32(int("00380011", 16))
    IssuerofAdmissionIDSequence = np.uint32(int("00380014", 16))
    RouteofAdmissions = np.uint32(int("00380016", 16))
    ScheduledAdmissionDate = np.uint32(int("0038001A", 16))
    ScheduledAdmissionTime = np.uint32(int("0038001B", 16))
    ScheduledDischargeDate = np.uint32(int("0038001C", 16))
    ScheduledDischargeTime = np.uint32(int("0038001D", 16))
    ScheduledPatientInstitutionResidence = np.uint32(int("0038001E", 16))
    AdmittingDate = np.uint32(int("00380020", 16))
    AdmittingTime = np.uint32(int("00380021", 16))
    DischargeDate = np.uint32(int("00380030", 16))
    DischargeTime = np.uint32(int("00380032", 16))
    DischargeDiagnosisDescription = np.uint32(int("00380040", 16))
    DischargeDiagnosisCodeSequence = np.uint32(int("00380044", 16))
    SpecialNeeds = np.uint32(int("00380050", 16))
    ServiceEpisodeID = np.uint32(int("00380060", 16))
    IssuerofServiceEpisodeID = np.uint32(int("00380061", 16))
    ServiceEpisodeDescription = np.uint32(int("00380062", 16))
    IssuerofServiceEpisodeIDSequence = np.uint32(int("00380064", 16))
    PertinentDocumentsSequence = np.uint32(int("00380100", 16))
    PertinentResourcesSequence = np.uint32(int("00380101", 16))
    ResourceDescription = np.uint32(int("00380102", 16))
    CurrentPatientLocation = np.uint32(int("00380300", 16))
    PatientsInstitutionResidence = np.uint32(int("00380400", 16))
    PatientState = np.uint32(int("00380500", 16))
    PatientClinicalTrialParticipationSequence = np.uint32(int("00380502", 16))
    VisitComments = np.uint32(int("00384000", 16))
    WaveformOriginality = np.uint32(int("003A0004", 16))
    NumberofWaveformChannels = np.uint32(int("003A0005", 16))
    NumberofWaveformSamples = np.uint32(int("003A0010", 16))
    SamplingFrequency = np.uint32(int("003A001A", 16))
    MultiplexGroupLabel = np.uint32(int("003A0020", 16))
    ChannelDefinitionSequence = np.uint32(int("003A0200", 16))
    WaveformChannelNumber = np.uint32(int("003A0202", 16))
    ChannelLabel = np.uint32(int("003A0203", 16))
    ChannelStatus = np.uint32(int("003A0205", 16))
    ChannelSourceSequence = np.uint32(int("003A0208", 16))
    ChannelSourceModifiersSequence = np.uint32(int("003A0209", 16))
    SourceWaveformSequence = np.uint32(int("003A020A", 16))
    ChannelDerivationDescription = np.uint32(int("003A020C", 16))
    ChannelSensitivity = np.uint32(int("003A0210", 16))
    ChannelSensitivityUnitsSequence = np.uint32(int("003A0211", 16))
    ChannelSensitivityCorrectionFactor = np.uint32(int("003A0212", 16))
    ChannelBaseline = np.uint32(int("003A0213", 16))
    ChannelTimeSkew = np.uint32(int("003A0214", 16))
    ChannelSampleSkew = np.uint32(int("003A0215", 16))
    ChannelOffset = np.uint32(int("003A0218", 16))
    WaveformBitsStored = np.uint32(int("003A021A", 16))
    FilterLowFrequency = np.uint32(int("003A0220", 16))
    FilterHighFrequency = np.uint32(int("003A0221", 16))
    NotchFilterFrequency = np.uint32(int("003A0222", 16))
    NotchFilterBandwidth = np.uint32(int("003A0223", 16))
    WaveformDataDisplayScale = np.uint32(int("003A0230", 16))
    WaveformDisplayBackgroundCIELabValue = np.uint32(int("003A0231", 16))
    WaveformPresentationGroupSequence = np.uint32(int("003A0240", 16))
    PresentationGroupNumber = np.uint32(int("003A0241", 16))
    ChannelDisplaySequence = np.uint32(int("003A0242", 16))
    ChannelRecommendedDisplayCIELabValue = np.uint32(int("003A0244", 16))
    ChannelPosition = np.uint32(int("003A0245", 16))
    DisplayShadingFlag = np.uint32(int("003A0246", 16))
    FractionalChannelDisplayScale = np.uint32(int("003A0247", 16))
    AbsoluteChannelDisplayScale = np.uint32(int("003A0248", 16))
    MultiplexedAudioChannelsDescriptionCodeSequence = np.uint32(int("003A0300", 16))
    ChannelIdentificationCode = np.uint32(int("003A0301", 16))
    ChannelMode = np.uint32(int("003A0302", 16))
    ScheduledStationAETitle = np.uint32(int("00400001", 16))
    ScheduledProcedureStepStartDate = np.uint32(int("00400002", 16))
    ScheduledProcedureStepStartTime = np.uint32(int("00400003", 16))
    ScheduledProcedureStepEndDate = np.uint32(int("00400004", 16))
    ScheduledProcedureStepEndTime = np.uint32(int("00400005", 16))
    ScheduledPerformingPhysiciansName = np.uint32(int("00400006", 16))
    ScheduledProcedureStepDescription = np.uint32(int("00400007", 16))
    ScheduledProtocolCodeSequence = np.uint32(int("00400008", 16))
    ScheduledProcedureStepID = np.uint32(int("00400009", 16))
    StageCodeSequence = np.uint32(int("0040000A", 16))
    ScheduledPerformingPhysicianIdentificationSequence = np.uint32(int("0040000B", 16))
    ScheduledStationName = np.uint32(int("00400010", 16))
    ScheduledProcedureStepLocation = np.uint32(int("00400011", 16))
    PreMedication = np.uint32(int("00400012", 16))
    ScheduledProcedureStepStatus = np.uint32(int("00400020", 16))
    OrderPlacerIdentifierSequence = np.uint32(int("00400026", 16))
    OrderFillerIdentifierSequence = np.uint32(int("00400027", 16))
    LocalNamespaceEntityID = np.uint32(int("00400031", 16))
    UniversalEntityID = np.uint32(int("00400032", 16))
    UniversalEntityIDType = np.uint32(int("00400033", 16))
    IdentifierTypeCode = np.uint32(int("00400035", 16))
    AssigningFacilitySequence = np.uint32(int("00400036", 16))
    AssigningJurisdictionCodeSequence = np.uint32(int("00400039", 16))
    AssigningAgencyorDepartmentCodeSequence = np.uint32(int("0040003A", 16))
    ScheduledProcedureStepSequence = np.uint32(int("00400100", 16))
    ReferencedNonImageCompositeSOPInstanceSequence = np.uint32(int("00400220", 16))
    PerformedStationAETitle = np.uint32(int("00400241", 16))
    PerformedStationName = np.uint32(int("00400242", 16))
    PerformedLocation = np.uint32(int("00400243", 16))
    PerformedProcedureStepStartDate = np.uint32(int("00400244", 16))
    PerformedProcedureStepStartTime = np.uint32(int("00400245", 16))
    PerformedProcedureStepEndDate = np.uint32(int("00400250", 16))
    PerformedProcedureStepEndTime = np.uint32(int("00400251", 16))
    PerformedProcedureStepStatus = np.uint32(int("00400252", 16))
    PerformedProcedureStepID = np.uint32(int("00400253", 16))
    PerformedProcedureStepDescription = np.uint32(int("00400254", 16))
    PerformedProcedureTypeDescription = np.uint32(int("00400255", 16))
    PerformedProtocolCodeSequence = np.uint32(int("00400260", 16))
    PerformedProtocolType = np.uint32(int("00400261", 16))
    ScheduledStepAttributesSequence = np.uint32(int("00400270", 16))
    RequestAttributesSequence = np.uint32(int("00400275", 16))
    CommentsonthePerformedProcedureStep = np.uint32(int("00400280", 16))
    PerformedProcedureStepDiscontinuationReasonCodeSequence = np.uint32(
        int("00400281", 16)
    )
    QuantitySequence = np.uint32(int("00400293", 16))
    Quantity = np.uint32(int("00400294", 16))
    MeasuringUnitsSequence = np.uint32(int("00400295", 16))
    BillingItemSequence = np.uint32(int("00400296", 16))
    TotalTimeofFluoroscopy = np.uint32(int("00400300", 16))
    TotalNumberofExposures = np.uint32(int("00400301", 16))
    EntranceDose = np.uint32(int("00400302", 16))
    ExposedArea = np.uint32(int("00400303", 16))
    DistanceSourcetoEntrance = np.uint32(int("00400306", 16))
    DistanceSourcetoSupport = np.uint32(int("00400307", 16))
    ExposureDoseSequence = np.uint32(int("0040030E", 16))
    CommentsonRadiationDose = np.uint32(int("00400310", 16))
    XRayOutput = np.uint32(int("00400312", 16))
    HalfValueLayer = np.uint32(int("00400314", 16))
    OrganDose = np.uint32(int("00400316", 16))
    OrganExposed = np.uint32(int("00400318", 16))
    BillingProcedureStepSequence = np.uint32(int("00400320", 16))
    FilmConsumptionSequence = np.uint32(int("00400321", 16))
    BillingSuppliesandDevicesSequence = np.uint32(int("00400324", 16))
    ReferencedProcedureStepSequence = np.uint32(int("00400330", 16))
    PerformedSeriesSequence = np.uint32(int("00400340", 16))
    CommentsontheScheduledProcedureStep = np.uint32(int("00400400", 16))
    ProtocolContextSequence = np.uint32(int("00400440", 16))
    ContentItemModifierSequence = np.uint32(int("00400441", 16))
    ScheduledSpecimenSequence = np.uint32(int("00400500", 16))
    SpecimenAccessionNumber = np.uint32(int("0040050A", 16))
    ContainerIdentifier = np.uint32(int("00400512", 16))
    IssueroftheContainerIdentifierSequence = np.uint32(int("00400513", 16))
    AlternateContainerIdentifierSequence = np.uint32(int("00400515", 16))
    ContainerTypeCodeSequence = np.uint32(int("00400518", 16))
    ContainerDescription = np.uint32(int("0040051A", 16))
    ContainerComponentSequence = np.uint32(int("00400520", 16))
    SpecimenSequence = np.uint32(int("00400550", 16))
    SpecimenIdentifier = np.uint32(int("00400551", 16))
    SpecimenDescriptionSequenceTrial = np.uint32(int("00400552", 16))
    SpecimenDescriptionTrial = np.uint32(int("00400553", 16))
    SpecimenUID = np.uint32(int("00400554", 16))
    AcquisitionContextSequence = np.uint32(int("00400555", 16))
    AcquisitionContextDescription = np.uint32(int("00400556", 16))
    SpecimenTypeCodeSequence = np.uint32(int("0040059A", 16))
    SpecimenDescriptionSequence = np.uint32(int("00400560", 16))
    IssueroftheSpecimenIdentifierSequence = np.uint32(int("00400562", 16))
    SpecimenShortDescription = np.uint32(int("00400600", 16))
    SpecimenDetailedDescription = np.uint32(int("00400602", 16))
    SpecimenPreparationSequence = np.uint32(int("00400610", 16))
    SpecimenPreparationStepContentItemSequence = np.uint32(int("00400612", 16))
    SpecimenLocalizationContentItemSequence = np.uint32(int("00400620", 16))
    SlideIdentifier = np.uint32(int("004006FA", 16))
    ImageCenterPointCoordinatesSequence = np.uint32(int("0040071A", 16))
    XOffsetinSlideCoordinateSystem = np.uint32(int("0040072A", 16))
    YOffsetinSlideCoordinateSystem = np.uint32(int("0040073A", 16))
    ZOffsetinSlideCoordinateSystem = np.uint32(int("0040074A", 16))
    PixelSpacingSequence = np.uint32(int("004008D8", 16))
    CoordinateSystemAxisCodeSequence = np.uint32(int("004008DA", 16))
    MeasurementUnitsCodeSequence = np.uint32(int("004008EA", 16))
    VitalStainCodeSequenceTrial = np.uint32(int("004009F8", 16))
    RequestedProcedureID = np.uint32(int("00401001", 16))
    ReasonfortheRequestedProcedure = np.uint32(int("00401002", 16))
    RequestedProcedurePriority = np.uint32(int("00401003", 16))
    PatientTransportArrangements = np.uint32(int("00401004", 16))
    RequestedProcedureLocation = np.uint32(int("00401005", 16))
    PlacerOrderNumberProcedure = np.uint32(int("00401006", 16))
    FillerOrderNumberProcedure = np.uint32(int("00401007", 16))
    ConfidentialityCode = np.uint32(int("00401008", 16))
    ReportingPriority = np.uint32(int("00401009", 16))
    ReasonforRequestedProcedureCodeSequence = np.uint32(int("0040100A", 16))
    NamesofIntendedRecipientsofResults = np.uint32(int("00401010", 16))
    IntendedRecipientsofResultsIdentificationSequence = np.uint32(int("00401011", 16))
    ReasonForPerformedProcedureCodeSequence = np.uint32(int("00401012", 16))
    RequestedProcedureDescriptionTrial = np.uint32(int("00401060", 16))
    PersonIdentificationCodeSequence = np.uint32(int("00401101", 16))
    PersonsAddress = np.uint32(int("00401102", 16))
    PersonsTelephoneNumbers = np.uint32(int("00401103", 16))
    PersonsTelecomInformation = np.uint32(int("00401104", 16))
    RequestedProcedureComments = np.uint32(int("00401400", 16))
    ReasonfortheImagingServiceRequest = np.uint32(int("00402001", 16))
    IssueDateofImagingServiceRequest = np.uint32(int("00402004", 16))
    IssueTimeofImagingServiceRequest = np.uint32(int("00402005", 16))
    PlacerOrderNumberImagingServiceRequestRetired = np.uint32(int("00402006", 16))
    FillerOrderNumberImagingServiceRequestRetired = np.uint32(int("00402007", 16))
    OrderEnteredBy = np.uint32(int("00402008", 16))
    OrderEnterersLocation = np.uint32(int("00402009", 16))
    OrderCallbackPhoneNumber = np.uint32(int("00402010", 16))
    OrderCallbackTelecomInformation = np.uint32(int("00402011", 16))
    PlacerOrderNumberImagingServiceRequest = np.uint32(int("00402016", 16))
    FillerOrderNumberImagingServiceRequest = np.uint32(int("00402017", 16))
    ImagingServiceRequestComments = np.uint32(int("00402400", 16))
    ConfidentialityConstraintonPatientDataDescription = np.uint32(int("00403001", 16))
    GeneralPurposeScheduledProcedureStepStatus = np.uint32(int("00404001", 16))
    GeneralPurposePerformedProcedureStepStatus = np.uint32(int("00404002", 16))
    GeneralPurposeScheduledProcedureStepPriority = np.uint32(int("00404003", 16))
    ScheduledProcessingApplicationsCodeSequence = np.uint32(int("00404004", 16))
    ScheduledProcedureStepStartDateTime = np.uint32(int("00404005", 16))
    MultipleCopiesFlag = np.uint32(int("00404006", 16))
    PerformedProcessingApplicationsCodeSequence = np.uint32(int("00404007", 16))
    HumanPerformerCodeSequence = np.uint32(int("00404009", 16))
    ScheduledProcedureStepModificationDateTime = np.uint32(int("00404010", 16))
    ExpectedCompletionDateTime = np.uint32(int("00404011", 16))
    ResultingGeneralPurposePerformedProcedureStepsSequence = np.uint32(
        int("00404015", 16)
    )
    ReferencedGeneralPurposeScheduledProcedureStepSequence = np.uint32(
        int("00404016", 16)
    )
    ScheduledWorkitemCodeSequence = np.uint32(int("00404018", 16))
    PerformedWorkitemCodeSequence = np.uint32(int("00404019", 16))
    InputAvailabilityFlag = np.uint32(int("00404020", 16))
    InputInformationSequence = np.uint32(int("00404021", 16))
    RelevantInformationSequence = np.uint32(int("00404022", 16))
    ReferencedGeneralPurposeScheduledProcedureStepTransactionUID = np.uint32(
        int("00404023", 16)
    )
    ScheduledStationNameCodeSequence = np.uint32(int("00404025", 16))
    ScheduledStationClassCodeSequence = np.uint32(int("00404026", 16))
    ScheduledStationGeographicLocationCodeSequence = np.uint32(int("00404027", 16))
    PerformedStationNameCodeSequence = np.uint32(int("00404028", 16))
    PerformedStationClassCodeSequence = np.uint32(int("00404029", 16))
    PerformedStationGeographicLocationCodeSequence = np.uint32(int("00404030", 16))
    RequestedSubsequentWorkitemCodeSequence = np.uint32(int("00404031", 16))
    NonDICOMOutputCodeSequence = np.uint32(int("00404032", 16))
    OutputInformationSequence = np.uint32(int("00404033", 16))
    ScheduledHumanPerformersSequence = np.uint32(int("00404034", 16))
    ActualHumanPerformersSequence = np.uint32(int("00404035", 16))
    HumanPerformersOrganization = np.uint32(int("00404036", 16))
    HumanPerformersName = np.uint32(int("00404037", 16))
    RawDataHandling = np.uint32(int("00404040", 16))
    InputReadinessState = np.uint32(int("00404041", 16))
    PerformedProcedureStepStartDateTime = np.uint32(int("00404050", 16))
    PerformedProcedureStepEndDateTime = np.uint32(int("00404051", 16))
    ProcedureStepCancellationDateTime = np.uint32(int("00404052", 16))
    OutputDestinationSequence = np.uint32(int("00404070", 16))
    DICOMStorageSequence = np.uint32(int("00404071", 16))
    STOWRSStorageSequence = np.uint32(int("00404072", 16))
    StorageURL = np.uint32(int("00404073", 16))
    XDSStorageSequence = np.uint32(int("00404074", 16))
    EntranceDoseinmGy = np.uint32(int("00408302", 16))
    ParametricMapFrameTypeSequence = np.uint32(int("00409092", 16))
    ReferencedImageRealWorldValueMappingSequence = np.uint32(int("00409094", 16))
    RealWorldValueMappingSequence = np.uint32(int("00409096", 16))
    PixelValueMappingCodeSequence = np.uint32(int("00409098", 16))
    LUTLabel = np.uint32(int("00409210", 16))
    RealWorldValueLastValueMapped = np.uint32(int("00409211", 16))
    RealWorldValueLUTData = np.uint32(int("00409212", 16))
    DoubleFloatRealWorldValueLastValueMapped = np.uint32(int("00409213", 16))
    DoubleFloatRealWorldValueFirstValueMapped = np.uint32(int("00409214", 16))
    RealWorldValueFirstValueMapped = np.uint32(int("00409216", 16))
    QuantityDefinitionSequence = np.uint32(int("00409220", 16))
    RealWorldValueIntercept = np.uint32(int("00409224", 16))
    RealWorldValueSlope = np.uint32(int("00409225", 16))
    FindingsFlagTrial = np.uint32(int("0040A007", 16))
    RelationshipType = np.uint32(int("0040A010", 16))
    FindingsSequenceTrial = np.uint32(int("0040A020", 16))
    FindingsGroupUIDTrial = np.uint32(int("0040A021", 16))
    ReferencedFindingsGroupUIDTrial = np.uint32(int("0040A022", 16))
    FindingsGroupRecordingDateTrial = np.uint32(int("0040A023", 16))
    FindingsGroupRecordingTimeTrial = np.uint32(int("0040A024", 16))
    FindingsSourceCategoryCodeSequenceTrial = np.uint32(int("0040A026", 16))
    VerifyingOrganization = np.uint32(int("0040A027", 16))
    DocumentingOrganizationIdentifierCodeSequenceTrial = np.uint32(int("0040A028", 16))
    VerificationDateTime = np.uint32(int("0040A030", 16))
    ObservationDateTime = np.uint32(int("0040A032", 16))
    ValueType = np.uint32(int("0040A040", 16))
    ConceptNameCodeSequence = np.uint32(int("0040A043", 16))
    MeasurementPrecisionDescriptionTrial = np.uint32(int("0040A047", 16))
    ContinuityOfContent = np.uint32(int("0040A050", 16))
    UrgencyorPriorityAlertsTrial = np.uint32(int("0040A057", 16))
    SequencingIndicatorTrial = np.uint32(int("0040A060", 16))
    DocumentIdentifierCodeSequenceTrial = np.uint32(int("0040A066", 16))
    DocumentAuthorTrial = np.uint32(int("0040A067", 16))
    DocumentAuthorIdentifierCodeSequenceTrial = np.uint32(int("0040A068", 16))
    IdentifierCodeSequenceTrial = np.uint32(int("0040A070", 16))
    VerifyingObserverSequence = np.uint32(int("0040A073", 16))
    ObjectBinaryIdentifierTrial = np.uint32(int("0040A074", 16))
    VerifyingObserverName = np.uint32(int("0040A075", 16))
    DocumentingObserverIdentifierCodeSequenceTrial = np.uint32(int("0040A076", 16))
    AuthorObserverSequence = np.uint32(int("0040A078", 16))
    ParticipantSequence = np.uint32(int("0040A07A", 16))
    CustodialOrganizationSequence = np.uint32(int("0040A07C", 16))
    ParticipationType = np.uint32(int("0040A080", 16))
    ParticipationDateTime = np.uint32(int("0040A082", 16))
    ObserverType = np.uint32(int("0040A084", 16))
    ProcedureIdentifierCodeSequenceTrial = np.uint32(int("0040A085", 16))
    VerifyingObserverIdentificationCodeSequence = np.uint32(int("0040A088", 16))
    ObjectDirectoryBinaryIdentifierTrial = np.uint32(int("0040A089", 16))
    EquivalentCDADocumentSequence = np.uint32(int("0040A090", 16))
    ReferencedWaveformChannels = np.uint32(int("0040A0B0", 16))
    DateofDocumentorVerbalTransactionTrial = np.uint32(int("0040A110", 16))
    TimeofDocumentCreationorVerbalTransactionTrial = np.uint32(int("0040A112", 16))
    DateTime = np.uint32(int("0040A120", 16))
    Date = np.uint32(int("0040A121", 16))
    Time = np.uint32(int("0040A122", 16))
    PersonName = np.uint32(int("0040A123", 16))
    UID = np.uint32(int("0040A124", 16))
    ReportStatusIDTrial = np.uint32(int("0040A125", 16))
    TemporalRangeType = np.uint32(int("0040A130", 16))
    ReferencedSamplePositions = np.uint32(int("0040A132", 16))
    ReferencedFrameNumbers = np.uint32(int("0040A136", 16))
    ReferencedTimeOffsets = np.uint32(int("0040A138", 16))
    ReferencedDateTime = np.uint32(int("0040A13A", 16))
    TextValue = np.uint32(int("0040A160", 16))
    FloatingPointValue = np.uint32(int("0040A161", 16))
    RationalNumeratorValue = np.uint32(int("0040A162", 16))
    RationalDenominatorValue = np.uint32(int("0040A163", 16))
    ObservationCategoryCodeSequenceTrial = np.uint32(int("0040A167", 16))
    ConceptCodeSequence = np.uint32(int("0040A168", 16))
    BibliographicCitationTrial = np.uint32(int("0040A16A", 16))
    PurposeofReferenceCodeSequence = np.uint32(int("0040A170", 16))
    ObservationUID = np.uint32(int("0040A171", 16))
    ReferencedObservationUIDTrial = np.uint32(int("0040A172", 16))
    ReferencedObservationClassTrial = np.uint32(int("0040A173", 16))
    ReferencedObjectObservationClassTrial = np.uint32(int("0040A174", 16))
    AnnotationGroupNumber = np.uint32(int("0040A180", 16))
    ObservationDateTrial = np.uint32(int("0040A192", 16))
    ObservationTimeTrial = np.uint32(int("0040A193", 16))
    MeasurementAutomationTrial = np.uint32(int("0040A194", 16))
    ModifierCodeSequence = np.uint32(int("0040A195", 16))
    IdentificationDescriptionTrial = np.uint32(int("0040A224", 16))
    CoordinatesSetGeometricTypeTrial = np.uint32(int("0040A290", 16))
    AlgorithmCodeSequenceTrial = np.uint32(int("0040A296", 16))
    AlgorithmDescriptionTrial = np.uint32(int("0040A297", 16))
    PixelCoordinatesSetTrial = np.uint32(int("0040A29A", 16))
    MeasuredValueSequence = np.uint32(int("0040A300", 16))
    NumericValueQualifierCodeSequence = np.uint32(int("0040A301", 16))
    CurrentObserverTrial = np.uint32(int("0040A307", 16))
    NumericValue = np.uint32(int("0040A30A", 16))
    ReferencedAccessionSequenceTrial = np.uint32(int("0040A313", 16))
    ReportStatusCommentTrial = np.uint32(int("0040A33A", 16))
    ProcedureContextSequenceTrial = np.uint32(int("0040A340", 16))
    VerbalSourceTrial = np.uint32(int("0040A352", 16))
    AddressTrial = np.uint32(int("0040A353", 16))
    TelephoneNumberTrial = np.uint32(int("0040A354", 16))
    VerbalSourceIdentifierCodeSequenceTrial = np.uint32(int("0040A358", 16))
    PredecessorDocumentsSequence = np.uint32(int("0040A360", 16))
    ReferencedRequestSequence = np.uint32(int("0040A370", 16))
    PerformedProcedureCodeSequence = np.uint32(int("0040A372", 16))
    CurrentRequestedProcedureEvidenceSequence = np.uint32(int("0040A375", 16))
    ReportDetailSequenceTrial = np.uint32(int("0040A380", 16))
    PertinentOtherEvidenceSequence = np.uint32(int("0040A385", 16))
    HL7StructuredDocumentReferenceSequence = np.uint32(int("0040A390", 16))
    ObservationSubjectUIDTrial = np.uint32(int("0040A402", 16))
    ObservationSubjectClassTrial = np.uint32(int("0040A403", 16))
    ObservationSubjectTypeCodeSequenceTrial = np.uint32(int("0040A404", 16))
    CompletionFlag = np.uint32(int("0040A491", 16))
    CompletionFlagDescription = np.uint32(int("0040A492", 16))
    VerificationFlag = np.uint32(int("0040A493", 16))
    ArchiveRequested = np.uint32(int("0040A494", 16))
    PreliminaryFlag = np.uint32(int("0040A496", 16))
    ContentTemplateSequence = np.uint32(int("0040A504", 16))
    IdenticalDocumentsSequence = np.uint32(int("0040A525", 16))
    ObservationSubjectContextFlagTrial = np.uint32(int("0040A600", 16))
    ObserverContextFlagTrial = np.uint32(int("0040A601", 16))
    ProcedureContextFlagTrial = np.uint32(int("0040A603", 16))
    ContentSequence = np.uint32(int("0040A730", 16))
    RelationshipSequenceTrial = np.uint32(int("0040A731", 16))
    RelationshipTypeCodeSequenceTrial = np.uint32(int("0040A732", 16))
    LanguageCodeSequenceTrial = np.uint32(int("0040A744", 16))
    UniformResourceLocatorTrial = np.uint32(int("0040A992", 16))
    WaveformAnnotationSequence = np.uint32(int("0040B020", 16))
    TemplateIdentifier = np.uint32(int("0040DB00", 16))
    TemplateVersion = np.uint32(int("0040DB06", 16))
    TemplateLocalVersion = np.uint32(int("0040DB07", 16))
    TemplateExtensionFlag = np.uint32(int("0040DB0B", 16))
    TemplateExtensionOrganizationUID = np.uint32(int("0040DB0C", 16))
    TemplateExtensionCreatorUID = np.uint32(int("0040DB0D", 16))
    ReferencedContentItemIdentifier = np.uint32(int("0040DB73", 16))
    HL7InstanceIdentifier = np.uint32(int("0040E001", 16))
    HL7DocumentEffectiveTime = np.uint32(int("0040E004", 16))
    HL7DocumentTypeCodeSequence = np.uint32(int("0040E006", 16))
    DocumentClassCodeSequence = np.uint32(int("0040E008", 16))
    RetrieveURI = np.uint32(int("0040E010", 16))
    RetrieveLocationUID = np.uint32(int("0040E011", 16))
    TypeofInstances = np.uint32(int("0040E020", 16))
    DICOMRetrievalSequence = np.uint32(int("0040E021", 16))
    DICOMMediaRetrievalSequence = np.uint32(int("0040E022", 16))
    WADORetrievalSequence = np.uint32(int("0040E023", 16))
    XDSRetrievalSequence = np.uint32(int("0040E024", 16))
    WADORSRetrievalSequence = np.uint32(int("0040E025", 16))
    RepositoryUniqueID = np.uint32(int("0040E030", 16))
    HomeCommunityID = np.uint32(int("0040E031", 16))
    DocumentTitle = np.uint32(int("00420010", 16))
    EncapsulatedDocument = np.uint32(int("00420011", 16))
    MIMETypeofEncapsulatedDocument = np.uint32(int("00420012", 16))
    SourceInstanceSequence = np.uint32(int("00420013", 16))
    ListofMIMETypes = np.uint32(int("00420014", 16))
    ProductPackageIdentifier = np.uint32(int("00440001", 16))
    SubstanceAdministrationApproval = np.uint32(int("00440002", 16))
    ApprovalStatusFurtherDescription = np.uint32(int("00440003", 16))
    ApprovalStatusDateTime = np.uint32(int("00440004", 16))
    ProductTypeCodeSequence = np.uint32(int("00440007", 16))
    ProductName = np.uint32(int("00440008", 16))
    ProductDescription = np.uint32(int("00440009", 16))
    ProductLotIdentifier = np.uint32(int("0044000A", 16))
    ProductExpirationDateTime = np.uint32(int("0044000B", 16))
    SubstanceAdministrationDateTime = np.uint32(int("00440010", 16))
    SubstanceAdministrationNotes = np.uint32(int("00440011", 16))
    SubstanceAdministrationDeviceID = np.uint32(int("00440012", 16))
    ProductParameterSequence = np.uint32(int("00440013", 16))
    SubstanceAdministrationParameterSequence = np.uint32(int("00440019", 16))
    LensDescription = np.uint32(int("00460012", 16))
    RightLensSequence = np.uint32(int("00460014", 16))
    LeftLensSequence = np.uint32(int("00460015", 16))
    UnspecifiedLateralityLensSequence = np.uint32(int("00460016", 16))
    CylinderSequence = np.uint32(int("00460018", 16))
    PrismSequence = np.uint32(int("00460028", 16))
    HorizontalPrismPower = np.uint32(int("00460030", 16))
    HorizontalPrismBase = np.uint32(int("00460032", 16))
    VerticalPrismPower = np.uint32(int("00460034", 16))
    VerticalPrismBase = np.uint32(int("00460036", 16))
    LensSegmentType = np.uint32(int("00460038", 16))
    OpticalTransmittance = np.uint32(int("00460040", 16))
    ChannelWidth = np.uint32(int("00460042", 16))
    PupilSize = np.uint32(int("00460044", 16))
    CornealSize = np.uint32(int("00460046", 16))
    AutorefractionRightEyeSequence = np.uint32(int("00460050", 16))
    AutorefractionLeftEyeSequence = np.uint32(int("00460052", 16))
    DistancePupillaryDistance = np.uint32(int("00460060", 16))
    NearPupillaryDistance = np.uint32(int("00460062", 16))
    IntermediatePupillaryDistance = np.uint32(int("00460063", 16))
    OtherPupillaryDistance = np.uint32(int("00460064", 16))
    KeratometryRightEyeSequence = np.uint32(int("00460070", 16))
    KeratometryLeftEyeSequence = np.uint32(int("00460071", 16))
    SteepKeratometricAxisSequence = np.uint32(int("00460074", 16))
    RadiusofCurvature = np.uint32(int("00460075", 16))
    KeratometricPower = np.uint32(int("00460076", 16))
    KeratometricAxis = np.uint32(int("00460077", 16))
    FlatKeratometricAxisSequence = np.uint32(int("00460080", 16))
    BackgroundColor = np.uint32(int("00460092", 16))
    Optotype = np.uint32(int("00460094", 16))
    OptotypePresentation = np.uint32(int("00460095", 16))
    SubjectiveRefractionRightEyeSequence = np.uint32(int("00460097", 16))
    SubjectiveRefractionLeftEyeSequence = np.uint32(int("00460098", 16))
    AddNearSequence = np.uint32(int("00460100", 16))
    AddIntermediateSequence = np.uint32(int("00460101", 16))
    AddOtherSequence = np.uint32(int("00460102", 16))
    AddPower = np.uint32(int("00460104", 16))
    ViewingDistance = np.uint32(int("00460106", 16))
    VisualAcuityTypeCodeSequence = np.uint32(int("00460121", 16))
    VisualAcuityRightEyeSequence = np.uint32(int("00460122", 16))
    VisualAcuityLeftEyeSequence = np.uint32(int("00460123", 16))
    VisualAcuityBothEyesOpenSequence = np.uint32(int("00460124", 16))
    ViewingDistanceType = np.uint32(int("00460125", 16))
    VisualAcuityModifiers = np.uint32(int("00460135", 16))
    DecimalVisualAcuity = np.uint32(int("00460137", 16))
    OptotypeDetailedDefinition = np.uint32(int("00460139", 16))
    ReferencedRefractiveMeasurementsSequence = np.uint32(int("00460145", 16))
    SpherePower = np.uint32(int("00460146", 16))
    CylinderPower = np.uint32(int("00460147", 16))
    CornealTopographySurface = np.uint32(int("00460201", 16))
    CornealVertexLocation = np.uint32(int("00460202", 16))
    PupilCentroidXCoordinate = np.uint32(int("00460203", 16))
    PupilCentroidYCoordinate = np.uint32(int("00460204", 16))
    EquivalentPupilRadius = np.uint32(int("00460205", 16))
    CornealTopographyMapTypeCodeSequence = np.uint32(int("00460207", 16))
    VerticesoftheOutlineofPupil = np.uint32(int("00460208", 16))
    CornealTopographyMappingNormalsSequence = np.uint32(int("00460210", 16))
    MaximumCornealCurvatureSequence = np.uint32(int("00460211", 16))
    MaximumCornealCurvature = np.uint32(int("00460212", 16))
    MaximumCornealCurvatureLocation = np.uint32(int("00460213", 16))
    MinimumKeratometricSequence = np.uint32(int("00460215", 16))
    SimulatedKeratometricCylinderSequence = np.uint32(int("00460218", 16))
    AverageCornealPower = np.uint32(int("00460220", 16))
    CornealISValue = np.uint32(int("00460224", 16))
    AnalyzedArea = np.uint32(int("00460227", 16))
    SurfaceRegularityIndex = np.uint32(int("00460230", 16))
    SurfaceAsymmetryIndex = np.uint32(int("00460232", 16))
    CornealEccentricityIndex = np.uint32(int("00460234", 16))
    KeratoconusPredictionIndex = np.uint32(int("00460236", 16))
    DecimalPotentialVisualAcuity = np.uint32(int("00460238", 16))
    CornealTopographyMapQualityEvaluation = np.uint32(int("00460242", 16))
    SourceImageCornealProcessedDataSequence = np.uint32(int("00460244", 16))
    CornealPointLocation = np.uint32(int("00460247", 16))
    CornealPointEstimated = np.uint32(int("00460248", 16))
    AxialPower = np.uint32(int("00460249", 16))
    TangentialPower = np.uint32(int("00460250", 16))
    RefractivePower = np.uint32(int("00460251", 16))
    RelativeElevation = np.uint32(int("00460252", 16))
    CornealWavefront = np.uint32(int("00460253", 16))
    ImagedVolumeWidth = np.uint32(int("00480001", 16))
    ImagedVolumeHeight = np.uint32(int("00480002", 16))
    ImagedVolumeDepth = np.uint32(int("00480003", 16))
    TotalPixelMatrixColumns = np.uint32(int("00480006", 16))
    TotalPixelMatrixRows = np.uint32(int("00480007", 16))
    TotalPixelMatrixOriginSequence = np.uint32(int("00480008", 16))
    SpecimenLabelinImage = np.uint32(int("00480010", 16))
    FocusMethod = np.uint32(int("00480011", 16))
    ExtendedDepthofField = np.uint32(int("00480012", 16))
    NumberofFocalPlanes = np.uint32(int("00480013", 16))
    DistanceBetweenFocalPlanes = np.uint32(int("00480014", 16))
    RecommendedAbsentPixelCIELabValue = np.uint32(int("00480015", 16))
    IlluminatorTypeCodeSequence = np.uint32(int("00480100", 16))
    ImageOrientationSlide = np.uint32(int("00480102", 16))
    OpticalPathSequence = np.uint32(int("00480105", 16))
    OpticalPathIdentifier = np.uint32(int("00480106", 16))
    OpticalPathDescription = np.uint32(int("00480107", 16))
    IlluminationColorCodeSequence = np.uint32(int("00480108", 16))
    SpecimenReferenceSequence = np.uint32(int("00480110", 16))
    CondenserLensPower = np.uint32(int("00480111", 16))
    ObjectiveLensPower = np.uint32(int("00480112", 16))
    ObjectiveLensNumericalAperture = np.uint32(int("00480113", 16))
    PaletteColorLookupTableSequence = np.uint32(int("00480120", 16))
    ReferencedImageNavigationSequence = np.uint32(int("00480200", 16))
    TopLeftHandCornerofLocalizerArea = np.uint32(int("00480201", 16))
    BottomRightHandCornerofLocalizerArea = np.uint32(int("00480202", 16))
    OpticalPathIdentificationSequence = np.uint32(int("00480207", 16))
    PlanePositionSlideSequence = np.uint32(int("0048021A", 16))
    ColumnPositionInTotalImagePixelMatrix = np.uint32(int("0048021E", 16))
    RowPositionInTotalImagePixelMatrix = np.uint32(int("0048021F", 16))
    PixelOriginInterpretation = np.uint32(int("00480301", 16))
    CalibrationImage = np.uint32(int("00500004", 16))
    DeviceSequence = np.uint32(int("00500010", 16))
    ContainerComponentTypeCodeSequence = np.uint32(int("00500012", 16))
    ContainerComponentThickness = np.uint32(int("00500013", 16))
    DeviceLength = np.uint32(int("00500014", 16))
    ContainerComponentWidth = np.uint32(int("00500015", 16))
    DeviceDiameter = np.uint32(int("00500016", 16))
    DeviceDiameterUnits = np.uint32(int("00500017", 16))
    DeviceVolume = np.uint32(int("00500018", 16))
    InterMarkerDistance = np.uint32(int("00500019", 16))
    ContainerComponentMaterial = np.uint32(int("0050001A", 16))
    ContainerComponentID = np.uint32(int("0050001B", 16))
    ContainerComponentLength = np.uint32(int("0050001C", 16))
    ContainerComponentDiameter = np.uint32(int("0050001D", 16))
    ContainerComponentDescription = np.uint32(int("0050001E", 16))
    DeviceDescription = np.uint32(int("00500020", 16))
    ContrastBolusIngredientPercentbyVolume = np.uint32(int("00520001", 16))
    OCTFocalDistance = np.uint32(int("00520002", 16))
    BeamSpotSize = np.uint32(int("00520003", 16))
    EffectiveRefractiveIndex = np.uint32(int("00520004", 16))
    OCTAcquisitionDomain = np.uint32(int("00520006", 16))
    OCTOpticalCenterWavelength = np.uint32(int("00520007", 16))
    AxialResolution = np.uint32(int("00520008", 16))
    RangingDepth = np.uint32(int("00520009", 16))
    AlineRate = np.uint32(int("00520011", 16))
    AlinesPerFrame = np.uint32(int("00520012", 16))
    CatheterRotationalRate = np.uint32(int("00520013", 16))
    AlinePixelSpacing = np.uint32(int("00520014", 16))
    ModeofPercutaneousAccessSequence = np.uint32(int("00520016", 16))
    IntravascularOCTFrameTypeSequence = np.uint32(int("00520025", 16))
    OCTZOffsetApplied = np.uint32(int("00520026", 16))
    IntravascularFrameContentSequence = np.uint32(int("00520027", 16))
    IntravascularLongitudinalDistance = np.uint32(int("00520028", 16))
    IntravascularOCTFrameContentSequence = np.uint32(int("00520029", 16))
    OCTZOffsetCorrection = np.uint32(int("00520030", 16))
    CatheterDirectionofRotation = np.uint32(int("00520031", 16))
    SeamLineLocation = np.uint32(int("00520033", 16))
    FirstAlineLocation = np.uint32(int("00520034", 16))
    SeamLineIndex = np.uint32(int("00520036", 16))
    NumberofPaddedAlines = np.uint32(int("00520038", 16))
    InterpolationType = np.uint32(int("00520039", 16))
    RefractiveIndexApplied = np.uint32(int("0052003A", 16))
    EnergyWindowVector = np.uint32(int("00540010", 16))
    NumberofEnergyWindows = np.uint32(int("00540011", 16))
    EnergyWindowInformationSequence = np.uint32(int("00540012", 16))
    EnergyWindowRangeSequence = np.uint32(int("00540013", 16))
    EnergyWindowLowerLimit = np.uint32(int("00540014", 16))
    EnergyWindowUpperLimit = np.uint32(int("00540015", 16))
    RadiopharmaceuticalInformationSequence = np.uint32(int("00540016", 16))
    ResidualSyringeCounts = np.uint32(int("00540017", 16))
    EnergyWindowName = np.uint32(int("00540018", 16))
    DetectorVector = np.uint32(int("00540020", 16))
    NumberofDetectors = np.uint32(int("00540021", 16))
    DetectorInformationSequence = np.uint32(int("00540022", 16))
    PhaseVector = np.uint32(int("00540030", 16))
    NumberofPhases = np.uint32(int("00540031", 16))
    PhaseInformationSequence = np.uint32(int("00540032", 16))
    NumberofFramesinPhase = np.uint32(int("00540033", 16))
    PhaseDelay = np.uint32(int("00540036", 16))
    PauseBetweenFrames = np.uint32(int("00540038", 16))
    PhaseDescription = np.uint32(int("00540039", 16))
    RotationVector = np.uint32(int("00540050", 16))
    NumberofRotations = np.uint32(int("00540051", 16))
    RotationInformationSequence = np.uint32(int("00540052", 16))
    NumberofFramesinRotation = np.uint32(int("00540053", 16))
    RRIntervalVector = np.uint32(int("00540060", 16))
    NumberofRRIntervals = np.uint32(int("00540061", 16))
    GatedInformationSequence = np.uint32(int("00540062", 16))
    DataInformationSequence = np.uint32(int("00540063", 16))
    TimeSlotVector = np.uint32(int("00540070", 16))
    NumberofTimeSlots = np.uint32(int("00540071", 16))
    TimeSlotInformationSequence = np.uint32(int("00540072", 16))
    TimeSlotTime = np.uint32(int("00540073", 16))
    SliceVector = np.uint32(int("00540080", 16))
    NumberofSlices = np.uint32(int("00540081", 16))
    AngularViewVector = np.uint32(int("00540090", 16))
    TimeSliceVector = np.uint32(int("00540100", 16))
    NumberofTimeSlices = np.uint32(int("00540101", 16))
    StartAngle = np.uint32(int("00540200", 16))
    TypeofDetectorMotion = np.uint32(int("00540202", 16))
    TriggerVector = np.uint32(int("00540210", 16))
    NumberofTriggersinPhase = np.uint32(int("00540211", 16))
    ViewCodeSequence = np.uint32(int("00540220", 16))
    ViewModifierCodeSequence = np.uint32(int("00540222", 16))
    RadionuclideCodeSequence = np.uint32(int("00540300", 16))
    AdministrationRouteCodeSequence = np.uint32(int("00540302", 16))
    RadiopharmaceuticalCodeSequence = np.uint32(int("00540304", 16))
    CalibrationDataSequence = np.uint32(int("00540306", 16))
    EnergyWindowNumber = np.uint32(int("00540308", 16))
    ImageID = np.uint32(int("00540400", 16))
    PatientOrientationCodeSequence = np.uint32(int("00540410", 16))
    PatientOrientationModifierCodeSequence = np.uint32(int("00540412", 16))
    PatientGantryRelationshipCodeSequence = np.uint32(int("00540414", 16))
    SliceProgressionDirection = np.uint32(int("00540500", 16))
    ScanProgressionDirection = np.uint32(int("00540501", 16))
    SeriesType = np.uint32(int("00541000", 16))
    Units = np.uint32(int("00541001", 16))
    CountsSource = np.uint32(int("00541002", 16))
    ReprojectionMethod = np.uint32(int("00541004", 16))
    SUVType = np.uint32(int("00541006", 16))
    RandomsCorrectionMethod = np.uint32(int("00541100", 16))
    AttenuationCorrectionMethod = np.uint32(int("00541101", 16))
    DecayCorrection = np.uint32(int("00541102", 16))
    ReconstructionMethod = np.uint32(int("00541103", 16))
    DetectorLinesofResponseUsed = np.uint32(int("00541104", 16))
    ScatterCorrectionMethod = np.uint32(int("00541105", 16))
    AxialAcceptance = np.uint32(int("00541200", 16))
    AxialMash = np.uint32(int("00541201", 16))
    TransverseMash = np.uint32(int("00541202", 16))
    DetectorElementSize = np.uint32(int("00541203", 16))
    CoincidenceWindowWidth = np.uint32(int("00541210", 16))
    SecondaryCountsType = np.uint32(int("00541220", 16))
    FrameReferenceTime = np.uint32(int("00541300", 16))
    PrimaryPromptsCountsAccumulated = np.uint32(int("00541310", 16))
    SecondaryCountsAccumulated = np.uint32(int("00541311", 16))
    SliceSensitivityFactor = np.uint32(int("00541320", 16))
    DecayFactor = np.uint32(int("00541321", 16))
    DoseCalibrationFactor = np.uint32(int("00541322", 16))
    ScatterFractionFactor = np.uint32(int("00541323", 16))
    DeadTimeFactor = np.uint32(int("00541324", 16))
    ImageIndex = np.uint32(int("00541330", 16))
    CountsIncluded = np.uint32(int("00541400", 16))
    DeadTimeCorrectionFlag = np.uint32(int("00541401", 16))
    HistogramSequence = np.uint32(int("00603000", 16))
    HistogramNumberofBins = np.uint32(int("00603002", 16))
    HistogramFirstBinValue = np.uint32(int("00603004", 16))
    HistogramLastBinValue = np.uint32(int("00603006", 16))
    HistogramBinWidth = np.uint32(int("00603008", 16))
    HistogramExplanation = np.uint32(int("00603010", 16))
    HistogramData = np.uint32(int("00603020", 16))
    SegmentationType = np.uint32(int("00620001", 16))
    SegmentSequence = np.uint32(int("00620002", 16))
    SegmentedPropertyCategoryCodeSequence = np.uint32(int("00620003", 16))
    SegmentNumber = np.uint32(int("00620004", 16))
    SegmentLabel = np.uint32(int("00620005", 16))
    SegmentDescription = np.uint32(int("00620006", 16))
    SegmentAlgorithmType = np.uint32(int("00620008", 16))
    SegmentAlgorithmName = np.uint32(int("00620009", 16))
    SegmentIdentificationSequence = np.uint32(int("0062000A", 16))
    ReferencedSegmentNumber = np.uint32(int("0062000B", 16))
    RecommendedDisplayGrayscaleValue = np.uint32(int("0062000C", 16))
    RecommendedDisplayCIELabValue = np.uint32(int("0062000D", 16))
    MaximumFractionalValue = np.uint32(int("0062000E", 16))
    SegmentedPropertyTypeCodeSequence = np.uint32(int("0062000F", 16))
    SegmentationFractionalType = np.uint32(int("00620010", 16))
    SegmentedPropertyTypeModifierCodeSequence = np.uint32(int("00620011", 16))
    UsedSegmentsSequence = np.uint32(int("00620012", 16))
    TrackingID = np.uint32(int("00620020", 16))
    TrackingUID = np.uint32(int("00620021", 16))
    DeformableRegistrationSequence = np.uint32(int("00640002", 16))
    SourceFrameofReferenceUID = np.uint32(int("00640003", 16))
    DeformableRegistrationGridSequence = np.uint32(int("00640005", 16))
    GridDimensions = np.uint32(int("00640007", 16))
    GridResolution = np.uint32(int("00640008", 16))
    VectorGridData = np.uint32(int("00640009", 16))
    PreDeformationMatrixRegistrationSequence = np.uint32(int("0064000F", 16))
    PostDeformationMatrixRegistrationSequence = np.uint32(int("00640010", 16))
    NumberofSurfaces = np.uint32(int("00660001", 16))
    SurfaceSequence = np.uint32(int("00660002", 16))
    SurfaceNumber = np.uint32(int("00660003", 16))
    SurfaceComments = np.uint32(int("00660004", 16))
    SurfaceProcessing = np.uint32(int("00660009", 16))
    SurfaceProcessingRatio = np.uint32(int("0066000A", 16))
    SurfaceProcessingDescription = np.uint32(int("0066000B", 16))
    RecommendedPresentationOpacity = np.uint32(int("0066000C", 16))
    RecommendedPresentationType = np.uint32(int("0066000D", 16))
    FiniteVolume = np.uint32(int("0066000E", 16))
    Manifold = np.uint32(int("00660010", 16))
    SurfacePointsSequence = np.uint32(int("00660011", 16))
    SurfacePointsNormalsSequence = np.uint32(int("00660012", 16))
    SurfaceMeshPrimitivesSequence = np.uint32(int("00660013", 16))
    NumberofSurfacePoints = np.uint32(int("00660015", 16))
    PointCoordinatesData = np.uint32(int("00660016", 16))
    PointPositionAccuracy = np.uint32(int("00660017", 16))
    MeanPointDistance = np.uint32(int("00660018", 16))
    MaximumPointDistance = np.uint32(int("00660019", 16))
    PointsBoundingBoxCoordinates = np.uint32(int("0066001A", 16))
    AxisofRotation = np.uint32(int("0066001B", 16))
    CenterofRotation = np.uint32(int("0066001C", 16))
    NumberofVectors = np.uint32(int("0066001E", 16))
    VectorDimensionality = np.uint32(int("0066001F", 16))
    VectorAccuracy = np.uint32(int("00660020", 16))
    VectorCoordinateData = np.uint32(int("00660021", 16))
    TrianglePointIndexList = np.uint32(int("00660023", 16))
    EdgePointIndexList = np.uint32(int("00660024", 16))
    VertexPointIndexList = np.uint32(int("00660025", 16))
    TriangleStripSequence = np.uint32(int("00660026", 16))
    TriangleFanSequence = np.uint32(int("00660027", 16))
    LineSequence = np.uint32(int("00660028", 16))
    PrimitivePointIndexList = np.uint32(int("00660029", 16))
    SurfaceCount = np.uint32(int("0066002A", 16))
    ReferencedSurfaceSequence = np.uint32(int("0066002B", 16))
    ReferencedSurfaceNumber = np.uint32(int("0066002C", 16))
    SegmentSurfaceGenerationAlgorithmIdentificationSequence = np.uint32(
        int("0066002D", 16)
    )
    SegmentSurfaceSourceInstanceSequence = np.uint32(int("0066002E", 16))
    AlgorithmFamilyCodeSequence = np.uint32(int("0066002F", 16))
    AlgorithmNameCodeSequence = np.uint32(int("00660030", 16))
    AlgorithmVersion = np.uint32(int("00660031", 16))
    AlgorithmParameters = np.uint32(int("00660032", 16))
    FacetSequence = np.uint32(int("00660034", 16))
    SurfaceProcessingAlgorithmIdentificationSequence = np.uint32(int("00660035", 16))
    AlgorithmName = np.uint32(int("00660036", 16))
    RecommendedPointRadius = np.uint32(int("00660037", 16))
    RecommendedLineThickness = np.uint32(int("00660038", 16))
    LongPrimitivePointIndexList = np.uint32(int("00660040", 16))
    LongTrianglePointIndexList = np.uint32(int("00660041", 16))
    LongEdgePointIndexList = np.uint32(int("00660042", 16))
    LongVertexPointIndexList = np.uint32(int("00660043", 16))
    TrackSetSequence = np.uint32(int("00660101", 16))
    TrackSequence = np.uint32(int("00660102", 16))
    RecommendedDisplayCIELabValueList = np.uint32(int("00660103", 16))
    TrackingAlgorithmIdentificationSequence = np.uint32(int("00660104", 16))
    TrackSetNumber = np.uint32(int("00660105", 16))
    TrackSetLabel = np.uint32(int("00660106", 16))
    TrackSetDescription = np.uint32(int("00660107", 16))
    TrackSetAnatomicalTypeCodeSequence = np.uint32(int("00660108", 16))
    MeasurementsSequence = np.uint32(int("00660121", 16))
    TrackSetStatisticsSequence = np.uint32(int("00660124", 16))
    FloatingPointValues = np.uint32(int("00660125", 16))
    TrackPointIndexList = np.uint32(int("00660129", 16))
    TrackStatisticsSequence = np.uint32(int("00660130", 16))
    MeasurementValuesSequence = np.uint32(int("00660132", 16))
    DiffusionAcquisitionCodeSequence = np.uint32(int("00660133", 16))
    DiffusionModelCodeSequence = np.uint32(int("00660134", 16))
    ImplantSize = np.uint32(int("00686210", 16))
    ImplantTemplateVersion = np.uint32(int("00686221", 16))
    ReplacedImplantTemplateSequence = np.uint32(int("00686222", 16))
    ImplantType = np.uint32(int("00686223", 16))
    DerivationImplantTemplateSequence = np.uint32(int("00686224", 16))
    OriginalImplantTemplateSequence = np.uint32(int("00686225", 16))
    EffectiveDateTime = np.uint32(int("00686226", 16))
    ImplantTargetAnatomySequence = np.uint32(int("00686230", 16))
    InformationFromManufacturerSequence = np.uint32(int("00686260", 16))
    NotificationFromManufacturerSequence = np.uint32(int("00686265", 16))
    InformationIssueDateTime = np.uint32(int("00686270", 16))
    InformationSummary = np.uint32(int("00686280", 16))
    ImplantRegulatoryDisapprovalCodeSequence = np.uint32(int("006862A0", 16))
    OverallTemplateSpatialTolerance = np.uint32(int("006862A5", 16))
    HPGLDocumentSequence = np.uint32(int("006862C0", 16))
    HPGLDocumentID = np.uint32(int("006862D0", 16))
    HPGLDocumentLabel = np.uint32(int("006862D5", 16))
    ViewOrientationCodeSequence = np.uint32(int("006862E0", 16))
    ViewOrientationModifier = np.uint32(int("006862F0", 16))
    HPGLDocumentScaling = np.uint32(int("006862F2", 16))
    HPGLDocument = np.uint32(int("00686300", 16))
    HPGLContourPenNumber = np.uint32(int("00686310", 16))
    HPGLPenSequence = np.uint32(int("00686320", 16))
    HPGLPenNumber = np.uint32(int("00686330", 16))
    HPGLPenLabel = np.uint32(int("00686340", 16))
    HPGLPenDescription = np.uint32(int("00686345", 16))
    RecommendedRotationPoint = np.uint32(int("00686346", 16))
    BoundingRectangle = np.uint32(int("00686347", 16))
    ImplantTemplate3DModelSurfaceNumber = np.uint32(int("00686350", 16))
    SurfaceModelDescriptionSequence = np.uint32(int("00686360", 16))
    SurfaceModelLabel = np.uint32(int("00686380", 16))
    SurfaceModelScalingFactor = np.uint32(int("00686390", 16))
    MaterialsCodeSequence = np.uint32(int("006863A0", 16))
    CoatingMaterialsCodeSequence = np.uint32(int("006863A4", 16))
    ImplantTypeCodeSequence = np.uint32(int("006863A8", 16))
    FixationMethodCodeSequence = np.uint32(int("006863AC", 16))
    MatingFeatureSetsSequence = np.uint32(int("006863B0", 16))
    MatingFeatureSetID = np.uint32(int("006863C0", 16))
    MatingFeatureSetLabel = np.uint32(int("006863D0", 16))
    MatingFeatureSequence = np.uint32(int("006863E0", 16))
    MatingFeatureID = np.uint32(int("006863F0", 16))
    MatingFeatureDegreeofFreedomSequence = np.uint32(int("00686400", 16))
    DegreeofFreedomID = np.uint32(int("00686410", 16))
    DegreeofFreedomType = np.uint32(int("00686420", 16))
    TwoDMatingFeatureCoordinatesSequence = np.uint32(int("00686430", 16))
    ReferencedHPGLDocumentID = np.uint32(int("00686440", 16))
    TwoDMatingPoint = np.uint32(int("00686450", 16))
    TwoDMatingAxes = np.uint32(int("00686460", 16))
    TwoDDegreeofFreedomSequence = np.uint32(int("00686470", 16))
    ThreeDDegreeofFreedomAxis = np.uint32(int("00686490", 16))
    RangeofFreedom = np.uint32(int("006864A0", 16))
    ThreeDMatingPoint = np.uint32(int("006864C0", 16))
    ThreeDMatingAxes = np.uint32(int("006864D0", 16))
    TwoDDegreeofFreedomAxis = np.uint32(int("006864F0", 16))
    PlanningLandmarkPointSequence = np.uint32(int("00686500", 16))
    PlanningLandmarkLineSequence = np.uint32(int("00686510", 16))
    PlanningLandmarkPlaneSequence = np.uint32(int("00686520", 16))
    PlanningLandmarkID = np.uint32(int("00686530", 16))
    PlanningLandmarkDescription = np.uint32(int("00686540", 16))
    PlanningLandmarkIdentificationCodeSequence = np.uint32(int("00686545", 16))
    TwoDPointCoordinatesSequence = np.uint32(int("00686550", 16))
    TwoDPointCoordinates = np.uint32(int("00686560", 16))
    ThreeDPointCoordinates = np.uint32(int("00686590", 16))
    TwoDLineCoordinatesSequence = np.uint32(int("006865A0", 16))
    TwoDLineCoordinates = np.uint32(int("006865B0", 16))
    ThreeDLineCoordinates = np.uint32(int("006865D0", 16))
    TwoDPlaneCoordinatesSequence = np.uint32(int("006865E0", 16))
    TwoDPlaneIntersection = np.uint32(int("006865F0", 16))
    ThreeDPlaneOrigin = np.uint32(int("00686610", 16))
    ThreeDPlaneNormal = np.uint32(int("00686620", 16))
    GraphicAnnotationSequence = np.uint32(int("00700001", 16))
    GraphicLayer = np.uint32(int("00700002", 16))
    BoundingBoxAnnotationUnits = np.uint32(int("00700003", 16))
    AnchorPointAnnotationUnits = np.uint32(int("00700004", 16))
    GraphicAnnotationUnits = np.uint32(int("00700005", 16))
    UnformattedTextValue = np.uint32(int("00700006", 16))
    TextObjectSequence = np.uint32(int("00700008", 16))
    GraphicObjectSequence = np.uint32(int("00700009", 16))
    BoundingBoxTopLeftHandCorner = np.uint32(int("00700010", 16))
    BoundingBoxBottomRightHandCorner = np.uint32(int("00700011", 16))
    BoundingBoxTextHorizontalJustification = np.uint32(int("00700012", 16))
    AnchorPoint = np.uint32(int("00700014", 16))
    AnchorPointVisibility = np.uint32(int("00700015", 16))
    GraphicDimensions = np.uint32(int("00700020", 16))
    NumberofGraphicPoints = np.uint32(int("00700021", 16))
    GraphicData = np.uint32(int("00700022", 16))
    GraphicType = np.uint32(int("00700023", 16))
    GraphicFilled = np.uint32(int("00700024", 16))
    ImageRotationRetired = np.uint32(int("00700040", 16))
    ImageHorizontalFlip = np.uint32(int("00700041", 16))
    ImageRotation = np.uint32(int("00700042", 16))
    DisplayedAreaTopLeftHandCornerTrial = np.uint32(int("00700050", 16))
    DisplayedAreaBottomRightHandCornerTrial = np.uint32(int("00700051", 16))
    DisplayedAreaTopLeftHandCorner = np.uint32(int("00700052", 16))
    DisplayedAreaBottomRightHandCorner = np.uint32(int("00700053", 16))
    DisplayedAreaSelectionSequence = np.uint32(int("0070005A", 16))
    GraphicLayerSequence = np.uint32(int("00700060", 16))
    GraphicLayerOrder = np.uint32(int("00700062", 16))
    GraphicLayerRecommendedDisplayGrayscaleValue = np.uint32(int("00700066", 16))
    GraphicLayerRecommendedDisplayRGBValue = np.uint32(int("00700067", 16))
    GraphicLayerDescription = np.uint32(int("00700068", 16))
    ContentLabel = np.uint32(int("00700080", 16))
    ContentDescription = np.uint32(int("00700081", 16))
    PresentationCreationDate = np.uint32(int("00700082", 16))
    PresentationCreationTime = np.uint32(int("00700083", 16))
    ContentCreatorsName = np.uint32(int("00700084", 16))
    ContentCreatorsIdentificationCodeSequence = np.uint32(int("00700086", 16))
    AlternateContentDescriptionSequence = np.uint32(int("00700087", 16))
    PresentationSizeMode = np.uint32(int("00700100", 16))
    PresentationPixelSpacing = np.uint32(int("00700101", 16))
    PresentationPixelAspectRatio = np.uint32(int("00700102", 16))
    PresentationPixelMagnificationRatio = np.uint32(int("00700103", 16))
    GraphicGroupLabel = np.uint32(int("00700207", 16))
    GraphicGroupDescription = np.uint32(int("00700208", 16))
    CompoundGraphicSequence = np.uint32(int("00700209", 16))
    CompoundGraphicInstanceID = np.uint32(int("00700226", 16))
    FontName = np.uint32(int("00700227", 16))
    FontNameType = np.uint32(int("00700228", 16))
    CSSFontName = np.uint32(int("00700229", 16))
    RotationAngle = np.uint32(int("00700230", 16))
    TextStyleSequence = np.uint32(int("00700231", 16))
    LineStyleSequence = np.uint32(int("00700232", 16))
    FillStyleSequence = np.uint32(int("00700233", 16))
    GraphicGroupSequence = np.uint32(int("00700234", 16))
    TextColorCIELabValue = np.uint32(int("00700241", 16))
    HorizontalAlignment = np.uint32(int("00700242", 16))
    VerticalAlignment = np.uint32(int("00700243", 16))
    ShadowStyle = np.uint32(int("00700244", 16))
    ShadowOffsetX = np.uint32(int("00700245", 16))
    ShadowOffsetY = np.uint32(int("00700246", 16))
    ShadowColorCIELabValue = np.uint32(int("00700247", 16))
    Underlined = np.uint32(int("00700248", 16))
    Bold = np.uint32(int("00700249", 16))
    Italic = np.uint32(int("00700250", 16))
    PatternOnColorCIELabValue = np.uint32(int("00700251", 16))
    PatternOffColorCIELabValue = np.uint32(int("00700252", 16))
    LineThickness = np.uint32(int("00700253", 16))
    LineDashingStyle = np.uint32(int("00700254", 16))
    LinePattern = np.uint32(int("00700255", 16))
    FillPattern = np.uint32(int("00700256", 16))
    FillMode = np.uint32(int("00700257", 16))
    ShadowOpacity = np.uint32(int("00700258", 16))
    GapLength = np.uint32(int("00700261", 16))
    DiameterofVisibility = np.uint32(int("00700262", 16))
    RotationPoint = np.uint32(int("00700273", 16))
    TickAlignment = np.uint32(int("00700274", 16))
    ShowTickLabel = np.uint32(int("00700278", 16))
    TickLabelAlignment = np.uint32(int("00700279", 16))
    CompoundGraphicUnits = np.uint32(int("00700282", 16))
    PatternOnOpacity = np.uint32(int("00700284", 16))
    PatternOffOpacity = np.uint32(int("00700285", 16))
    MajorTicksSequence = np.uint32(int("00700287", 16))
    TickPosition = np.uint32(int("00700288", 16))
    TickLabel = np.uint32(int("00700289", 16))
    CompoundGraphicType = np.uint32(int("00700294", 16))
    GraphicGroupID = np.uint32(int("00700295", 16))
    ShapeType = np.uint32(int("00700306", 16))
    RegistrationSequence = np.uint32(int("00700308", 16))
    MatrixRegistrationSequence = np.uint32(int("00700309", 16))
    MatrixSequence = np.uint32(int("0070030A", 16))
    FrameofReferencetoDisplayedCoordinateSystemTransformationMatrix = np.uint32(
        int("0070030B", 16)
    )
    FrameofReferenceTransformationMatrixType = np.uint32(int("0070030C", 16))
    RegistrationTypeCodeSequence = np.uint32(int("0070030D", 16))
    FiducialDescription = np.uint32(int("0070030F", 16))
    FiducialIdentifier = np.uint32(int("00700310", 16))
    FiducialIdentifierCodeSequence = np.uint32(int("00700311", 16))
    ContourUncertaintyRadius = np.uint32(int("00700312", 16))
    UsedFiducialsSequence = np.uint32(int("00700314", 16))
    GraphicCoordinatesDataSequence = np.uint32(int("00700318", 16))
    FiducialUID = np.uint32(int("0070031A", 16))
    FiducialSetSequence = np.uint32(int("0070031C", 16))
    FiducialSequence = np.uint32(int("0070031E", 16))
    GraphicLayerRecommendedDisplayCIELabValue = np.uint32(int("00700401", 16))
    BlendingSequence = np.uint32(int("00700402", 16))
    RelativeOpacity = np.uint32(int("00700403", 16))
    ReferencedSpatialRegistrationSequence = np.uint32(int("00700404", 16))
    BlendingPosition = np.uint32(int("00700405", 16))
    PresentationDisplayCollectionUID = np.uint32(int("00701101", 16))
    PresentationSequenceCollectionUID = np.uint32(int("00701102", 16))
    PresentationSequencePositionIndex = np.uint32(int("00701103", 16))
    RenderedImageReferenceSequence = np.uint32(int("00701104", 16))
    VolumetricPresentationStateInputSequence = np.uint32(int("00701201", 16))
    PresentationInputType = np.uint32(int("00701202", 16))
    InputSequencePositionIndex = np.uint32(int("00701203", 16))
    Crop = np.uint32(int("00701204", 16))
    CroppingSpecificationIndex = np.uint32(int("00701205", 16))
    CompositingMethod = np.uint32(int("00701206", 16))
    VolumetricPresentationInputNumber = np.uint32(int("00701207", 16))
    ImageVolumeGeometry = np.uint32(int("00701208", 16))
    VolumeCroppingSequence = np.uint32(int("00701301", 16))
    VolumeCroppingMethod = np.uint32(int("00701302", 16))
    BoundingBoxCrop = np.uint32(int("00701303", 16))
    ObliqueCroppingPlaneSequence = np.uint32(int("00701304", 16))
    Plane = np.uint32(int("00701305", 16))
    PlaneNormal = np.uint32(int("00701306", 16))
    CroppingSpecificationNumber = np.uint32(int("00701309", 16))
    MultiPlanarReconstructionStyle = np.uint32(int("00701501", 16))
    MPRThicknessType = np.uint32(int("00701502", 16))
    MPRSlabThickness = np.uint32(int("00701503", 16))
    MPRTopLeftHandCorner = np.uint32(int("00701505", 16))
    MPRViewWidthDirection = np.uint32(int("00701507", 16))
    MPRViewWidth = np.uint32(int("00701508", 16))
    NumberofVolumetricCurvePoints = np.uint32(int("0070150C", 16))
    VolumetricCurvePoints = np.uint32(int("0070150D", 16))
    MPRViewHeightDirection = np.uint32(int("00701511", 16))
    MPRViewHeight = np.uint32(int("00701512", 16))
    PresentationStateClassificationComponentSequence = np.uint32(int("00701801", 16))
    ComponentType = np.uint32(int("00701802", 16))
    ComponentInputSequence = np.uint32(int("00701803", 16))
    VolumetricPresentationInputIndex = np.uint32(int("00701804", 16))
    PresentationStateCompositorComponentSequence = np.uint32(int("00701805", 16))
    WeightingTransferFunctionSequence = np.uint32(int("00701806", 16))
    WeightingLookupTableDescriptor = np.uint32(int("00701807", 16))
    WeightingLookupTableData = np.uint32(int("00701808", 16))
    VolumetricAnnotationSequence = np.uint32(int("00701901", 16))
    ReferencedStructuredContextSequence = np.uint32(int("00701903", 16))
    ReferencedContentItem = np.uint32(int("00701904", 16))
    VolumetricPresentationInputAnnotationSequence = np.uint32(int("00701905", 16))
    AnnotationClipping = np.uint32(int("00701907", 16))
    PresentationAnimationStyle = np.uint32(int("00701A01", 16))
    RecommendedAnimationRate = np.uint32(int("00701A03", 16))
    AnimationCurveSequence = np.uint32(int("00701A04", 16))
    AnimationStepSize = np.uint32(int("00701A05", 16))
    HangingProtocolName = np.uint32(int("00720002", 16))
    HangingProtocolDescription = np.uint32(int("00720004", 16))
    HangingProtocolLevel = np.uint32(int("00720006", 16))
    HangingProtocolCreator = np.uint32(int("00720008", 16))
    HangingProtocolCreationDateTime = np.uint32(int("0072000A", 16))
    HangingProtocolDefinitionSequence = np.uint32(int("0072000C", 16))
    HangingProtocolUserIdentificationCodeSequence = np.uint32(int("0072000E", 16))
    HangingProtocolUserGroupName = np.uint32(int("00720010", 16))
    SourceHangingProtocolSequence = np.uint32(int("00720012", 16))
    NumberofPriorsReferenced = np.uint32(int("00720014", 16))
    ImageSetsSequence = np.uint32(int("00720020", 16))
    ImageSetSelectorSequence = np.uint32(int("00720022", 16))
    ImageSetSelectorUsageFlag = np.uint32(int("00720024", 16))
    SelectorAttribute = np.uint32(int("00720026", 16))
    SelectorValueNumber = np.uint32(int("00720028", 16))
    TimeBasedImageSetsSequence = np.uint32(int("00720030", 16))
    ImageSetNumber = np.uint32(int("00720032", 16))
    ImageSetSelectorCategory = np.uint32(int("00720034", 16))
    RelativeTime = np.uint32(int("00720038", 16))
    RelativeTimeUnits = np.uint32(int("0072003A", 16))
    AbstractPriorValue = np.uint32(int("0072003C", 16))
    AbstractPriorCodeSequence = np.uint32(int("0072003E", 16))
    ImageSetLabel = np.uint32(int("00720040", 16))
    SelectorAttributeVR = np.uint32(int("00720050", 16))
    SelectorSequencePointer = np.uint32(int("00720052", 16))
    SelectorSequencePointerPrivateCreator = np.uint32(int("00720054", 16))
    SelectorAttributePrivateCreator = np.uint32(int("00720056", 16))
    SelectorAEValue = np.uint32(int("0072005E", 16))
    SelectorASValue = np.uint32(int("0072005F", 16))
    SelectorATValue = np.uint32(int("00720060", 16))
    SelectorDAValue = np.uint32(int("00720061", 16))
    SelectorCSValue = np.uint32(int("00720062", 16))
    SelectorDTValue = np.uint32(int("00720063", 16))
    SelectorISValue = np.uint32(int("00720064", 16))
    SelectorOBValue = np.uint32(int("00720065", 16))
    SelectorLOValue = np.uint32(int("00720066", 16))
    SelectorOFValue = np.uint32(int("00720067", 16))
    SelectorLTValue = np.uint32(int("00720068", 16))
    SelectorOWValue = np.uint32(int("00720069", 16))
    SelectorPNValue = np.uint32(int("0072006A", 16))
    SelectorTMValue = np.uint32(int("0072006B", 16))
    SelectorSHValue = np.uint32(int("0072006C", 16))
    SelectorUNValue = np.uint32(int("0072006D", 16))
    SelectorSTValue = np.uint32(int("0072006E", 16))
    SelectorUCValue = np.uint32(int("0072006F", 16))
    SelectorUTValue = np.uint32(int("00720070", 16))
    SelectorURValue = np.uint32(int("00720071", 16))
    SelectorDSValue = np.uint32(int("00720072", 16))
    SelectorODValue = np.uint32(int("00720073", 16))
    SelectorFDValue = np.uint32(int("00720074", 16))
    SelectorOLValue = np.uint32(int("00720075", 16))
    SelectorFLValue = np.uint32(int("00720076", 16))
    SelectorULValue = np.uint32(int("00720078", 16))
    SelectorUSValue = np.uint32(int("0072007A", 16))
    SelectorSLValue = np.uint32(int("0072007C", 16))
    SelectorSSValue = np.uint32(int("0072007E", 16))
    SelectorUIValue = np.uint32(int("0072007F", 16))
    SelectorCodeSequenceValue = np.uint32(int("00720080", 16))
    NumberofScreens = np.uint32(int("00720100", 16))
    NominalScreenDefinitionSequence = np.uint32(int("00720102", 16))
    NumberofVerticalPixels = np.uint32(int("00720104", 16))
    NumberofHorizontalPixels = np.uint32(int("00720106", 16))
    DisplayEnvironmentSpatialPosition = np.uint32(int("00720108", 16))
    ScreenMinimumGrayscaleBitDepth = np.uint32(int("0072010A", 16))
    ScreenMinimumColorBitDepth = np.uint32(int("0072010C", 16))
    ApplicationMaximumRepaintTime = np.uint32(int("0072010E", 16))
    DisplaySetsSequence = np.uint32(int("00720200", 16))
    DisplaySetNumber = np.uint32(int("00720202", 16))
    DisplaySetLabel = np.uint32(int("00720203", 16))
    DisplaySetPresentationGroup = np.uint32(int("00720204", 16))
    DisplaySetPresentationGroupDescription = np.uint32(int("00720206", 16))
    PartialDataDisplayHandling = np.uint32(int("00720208", 16))
    SynchronizedScrollingSequence = np.uint32(int("00720210", 16))
    DisplaySetScrollingGroup = np.uint32(int("00720212", 16))
    NavigationIndicatorSequence = np.uint32(int("00720214", 16))
    NavigationDisplaySet = np.uint32(int("00720216", 16))
    ReferenceDisplaySets = np.uint32(int("00720218", 16))
    ImageBoxesSequence = np.uint32(int("00720300", 16))
    ImageBoxNumber = np.uint32(int("00720302", 16))
    ImageBoxLayoutType = np.uint32(int("00720304", 16))
    ImageBoxTileHorizontalDimension = np.uint32(int("00720306", 16))
    ImageBoxTileVerticalDimension = np.uint32(int("00720308", 16))
    ImageBoxScrollDirection = np.uint32(int("00720310", 16))
    ImageBoxSmallScrollType = np.uint32(int("00720312", 16))
    ImageBoxSmallScrollAmount = np.uint32(int("00720314", 16))
    ImageBoxLargeScrollType = np.uint32(int("00720316", 16))
    ImageBoxLargeScrollAmount = np.uint32(int("00720318", 16))
    ImageBoxOverlapPriority = np.uint32(int("00720320", 16))
    CineRelativetoRealTime = np.uint32(int("00720330", 16))
    FilterOperationsSequence = np.uint32(int("00720400", 16))
    FilterbyCategory = np.uint32(int("00720402", 16))
    FilterbyAttributePresence = np.uint32(int("00720404", 16))
    FilterbyOperator = np.uint32(int("00720406", 16))
    StructuredDisplayBackgroundCIELabValue = np.uint32(int("00720420", 16))
    EmptyImageBoxCIELabValue = np.uint32(int("00720421", 16))
    StructuredDisplayImageBoxSequence = np.uint32(int("00720422", 16))
    StructuredDisplayTextBoxSequence = np.uint32(int("00720424", 16))
    ReferencedFirstFrameSequence = np.uint32(int("00720427", 16))
    ImageBoxSynchronizationSequence = np.uint32(int("00720430", 16))
    SynchronizedImageBoxList = np.uint32(int("00720432", 16))
    TypeofSynchronization = np.uint32(int("00720434", 16))
    BlendingOperationType = np.uint32(int("00720500", 16))
    ReformattingOperationType = np.uint32(int("00720510", 16))
    ReformattingThickness = np.uint32(int("00720512", 16))
    ReformattingInterval = np.uint32(int("00720514", 16))
    ReformattingOperationInitialViewDirection = np.uint32(int("00720516", 16))
    ThreeDRenderingType = np.uint32(int("00720520", 16))
    SortingOperationsSequence = np.uint32(int("00720600", 16))
    SortbyCategory = np.uint32(int("00720602", 16))
    SortingDirection = np.uint32(int("00720604", 16))
    DisplaySetPatientOrientation = np.uint32(int("00720700", 16))
    VOIType = np.uint32(int("00720702", 16))
    PseudoColorType = np.uint32(int("00720704", 16))
    PseudoColorPaletteInstanceReferenceSequence = np.uint32(int("00720705", 16))
    ShowGrayscaleInverted = np.uint32(int("00720706", 16))
    ShowImageTrueSizeFlag = np.uint32(int("00720710", 16))
    ShowGraphicAnnotationFlag = np.uint32(int("00720712", 16))
    ShowPatientDemographicsFlag = np.uint32(int("00720714", 16))
    ShowAcquisitionTechniquesFlag = np.uint32(int("00720716", 16))
    DisplaySetHorizontalJustification = np.uint32(int("00720717", 16))
    DisplaySetVerticalJustification = np.uint32(int("00720718", 16))
    ContinuationStartMeterset = np.uint32(int("00740120", 16))
    ContinuationEndMeterset = np.uint32(int("00740121", 16))
    ProcedureStepState = np.uint32(int("00741000", 16))
    ProcedureStepProgressInformationSequence = np.uint32(int("00741002", 16))
    ProcedureStepProgress = np.uint32(int("00741004", 16))
    ProcedureStepProgressDescription = np.uint32(int("00741006", 16))
    ProcedureStepCommunicationsURISequence = np.uint32(int("00741008", 16))
    ContactURI = np.uint32(int("0074100A", 16))
    ContactDisplayName = np.uint32(int("0074100C", 16))
    ProcedureStepDiscontinuationReasonCodeSequence = np.uint32(int("0074100E", 16))
    BeamTaskSequence = np.uint32(int("00741020", 16))
    BeamTaskType = np.uint32(int("00741022", 16))
    BeamOrderIndexTrial = np.uint32(int("00741024", 16))
    AutosequenceFlag = np.uint32(int("00741025", 16))
    TableTopVerticalAdjustedPosition = np.uint32(int("00741026", 16))
    TableTopLongitudinalAdjustedPosition = np.uint32(int("00741027", 16))
    TableTopLateralAdjustedPosition = np.uint32(int("00741028", 16))
    PatientSupportAdjustedAngle = np.uint32(int("0074102A", 16))
    TableTopEccentricAdjustedAngle = np.uint32(int("0074102B", 16))
    TableTopPitchAdjustedAngle = np.uint32(int("0074102C", 16))
    TableTopRollAdjustedAngle = np.uint32(int("0074102D", 16))
    DeliveryVerificationImageSequence = np.uint32(int("00741030", 16))
    VerificationImageTiming = np.uint32(int("00741032", 16))
    DoubleExposureFlag = np.uint32(int("00741034", 16))
    DoubleExposureOrdering = np.uint32(int("00741036", 16))
    DoubleExposureMetersetTrial = np.uint32(int("00741038", 16))
    DoubleExposureFieldDeltaTrial = np.uint32(int("0074103A", 16))
    RelatedReferenceRTImageSequence = np.uint32(int("00741040", 16))
    GeneralMachineVerificationSequence = np.uint32(int("00741042", 16))
    ConventionalMachineVerificationSequence = np.uint32(int("00741044", 16))
    IonMachineVerificationSequence = np.uint32(int("00741046", 16))
    FailedAttributesSequence = np.uint32(int("00741048", 16))
    OverriddenAttributesSequence = np.uint32(int("0074104A", 16))
    ConventionalControlPointVerificationSequence = np.uint32(int("0074104C", 16))
    IonControlPointVerificationSequence = np.uint32(int("0074104E", 16))
    AttributeOccurrenceSequence = np.uint32(int("00741050", 16))
    AttributeOccurrencePointer = np.uint32(int("00741052", 16))
    AttributeItemSelector = np.uint32(int("00741054", 16))
    AttributeOccurrencePrivateCreator = np.uint32(int("00741056", 16))
    SelectorSequencePointerItems = np.uint32(int("00741057", 16))
    ScheduledProcedureStepPriority = np.uint32(int("00741200", 16))
    WorklistLabel = np.uint32(int("00741202", 16))
    ProcedureStepLabel = np.uint32(int("00741204", 16))
    ScheduledProcessingParametersSequence = np.uint32(int("00741210", 16))
    PerformedProcessingParametersSequence = np.uint32(int("00741212", 16))
    UnifiedProcedureStepPerformedProcedureSequence = np.uint32(int("00741216", 16))
    RelatedProcedureStepSequence = np.uint32(int("00741220", 16))
    ProcedureStepRelationshipType = np.uint32(int("00741222", 16))
    ReplacedProcedureStepSequence = np.uint32(int("00741224", 16))
    DeletionLock = np.uint32(int("00741230", 16))
    ReceivingAE = np.uint32(int("00741234", 16))
    RequestingAE = np.uint32(int("00741236", 16))
    ReasonforCancellation = np.uint32(int("00741238", 16))
    SCPStatus = np.uint32(int("00741242", 16))
    SubscriptionListStatus = np.uint32(int("00741244", 16))
    UnifiedProcedureStepListStatus = np.uint32(int("00741246", 16))
    BeamOrderIndex = np.uint32(int("00741324", 16))
    DoubleExposureMeterset = np.uint32(int("00741338", 16))
    DoubleExposureFieldDelta = np.uint32(int("0074133A", 16))
    BrachyTaskSequence = np.uint32(int("00741401", 16))
    ContinuationStartTotalReferenceAirKerma = np.uint32(int("00741402", 16))
    ContinuationEndTotalReferenceAirKerma = np.uint32(int("00741403", 16))
    ContinuationPulseNumber = np.uint32(int("00741404", 16))
    ChannelDeliveryOrderSequence = np.uint32(int("00741405", 16))
    ReferencedChannelNumber = np.uint32(int("00741406", 16))
    StartCumulativeTimeWeight = np.uint32(int("00741407", 16))
    EndCumulativeTimeWeight = np.uint32(int("00741408", 16))
    OmittedChannelSequence = np.uint32(int("00741409", 16))
    ReasonforChannelOmission = np.uint32(int("0074140A", 16))
    ReasonforChannelOmissionDescription = np.uint32(int("0074140B", 16))
    ChannelDeliveryOrderIndex = np.uint32(int("0074140C", 16))
    ChannelDeliveryContinuationSequence = np.uint32(int("0074140D", 16))
    OmittedApplicationSetupSequence = np.uint32(int("0074140E", 16))
    ImplantAssemblyTemplateName = np.uint32(int("00760001", 16))
    ImplantAssemblyTemplateIssuer = np.uint32(int("00760003", 16))
    ImplantAssemblyTemplateVersion = np.uint32(int("00760006", 16))
    ReplacedImplantAssemblyTemplateSequence = np.uint32(int("00760008", 16))
    ImplantAssemblyTemplateType = np.uint32(int("0076000A", 16))
    OriginalImplantAssemblyTemplateSequence = np.uint32(int("0076000C", 16))
    DerivationImplantAssemblyTemplateSequence = np.uint32(int("0076000E", 16))
    ImplantAssemblyTemplateTargetAnatomySequence = np.uint32(int("00760010", 16))
    ProcedureTypeCodeSequence = np.uint32(int("00760020", 16))
    SurgicalTechnique = np.uint32(int("00760030", 16))
    ComponentTypesSequence = np.uint32(int("00760032", 16))
    ComponentTypeCodeSequence = np.uint32(int("00760034", 16))
    ExclusiveComponentType = np.uint32(int("00760036", 16))
    MandatoryComponentType = np.uint32(int("00760038", 16))
    ComponentSequence = np.uint32(int("00760040", 16))
    ComponentID = np.uint32(int("00760055", 16))
    ComponentAssemblySequence = np.uint32(int("00760060", 16))
    Component1ReferencedID = np.uint32(int("00760070", 16))
    Component1ReferencedMatingFeatureSetID = np.uint32(int("00760080", 16))
    Component1ReferencedMatingFeatureID = np.uint32(int("00760090", 16))
    Component2ReferencedID = np.uint32(int("007600A0", 16))
    Component2ReferencedMatingFeatureSetID = np.uint32(int("007600B0", 16))
    Component2ReferencedMatingFeatureID = np.uint32(int("007600C0", 16))
    ImplantTemplateGroupName = np.uint32(int("00780001", 16))
    ImplantTemplateGroupDescription = np.uint32(int("00780010", 16))
    ImplantTemplateGroupIssuer = np.uint32(int("00780020", 16))
    ImplantTemplateGroupVersion = np.uint32(int("00780024", 16))
    ReplacedImplantTemplateGroupSequence = np.uint32(int("00780026", 16))
    ImplantTemplateGroupTargetAnatomySequence = np.uint32(int("00780028", 16))
    ImplantTemplateGroupMembersSequence = np.uint32(int("0078002A", 16))
    ImplantTemplateGroupMemberID = np.uint32(int("0078002E", 16))
    ThreeDImplantTemplateGroupMemberMatchingPoint = np.uint32(int("00780050", 16))
    ThreeDImplantTemplateGroupMemberMatchingAxes = np.uint32(int("00780060", 16))
    ImplantTemplateGroupMemberMatching2DCoordinatesSequence = np.uint32(
        int("00780070", 16)
    )
    TwoDImplantTemplateGroupMemberMatchingPoint = np.uint32(int("00780090", 16))
    TwoDImplantTemplateGroupMemberMatchingAxes = np.uint32(int("007800A0", 16))
    ImplantTemplateGroupVariationDimensionSequence = np.uint32(int("007800B0", 16))
    ImplantTemplateGroupVariationDimensionName = np.uint32(int("007800B2", 16))
    ImplantTemplateGroupVariationDimensionRankSequence = np.uint32(int("007800B4", 16))
    ReferencedImplantTemplateGroupMemberID = np.uint32(int("007800B6", 16))
    ImplantTemplateGroupVariationDimensionRank = np.uint32(int("007800B8", 16))
    SurfaceScanAcquisitionTypeCodeSequence = np.uint32(int("00800001", 16))
    SurfaceScanModeCodeSequence = np.uint32(int("00800002", 16))
    RegistrationMethodCodeSequence = np.uint32(int("00800003", 16))
    ShotDurationTime = np.uint32(int("00800004", 16))
    ShotOffsetTime = np.uint32(int("00800005", 16))
    SurfacePointPresentationValueData = np.uint32(int("00800006", 16))
    SurfacePointColorCIELabValueData = np.uint32(int("00800007", 16))
    UVMappingSequence = np.uint32(int("00800008", 16))
    TextureLabel = np.uint32(int("00800009", 16))
    UValueData = np.uint32(int("00800010", 16))
    VValueData = np.uint32(int("00800011", 16))
    ReferencedTextureSequence = np.uint32(int("00800012", 16))
    ReferencedSurfaceDataSequence = np.uint32(int("00800013", 16))
    AssessmentSummary = np.uint32(int("00820001", 16))
    AssessmentSummaryDescription = np.uint32(int("00820003", 16))
    AssessedSOPInstanceSequence = np.uint32(int("00820004", 16))
    ReferencedComparisonSOPInstanceSequence = np.uint32(int("00820005", 16))
    NumberofAssessmentObservations = np.uint32(int("00820006", 16))
    AssessmentObservationsSequence = np.uint32(int("00820007", 16))
    ObservationSignificance = np.uint32(int("00820008", 16))
    ObservationDescription = np.uint32(int("0082000A", 16))
    StructuredConstraintObservationSequence = np.uint32(int("0082000C", 16))
    AssessedAttributeValueSequence = np.uint32(int("00820010", 16))
    AssessmentSetID = np.uint32(int("00820016", 16))
    AssessmentRequesterSequence = np.uint32(int("00820017", 16))
    SelectorAttributeName = np.uint32(int("00820018", 16))
    SelectorAttributeKeyword = np.uint32(int("00820019", 16))
    AssessmentTypeCodeSequence = np.uint32(int("00820021", 16))
    ObservationBasisCodeSequence = np.uint32(int("00820022", 16))
    AssessmentLabel = np.uint32(int("00820023", 16))
    ConstraintType = np.uint32(int("00820032", 16))
    SpecificationSelectionGuidance = np.uint32(int("00820033", 16))
    ConstraintValueSequence = np.uint32(int("00820034", 16))
    RecommendedDefaultValueSequence = np.uint32(int("00820035", 16))
    ConstraintViolationSignificance = np.uint32(int("00820036", 16))
    ConstraintViolationCondition = np.uint32(int("00820037", 16))
    StorageMediaFilesetID = np.uint32(int("00880130", 16))
    StorageMediaFilesetUID = np.uint32(int("00880140", 16))
    IconImageSequence = np.uint32(int("00880200", 16))
    TopicTitle = np.uint32(int("00880904", 16))
    TopicSubject = np.uint32(int("00880906", 16))
    TopicAuthor = np.uint32(int("00880910", 16))
    TopicKeywords = np.uint32(int("00880912", 16))
    SOPInstanceStatus = np.uint32(int("01000410", 16))
    SOPAuthorizationDateTime = np.uint32(int("01000420", 16))
    SOPAuthorizationComment = np.uint32(int("01000424", 16))
    AuthorizationEquipmentCertificationNumber = np.uint32(int("01000426", 16))
    MACIDNumber = np.uint32(int("04000005", 16))
    MACCalculationTransferSyntaxUID = np.uint32(int("04000010", 16))
    MACAlgorithm = np.uint32(int("04000015", 16))
    DataElementsSigned = np.uint32(int("04000020", 16))
    DigitalSignatureUID = np.uint32(int("04000100", 16))
    DigitalSignatureDateTime = np.uint32(int("04000105", 16))
    CertificateType = np.uint32(int("04000110", 16))
    CertificateofSigner = np.uint32(int("04000115", 16))
    Signature = np.uint32(int("04000120", 16))
    CertifiedTimestampType = np.uint32(int("04000305", 16))
    CertifiedTimestamp = np.uint32(int("04000310", 16))
    DigitalSignaturePurposeCodeSequence = np.uint32(int("04000401", 16))
    ReferencedDigitalSignatureSequence = np.uint32(int("04000402", 16))
    ReferencedSOPInstanceMACSequence = np.uint32(int("04000403", 16))
    MAC = np.uint32(int("04000404", 16))
    EncryptedAttributesSequence = np.uint32(int("04000500", 16))
    EncryptedContentTransferSyntaxUID = np.uint32(int("04000510", 16))
    EncryptedContent = np.uint32(int("04000520", 16))
    ModifiedAttributesSequence = np.uint32(int("04000550", 16))
    OriginalAttributesSequence = np.uint32(int("04000561", 16))
    AttributeModificationDateTime = np.uint32(int("04000562", 16))
    ModifyingSystem = np.uint32(int("04000563", 16))
    SourceofPreviousValues = np.uint32(int("04000564", 16))
    ReasonfortheAttributeModification = np.uint32(int("04000565", 16))
    NumberofCopies = np.uint32(int("20000010", 16))
    PrinterConfigurationSequence = np.uint32(int("2000001E", 16))
    PrintPriority = np.uint32(int("20000020", 16))
    MediumType = np.uint32(int("20000030", 16))
    FilmDestination = np.uint32(int("20000040", 16))
    FilmSessionLabel = np.uint32(int("20000050", 16))
    MemoryAllocation = np.uint32(int("20000060", 16))
    MaximumMemoryAllocation = np.uint32(int("20000061", 16))
    ColorImagePrintingFlag = np.uint32(int("20000062", 16))
    CollationFlag = np.uint32(int("20000063", 16))
    AnnotationFlag = np.uint32(int("20000065", 16))
    ImageOverlayFlag = np.uint32(int("20000067", 16))
    PresentationLUTFlag = np.uint32(int("20000069", 16))
    ImageBoxPresentationLUTFlag = np.uint32(int("2000006A", 16))
    MemoryBitDepth = np.uint32(int("200000A0", 16))
    PrintingBitDepth = np.uint32(int("200000A1", 16))
    MediaInstalledSequence = np.uint32(int("200000A2", 16))
    OtherMediaAvailableSequence = np.uint32(int("200000A4", 16))
    SupportedImageDisplayFormatsSequence = np.uint32(int("200000A8", 16))
    ReferencedFilmBoxSequence = np.uint32(int("20000500", 16))
    ReferencedStoredPrintSequence = np.uint32(int("20000510", 16))
    ImageDisplayFormat = np.uint32(int("20100010", 16))
    AnnotationDisplayFormatID = np.uint32(int("20100030", 16))
    FilmOrientation = np.uint32(int("20100040", 16))
    FilmSizeID = np.uint32(int("20100050", 16))
    PrinterResolutionID = np.uint32(int("20100052", 16))
    DefaultPrinterResolutionID = np.uint32(int("20100054", 16))
    MagnificationType = np.uint32(int("20100060", 16))
    SmoothingType = np.uint32(int("20100080", 16))
    DefaultMagnificationType = np.uint32(int("201000A6", 16))
    OtherMagnificationTypesAvailable = np.uint32(int("201000A7", 16))
    DefaultSmoothingType = np.uint32(int("201000A8", 16))
    OtherSmoothingTypesAvailable = np.uint32(int("201000A9", 16))
    BorderDensity = np.uint32(int("20100100", 16))
    EmptyImageDensity = np.uint32(int("20100110", 16))
    MinDensity = np.uint32(int("20100120", 16))
    MaxDensity = np.uint32(int("20100130", 16))
    Trim = np.uint32(int("20100140", 16))
    ConfigurationInformation = np.uint32(int("20100150", 16))
    ConfigurationInformationDescription = np.uint32(int("20100152", 16))
    MaximumCollatedFilms = np.uint32(int("20100154", 16))
    Illumination = np.uint32(int("2010015E", 16))
    ReflectedAmbientLight = np.uint32(int("20100160", 16))
    PrinterPixelSpacing = np.uint32(int("20100376", 16))
    ReferencedFilmSessionSequence = np.uint32(int("20100500", 16))
    ReferencedImageBoxSequence = np.uint32(int("20100510", 16))
    ReferencedBasicAnnotationBoxSequence = np.uint32(int("20100520", 16))
    ImageBoxPosition = np.uint32(int("20200010", 16))
    Polarity = np.uint32(int("20200020", 16))
    RequestedImageSize = np.uint32(int("20200030", 16))
    RequestedDecimateCropBehavior = np.uint32(int("20200040", 16))
    RequestedResolutionID = np.uint32(int("20200050", 16))
    RequestedImageSizeFlag = np.uint32(int("202000A0", 16))
    DecimateCropResult = np.uint32(int("202000A2", 16))
    BasicGrayscaleImageSequence = np.uint32(int("20200110", 16))
    BasicColorImageSequence = np.uint32(int("20200111", 16))
    ReferencedImageOverlayBoxSequence = np.uint32(int("20200130", 16))
    ReferencedVOILUTBoxSequence = np.uint32(int("20200140", 16))
    AnnotationPosition = np.uint32(int("20300010", 16))
    TextString = np.uint32(int("20300020", 16))
    ReferencedOverlayPlaneSequence = np.uint32(int("20400010", 16))
    ReferencedOverlayPlaneGroups = np.uint32(int("20400011", 16))
    OverlayPixelDataSequence = np.uint32(int("20400020", 16))
    OverlayMagnificationType = np.uint32(int("20400060", 16))
    OverlaySmoothingType = np.uint32(int("20400070", 16))
    OverlayorImageMagnification = np.uint32(int("20400072", 16))
    MagnifytoNumberofColumns = np.uint32(int("20400074", 16))
    OverlayForegroundDensity = np.uint32(int("20400080", 16))
    OverlayBackgroundDensity = np.uint32(int("20400082", 16))
    OverlayMode = np.uint32(int("20400090", 16))
    ThresholdDensity = np.uint32(int("20400100", 16))
    ReferencedImageBoxSequenceRetired = np.uint32(int("20400500", 16))
    PresentationLUTSequence = np.uint32(int("20500010", 16))
    PresentationLUTShape = np.uint32(int("20500020", 16))
    ReferencedPresentationLUTSequence = np.uint32(int("20500500", 16))
    PrintJobID = np.uint32(int("21000010", 16))
    ExecutionStatus = np.uint32(int("21000020", 16))
    ExecutionStatusInfo = np.uint32(int("21000030", 16))
    CreationDate = np.uint32(int("21000040", 16))
    CreationTime = np.uint32(int("21000050", 16))
    Originator = np.uint32(int("21000070", 16))
    DestinationAE = np.uint32(int("21000140", 16))
    OwnerID = np.uint32(int("21000160", 16))
    NumberofFilms = np.uint32(int("21000170", 16))
    ReferencedPrintJobSequencePullStoredPrint = np.uint32(int("21000500", 16))
    PrinterStatus = np.uint32(int("21100010", 16))
    PrinterStatusInfo = np.uint32(int("21100020", 16))
    PrinterName = np.uint32(int("21100030", 16))
    PrintQueueID = np.uint32(int("21100099", 16))
    QueueStatus = np.uint32(int("21200010", 16))
    PrintJobDescriptionSequence = np.uint32(int("21200050", 16))
    ReferencedPrintJobSequence = np.uint32(int("21200070", 16))
    PrintManagementCapabilitiesSequence = np.uint32(int("21300010", 16))
    PrinterCharacteristicsSequence = np.uint32(int("21300015", 16))
    FilmBoxContentSequence = np.uint32(int("21300030", 16))
    ImageBoxContentSequence = np.uint32(int("21300040", 16))
    AnnotationContentSequence = np.uint32(int("21300050", 16))
    ImageOverlayBoxContentSequence = np.uint32(int("21300060", 16))
    PresentationLUTContentSequence = np.uint32(int("21300080", 16))
    ProposedStudySequence = np.uint32(int("213000A0", 16))
    OriginalImageSequence = np.uint32(int("213000C0", 16))
    LabelUsingInformationExtractedFromInstances = np.uint32(int("22000001", 16))
    LabelText = np.uint32(int("22000002", 16))
    LabelStyleSelection = np.uint32(int("22000003", 16))
    MediaDisposition = np.uint32(int("22000004", 16))
    BarcodeValue = np.uint32(int("22000005", 16))
    BarcodeSymbology = np.uint32(int("22000006", 16))
    AllowMediaSplitting = np.uint32(int("22000007", 16))
    IncludeNonDICOMObjects = np.uint32(int("22000008", 16))
    IncludeDisplayApplication = np.uint32(int("22000009", 16))
    PreserveCompositeInstancesAfterMediaCreation = np.uint32(int("2200000A", 16))
    TotalNumberofPiecesofMediaCreated = np.uint32(int("2200000B", 16))
    RequestedMediaApplicationProfile = np.uint32(int("2200000C", 16))
    ReferencedStorageMediaSequence = np.uint32(int("2200000D", 16))
    FailureAttributes = np.uint32(int("2200000E", 16))
    AllowLossyCompression = np.uint32(int("2200000F", 16))
    RequestPriority = np.uint32(int("22000020", 16))
    RTImageLabel = np.uint32(int("30020002", 16))
    RTImageName = np.uint32(int("30020003", 16))
    RTImageDescription = np.uint32(int("30020004", 16))
    ReportedValuesOrigin = np.uint32(int("3002000A", 16))
    RTImagePlane = np.uint32(int("3002000C", 16))
    XRayImageReceptorTranslation = np.uint32(int("3002000D", 16))
    XRayImageReceptorAngle = np.uint32(int("3002000E", 16))
    RTImageOrientation = np.uint32(int("30020010", 16))
    ImagePlanePixelSpacing = np.uint32(int("30020011", 16))
    RTImagePosition = np.uint32(int("30020012", 16))
    RadiationMachineName = np.uint32(int("30020020", 16))
    RadiationMachineSAD = np.uint32(int("30020022", 16))
    RadiationMachineSSD = np.uint32(int("30020024", 16))
    RTImageSID = np.uint32(int("30020026", 16))
    SourcetoReferenceObjectDistance = np.uint32(int("30020028", 16))
    FractionNumber = np.uint32(int("30020029", 16))
    ExposureSequence = np.uint32(int("30020030", 16))
    MetersetExposure = np.uint32(int("30020032", 16))
    DiaphragmPosition = np.uint32(int("30020034", 16))
    FluenceMapSequence = np.uint32(int("30020040", 16))
    FluenceDataSource = np.uint32(int("30020041", 16))
    FluenceDataScale = np.uint32(int("30020042", 16))
    PrimaryFluenceModeSequence = np.uint32(int("30020050", 16))
    FluenceMode = np.uint32(int("30020051", 16))
    FluenceModeID = np.uint32(int("30020052", 16))
    DVHType = np.uint32(int("30040001", 16))
    DoseUnits = np.uint32(int("30040002", 16))
    DoseType = np.uint32(int("30040004", 16))
    SpatialTransformofDose = np.uint32(int("30040005", 16))
    DoseComment = np.uint32(int("30040006", 16))
    NormalizationPoint = np.uint32(int("30040008", 16))
    DoseSummationType = np.uint32(int("3004000A", 16))
    GridFrameOffsetVector = np.uint32(int("3004000C", 16))
    DoseGridScaling = np.uint32(int("3004000E", 16))
    RTDoseROISequence = np.uint32(int("30040010", 16))
    DoseValue = np.uint32(int("30040012", 16))
    TissueHeterogeneityCorrection = np.uint32(int("30040014", 16))
    DVHNormalizationPoint = np.uint32(int("30040040", 16))
    DVHNormalizationDoseValue = np.uint32(int("30040042", 16))
    DVHSequence = np.uint32(int("30040050", 16))
    DVHDoseScaling = np.uint32(int("30040052", 16))
    DVHVolumeUnits = np.uint32(int("30040054", 16))
    DVHNumberofBins = np.uint32(int("30040056", 16))
    DVHData = np.uint32(int("30040058", 16))
    DVHReferencedROISequence = np.uint32(int("30040060", 16))
    DVHROIContributionType = np.uint32(int("30040062", 16))
    DVHMinimumDose = np.uint32(int("30040070", 16))
    DVHMaximumDose = np.uint32(int("30040072", 16))
    DVHMeanDose = np.uint32(int("30040074", 16))
    StructureSetLabel = np.uint32(int("30060002", 16))
    StructureSetName = np.uint32(int("30060004", 16))
    StructureSetDescription = np.uint32(int("30060006", 16))
    StructureSetDate = np.uint32(int("30060008", 16))
    StructureSetTime = np.uint32(int("30060009", 16))
    ReferencedFrameofReferenceSequence = np.uint32(int("30060010", 16))
    RTReferencedStudySequence = np.uint32(int("30060012", 16))
    RTReferencedSeriesSequence = np.uint32(int("30060014", 16))
    ContourImageSequence = np.uint32(int("30060016", 16))
    PredecessorStructureSetSequence = np.uint32(int("30060018", 16))
    StructureSetROISequence = np.uint32(int("30060020", 16))
    ROINumber = np.uint32(int("30060022", 16))
    ReferencedFrameofReferenceUID = np.uint32(int("30060024", 16))
    ROIName = np.uint32(int("30060026", 16))
    ROIDescription = np.uint32(int("30060028", 16))
    ROIDisplayColor = np.uint32(int("3006002A", 16))
    ROIVolume = np.uint32(int("3006002C", 16))
    RTRelatedROISequence = np.uint32(int("30060030", 16))
    RTROIRelationship = np.uint32(int("30060033", 16))
    ROIGenerationAlgorithm = np.uint32(int("30060036", 16))
    ROIGenerationDescription = np.uint32(int("30060038", 16))
    ROIContourSequence = np.uint32(int("30060039", 16))
    ContourSequence = np.uint32(int("30060040", 16))
    ContourGeometricType = np.uint32(int("30060042", 16))
    ContourSlabThickness = np.uint32(int("30060044", 16))
    ContourOffsetVector = np.uint32(int("30060045", 16))
    NumberofContourPoints = np.uint32(int("30060046", 16))
    ContourNumber = np.uint32(int("30060048", 16))
    AttachedContours = np.uint32(int("30060049", 16))
    ContourData = np.uint32(int("30060050", 16))
    RTROIObservationsSequence = np.uint32(int("30060080", 16))
    ObservationNumber = np.uint32(int("30060082", 16))
    ReferencedROINumber = np.uint32(int("30060084", 16))
    ROIObservationLabel = np.uint32(int("30060085", 16))
    RTROIIdentificationCodeSequence = np.uint32(int("30060086", 16))
    ROIObservationDescription = np.uint32(int("30060088", 16))
    RelatedRTROIObservationsSequence = np.uint32(int("300600A0", 16))
    RTROIInterpretedType = np.uint32(int("300600A4", 16))
    ROIInterpreter = np.uint32(int("300600A6", 16))
    ROIPhysicalPropertiesSequence = np.uint32(int("300600B0", 16))
    ROIPhysicalProperty = np.uint32(int("300600B2", 16))
    ROIPhysicalPropertyValue = np.uint32(int("300600B4", 16))
    ROIElementalCompositionSequence = np.uint32(int("300600B6", 16))
    ROIElementalCompositionAtomicNumber = np.uint32(int("300600B7", 16))
    ROIElementalCompositionAtomicMassFraction = np.uint32(int("300600B8", 16))
    AdditionalRTROIIdentificationCodeSequence = np.uint32(int("300600B9", 16))
    FrameofReferenceRelationshipSequence = np.uint32(int("300600C0", 16))
    RelatedFrameofReferenceUID = np.uint32(int("300600C2", 16))
    FrameofReferenceTransformationType = np.uint32(int("300600C4", 16))
    FrameofReferenceTransformationMatrix = np.uint32(int("300600C6", 16))
    FrameofReferenceTransformationComment = np.uint32(int("300600C8", 16))
    MeasuredDoseReferenceSequence = np.uint32(int("30080010", 16))
    MeasuredDoseDescription = np.uint32(int("30080012", 16))
    MeasuredDoseType = np.uint32(int("30080014", 16))
    MeasuredDoseValue = np.uint32(int("30080016", 16))
    TreatmentSessionBeamSequence = np.uint32(int("30080020", 16))
    TreatmentSessionIonBeamSequence = np.uint32(int("30080021", 16))
    CurrentFractionNumber = np.uint32(int("30080022", 16))
    TreatmentControlPointDate = np.uint32(int("30080024", 16))
    TreatmentControlPointTime = np.uint32(int("30080025", 16))
    TreatmentTerminationStatus = np.uint32(int("3008002A", 16))
    TreatmentTerminationCode = np.uint32(int("3008002B", 16))
    TreatmentVerificationStatus = np.uint32(int("3008002C", 16))
    ReferencedTreatmentRecordSequence = np.uint32(int("30080030", 16))
    SpecifiedPrimaryMeterset = np.uint32(int("30080032", 16))
    SpecifiedSecondaryMeterset = np.uint32(int("30080033", 16))
    DeliveredPrimaryMeterset = np.uint32(int("30080036", 16))
    DeliveredSecondaryMeterset = np.uint32(int("30080037", 16))
    SpecifiedTreatmentTime = np.uint32(int("3008003A", 16))
    DeliveredTreatmentTime = np.uint32(int("3008003B", 16))
    ControlPointDeliverySequence = np.uint32(int("30080040", 16))
    IonControlPointDeliverySequence = np.uint32(int("30080041", 16))
    SpecifiedMeterset = np.uint32(int("30080042", 16))
    DeliveredMeterset = np.uint32(int("30080044", 16))
    MetersetRateSet = np.uint32(int("30080045", 16))
    MetersetRateDelivered = np.uint32(int("30080046", 16))
    ScanSpotMetersetsDelivered = np.uint32(int("30080047", 16))
    DoseRateDelivered = np.uint32(int("30080048", 16))
    TreatmentSummaryCalculatedDoseReferenceSequence = np.uint32(int("30080050", 16))
    CumulativeDosetoDoseReference = np.uint32(int("30080052", 16))
    FirstTreatmentDate = np.uint32(int("30080054", 16))
    MostRecentTreatmentDate = np.uint32(int("30080056", 16))
    NumberofFractionsDelivered = np.uint32(int("3008005A", 16))
    OverrideSequence = np.uint32(int("30080060", 16))
    ParameterSequencePointer = np.uint32(int("30080061", 16))
    OverrideParameterPointer = np.uint32(int("30080062", 16))
    ParameterItemIndex = np.uint32(int("30080063", 16))
    MeasuredDoseReferenceNumber = np.uint32(int("30080064", 16))
    ParameterPointer = np.uint32(int("30080065", 16))
    OverrideReason = np.uint32(int("30080066", 16))
    CorrectedParameterSequence = np.uint32(int("30080068", 16))
    CorrectionValue = np.uint32(int("3008006A", 16))
    CalculatedDoseReferenceSequence = np.uint32(int("30080070", 16))
    CalculatedDoseReferenceNumber = np.uint32(int("30080072", 16))
    CalculatedDoseReferenceDescription = np.uint32(int("30080074", 16))
    CalculatedDoseReferenceDoseValue = np.uint32(int("30080076", 16))
    StartMeterset = np.uint32(int("30080078", 16))
    EndMeterset = np.uint32(int("3008007A", 16))
    ReferencedMeasuredDoseReferenceSequence = np.uint32(int("30080080", 16))
    ReferencedMeasuredDoseReferenceNumber = np.uint32(int("30080082", 16))
    ReferencedCalculatedDoseReferenceSequence = np.uint32(int("30080090", 16))
    ReferencedCalculatedDoseReferenceNumber = np.uint32(int("30080092", 16))
    BeamLimitingDeviceLeafPairsSequence = np.uint32(int("300800A0", 16))
    RecordedWedgeSequence = np.uint32(int("300800B0", 16))
    RecordedCompensatorSequence = np.uint32(int("300800C0", 16))
    RecordedBlockSequence = np.uint32(int("300800D0", 16))
    TreatmentSummaryMeasuredDoseReferenceSequence = np.uint32(int("300800E0", 16))
    RecordedSnoutSequence = np.uint32(int("300800F0", 16))
    RecordedRangeShifterSequence = np.uint32(int("300800F2", 16))
    RecordedLateralSpreadingDeviceSequence = np.uint32(int("300800F4", 16))
    RecordedRangeModulatorSequence = np.uint32(int("300800F6", 16))
    RecordedSourceSequence = np.uint32(int("30080100", 16))
    SourceSerialNumber = np.uint32(int("30080105", 16))
    TreatmentSessionApplicationSetupSequence = np.uint32(int("30080110", 16))
    ApplicationSetupCheck = np.uint32(int("30080116", 16))
    RecordedBrachyAccessoryDeviceSequence = np.uint32(int("30080120", 16))
    ReferencedBrachyAccessoryDeviceNumber = np.uint32(int("30080122", 16))
    RecordedChannelSequence = np.uint32(int("30080130", 16))
    SpecifiedChannelTotalTime = np.uint32(int("30080132", 16))
    DeliveredChannelTotalTime = np.uint32(int("30080134", 16))
    SpecifiedNumberofPulses = np.uint32(int("30080136", 16))
    DeliveredNumberofPulses = np.uint32(int("30080138", 16))
    SpecifiedPulseRepetitionInterval = np.uint32(int("3008013A", 16))
    DeliveredPulseRepetitionInterval = np.uint32(int("3008013C", 16))
    RecordedSourceApplicatorSequence = np.uint32(int("30080140", 16))
    ReferencedSourceApplicatorNumber = np.uint32(int("30080142", 16))
    RecordedChannelShieldSequence = np.uint32(int("30080150", 16))
    ReferencedChannelShieldNumber = np.uint32(int("30080152", 16))
    BrachyControlPointDeliveredSequence = np.uint32(int("30080160", 16))
    SafePositionExitDate = np.uint32(int("30080162", 16))
    SafePositionExitTime = np.uint32(int("30080164", 16))
    SafePositionReturnDate = np.uint32(int("30080166", 16))
    SafePositionReturnTime = np.uint32(int("30080168", 16))
    PulseSpecificBrachyControlPointDeliveredSequence = np.uint32(int("30080171", 16))
    PulseNumber = np.uint32(int("30080172", 16))
    BrachyPulseControlPointDeliveredSequence = np.uint32(int("30080173", 16))
    CurrentTreatmentStatus = np.uint32(int("30080200", 16))
    TreatmentStatusComment = np.uint32(int("30080202", 16))
    FractionGroupSummarySequence = np.uint32(int("30080220", 16))
    ReferencedFractionNumber = np.uint32(int("30080223", 16))
    FractionGroupType = np.uint32(int("30080224", 16))
    BeamStopperPosition = np.uint32(int("30080230", 16))
    FractionStatusSummarySequence = np.uint32(int("30080240", 16))
    TreatmentDate = np.uint32(int("30080250", 16))
    TreatmentTime = np.uint32(int("30080251", 16))
    RTPlanLabel = np.uint32(int("300A0002", 16))
    RTPlanName = np.uint32(int("300A0003", 16))
    RTPlanDescription = np.uint32(int("300A0004", 16))
    RTPlanDate = np.uint32(int("300A0006", 16))
    RTPlanTime = np.uint32(int("300A0007", 16))
    TreatmentProtocols = np.uint32(int("300A0009", 16))
    PlanIntent = np.uint32(int("300A000A", 16))
    TreatmentSites = np.uint32(int("300A000B", 16))
    RTPlanGeometry = np.uint32(int("300A000C", 16))
    PrescriptionDescription = np.uint32(int("300A000E", 16))
    DoseReferenceSequence = np.uint32(int("300A0010", 16))
    DoseReferenceNumber = np.uint32(int("300A0012", 16))
    DoseReferenceUID = np.uint32(int("300A0013", 16))
    DoseReferenceStructureType = np.uint32(int("300A0014", 16))
    NominalBeamEnergyUnit = np.uint32(int("300A0015", 16))
    DoseReferenceDescription = np.uint32(int("300A0016", 16))
    DoseReferencePointCoordinates = np.uint32(int("300A0018", 16))
    NominalPriorDose = np.uint32(int("300A001A", 16))
    DoseReferenceType = np.uint32(int("300A0020", 16))
    ConstraintWeight = np.uint32(int("300A0021", 16))
    DeliveryWarningDose = np.uint32(int("300A0022", 16))
    DeliveryMaximumDose = np.uint32(int("300A0023", 16))
    TargetMinimumDose = np.uint32(int("300A0025", 16))
    TargetPrescriptionDose = np.uint32(int("300A0026", 16))
    TargetMaximumDose = np.uint32(int("300A0027", 16))
    TargetUnderdoseVolumeFraction = np.uint32(int("300A0028", 16))
    OrganatRiskFullvolumeDose = np.uint32(int("300A002A", 16))
    OrganatRiskLimitDose = np.uint32(int("300A002B", 16))
    OrganatRiskMaximumDose = np.uint32(int("300A002C", 16))
    OrganatRiskOverdoseVolumeFraction = np.uint32(int("300A002D", 16))
    ToleranceTableSequence = np.uint32(int("300A0040", 16))
    ToleranceTableNumber = np.uint32(int("300A0042", 16))
    ToleranceTableLabel = np.uint32(int("300A0043", 16))
    GantryAngleTolerance = np.uint32(int("300A0044", 16))
    BeamLimitingDeviceAngleTolerance = np.uint32(int("300A0046", 16))
    BeamLimitingDeviceToleranceSequence = np.uint32(int("300A0048", 16))
    BeamLimitingDevicePositionTolerance = np.uint32(int("300A004A", 16))
    SnoutPositionTolerance = np.uint32(int("300A004B", 16))
    PatientSupportAngleTolerance = np.uint32(int("300A004C", 16))
    TableTopEccentricAngleTolerance = np.uint32(int("300A004E", 16))
    TableTopPitchAngleTolerance = np.uint32(int("300A004F", 16))
    TableTopRollAngleTolerance = np.uint32(int("300A0050", 16))
    TableTopVerticalPositionTolerance = np.uint32(int("300A0051", 16))
    TableTopLongitudinalPositionTolerance = np.uint32(int("300A0052", 16))
    TableTopLateralPositionTolerance = np.uint32(int("300A0053", 16))
    RTPlanRelationship = np.uint32(int("300A0055", 16))
    FractionGroupSequence = np.uint32(int("300A0070", 16))
    FractionGroupNumber = np.uint32(int("300A0071", 16))
    FractionGroupDescription = np.uint32(int("300A0072", 16))
    NumberofFractionsPlanned = np.uint32(int("300A0078", 16))
    NumberofFractionPatternDigitsPerDay = np.uint32(int("300A0079", 16))
    RepeatFractionCycleLength = np.uint32(int("300A007A", 16))
    FractionPattern = np.uint32(int("300A007B", 16))
    NumberofBeams = np.uint32(int("300A0080", 16))
    BeamDoseSpecificationPoint = np.uint32(int("300A0082", 16))
    BeamDose = np.uint32(int("300A0084", 16))
    BeamMeterset = np.uint32(int("300A0086", 16))
    BeamDosePointDepth = np.uint32(int("300A0088", 16))
    BeamDosePointEquivalentDepth = np.uint32(int("300A0089", 16))
    BeamDosePointSSD = np.uint32(int("300A008A", 16))
    BeamDoseMeaning = np.uint32(int("300A008B", 16))
    BeamDoseVerificationControlPointSequence = np.uint32(int("300A008C", 16))
    AverageBeamDosePointDepth = np.uint32(int("300A008D", 16))
    AverageBeamDosePointEquivalentDepth = np.uint32(int("300A008E", 16))
    AverageBeamDosePointSSD = np.uint32(int("300A008F", 16))
    BeamDoseType = np.uint32(int("300A0090", 16))
    AlternateBeamDose = np.uint32(int("300A0091", 16))
    AlternateBeamDoseType = np.uint32(int("300A0092", 16))
    NumberofBrachyApplicationSetups = np.uint32(int("300A00A0", 16))
    BrachyApplicationSetupDoseSpecificationPoint = np.uint32(int("300A00A2", 16))
    BrachyApplicationSetupDose = np.uint32(int("300A00A4", 16))
    BeamSequence = np.uint32(int("300A00B0", 16))
    TreatmentMachineName = np.uint32(int("300A00B2", 16))
    PrimaryDosimeterUnit = np.uint32(int("300A00B3", 16))
    SourceAxisDistance = np.uint32(int("300A00B4", 16))
    BeamLimitingDeviceSequence = np.uint32(int("300A00B6", 16))
    RTBeamLimitingDeviceType = np.uint32(int("300A00B8", 16))
    SourcetoBeamLimitingDeviceDistance = np.uint32(int("300A00BA", 16))
    IsocentertoBeamLimitingDeviceDistance = np.uint32(int("300A00BB", 16))
    NumberofLeafJawPairs = np.uint32(int("300A00BC", 16))
    LeafPositionBoundaries = np.uint32(int("300A00BE", 16))
    BeamNumber = np.uint32(int("300A00C0", 16))
    BeamName = np.uint32(int("300A00C2", 16))
    BeamDescription = np.uint32(int("300A00C3", 16))
    BeamType = np.uint32(int("300A00C4", 16))
    BeamDeliveryDurationLimit = np.uint32(int("300A00C5", 16))
    RadiationType = np.uint32(int("300A00C6", 16))
    HighDoseTechniqueType = np.uint32(int("300A00C7", 16))
    ReferenceImageNumber = np.uint32(int("300A00C8", 16))
    PlannedVerificationImageSequence = np.uint32(int("300A00CA", 16))
    ImagingDeviceSpecificAcquisitionParameters = np.uint32(int("300A00CC", 16))
    TreatmentDeliveryType = np.uint32(int("300A00CE", 16))
    NumberofWedges = np.uint32(int("300A00D0", 16))
    WedgeSequence = np.uint32(int("300A00D1", 16))
    WedgeNumber = np.uint32(int("300A00D2", 16))
    WedgeType = np.uint32(int("300A00D3", 16))
    WedgeID = np.uint32(int("300A00D4", 16))
    WedgeAngle = np.uint32(int("300A00D5", 16))
    WedgeFactor = np.uint32(int("300A00D6", 16))
    TotalWedgeTrayWaterEquivalentThickness = np.uint32(int("300A00D7", 16))
    WedgeOrientation = np.uint32(int("300A00D8", 16))
    IsocentertoWedgeTrayDistance = np.uint32(int("300A00D9", 16))
    SourcetoWedgeTrayDistance = np.uint32(int("300A00DA", 16))
    WedgeThinEdgePosition = np.uint32(int("300A00DB", 16))
    BolusID = np.uint32(int("300A00DC", 16))
    BolusDescription = np.uint32(int("300A00DD", 16))
    EffectiveWedgeAngle = np.uint32(int("300A00DE", 16))
    NumberofCompensators = np.uint32(int("300A00E0", 16))
    MaterialID = np.uint32(int("300A00E1", 16))
    TotalCompensatorTrayFactor = np.uint32(int("300A00E2", 16))
    CompensatorSequence = np.uint32(int("300A00E3", 16))
    CompensatorNumber = np.uint32(int("300A00E4", 16))
    CompensatorID = np.uint32(int("300A00E5", 16))
    SourcetoCompensatorTrayDistance = np.uint32(int("300A00E6", 16))
    CompensatorRows = np.uint32(int("300A00E7", 16))
    CompensatorColumns = np.uint32(int("300A00E8", 16))
    CompensatorPixelSpacing = np.uint32(int("300A00E9", 16))
    CompensatorPosition = np.uint32(int("300A00EA", 16))
    CompensatorTransmissionData = np.uint32(int("300A00EB", 16))
    CompensatorThicknessData = np.uint32(int("300A00EC", 16))
    NumberofBoli = np.uint32(int("300A00ED", 16))
    CompensatorType = np.uint32(int("300A00EE", 16))
    CompensatorTrayID = np.uint32(int("300A00EF", 16))
    NumberofBlocks = np.uint32(int("300A00F0", 16))
    TotalBlockTrayFactor = np.uint32(int("300A00F2", 16))
    TotalBlockTrayWaterEquivalentThickness = np.uint32(int("300A00F3", 16))
    BlockSequence = np.uint32(int("300A00F4", 16))
    BlockTrayID = np.uint32(int("300A00F5", 16))
    SourcetoBlockTrayDistance = np.uint32(int("300A00F6", 16))
    IsocentertoBlockTrayDistance = np.uint32(int("300A00F7", 16))
    BlockType = np.uint32(int("300A00F8", 16))
    AccessoryCode = np.uint32(int("300A00F9", 16))
    BlockDivergence = np.uint32(int("300A00FA", 16))
    BlockMountingPosition = np.uint32(int("300A00FB", 16))
    BlockNumber = np.uint32(int("300A00FC", 16))
    BlockName = np.uint32(int("300A00FE", 16))
    BlockThickness = np.uint32(int("300A0100", 16))
    BlockTransmission = np.uint32(int("300A0102", 16))
    BlockNumberofPoints = np.uint32(int("300A0104", 16))
    BlockData = np.uint32(int("300A0106", 16))
    ApplicatorSequence = np.uint32(int("300A0107", 16))
    ApplicatorID = np.uint32(int("300A0108", 16))
    ApplicatorType = np.uint32(int("300A0109", 16))
    ApplicatorDescription = np.uint32(int("300A010A", 16))
    CumulativeDoseReferenceCoefficient = np.uint32(int("300A010C", 16))
    FinalCumulativeMetersetWeight = np.uint32(int("300A010E", 16))
    NumberofControlPoints = np.uint32(int("300A0110", 16))
    ControlPointSequence = np.uint32(int("300A0111", 16))
    ControlPointIndex = np.uint32(int("300A0112", 16))
    NominalBeamEnergy = np.uint32(int("300A0114", 16))
    DoseRateSet = np.uint32(int("300A0115", 16))
    WedgePositionSequence = np.uint32(int("300A0116", 16))
    WedgePosition = np.uint32(int("300A0118", 16))
    BeamLimitingDevicePositionSequence = np.uint32(int("300A011A", 16))
    LeafJawPositions = np.uint32(int("300A011C", 16))
    GantryAngle = np.uint32(int("300A011E", 16))
    GantryRotationDirection = np.uint32(int("300A011F", 16))
    BeamLimitingDeviceAngle = np.uint32(int("300A0120", 16))
    BeamLimitingDeviceRotationDirection = np.uint32(int("300A0121", 16))
    PatientSupportAngle = np.uint32(int("300A0122", 16))
    PatientSupportRotationDirection = np.uint32(int("300A0123", 16))
    TableTopEccentricAxisDistance = np.uint32(int("300A0124", 16))
    TableTopEccentricAngle = np.uint32(int("300A0125", 16))
    TableTopEccentricRotationDirection = np.uint32(int("300A0126", 16))
    TableTopVerticalPosition = np.uint32(int("300A0128", 16))
    TableTopLongitudinalPosition = np.uint32(int("300A0129", 16))
    TableTopLateralPosition = np.uint32(int("300A012A", 16))
    IsocenterPosition = np.uint32(int("300A012C", 16))
    SurfaceEntryPoint = np.uint32(int("300A012E", 16))
    SourcetoSurfaceDistance = np.uint32(int("300A0130", 16))
    AverageBeamDosePointSourcetoExternalContourDistance = np.uint32(int("300A0131", 16))
    SourcetoExternalContourDistance = np.uint32(int("300A0132", 16))
    ExternalContourEntryPoint = np.uint32(int("300A0133", 16))
    CumulativeMetersetWeight = np.uint32(int("300A0134", 16))
    TableTopPitchAngle = np.uint32(int("300A0140", 16))
    TableTopPitchRotationDirection = np.uint32(int("300A0142", 16))
    TableTopRollAngle = np.uint32(int("300A0144", 16))
    TableTopRollRotationDirection = np.uint32(int("300A0146", 16))
    HeadFixationAngle = np.uint32(int("300A0148", 16))
    GantryPitchAngle = np.uint32(int("300A014A", 16))
    GantryPitchRotationDirection = np.uint32(int("300A014C", 16))
    GantryPitchAngleTolerance = np.uint32(int("300A014E", 16))
    FixationEye = np.uint32(int("300A0150", 16))
    ChairHeadFramePosition = np.uint32(int("300A0151", 16))
    HeadFixationAngleTolerance = np.uint32(int("300A0152", 16))
    ChairHeadFramePositionTolerance = np.uint32(int("300A0153", 16))
    FixationLightAzimuthalAngleTolerance = np.uint32(int("300A0154", 16))
    FixationLightPolarAngleTolerance = np.uint32(int("300A0155", 16))
    PatientSetupSequence = np.uint32(int("300A0180", 16))
    PatientSetupNumber = np.uint32(int("300A0182", 16))
    PatientSetupLabel = np.uint32(int("300A0183", 16))
    PatientAdditionalPosition = np.uint32(int("300A0184", 16))
    FixationDeviceSequence = np.uint32(int("300A0190", 16))
    FixationDeviceType = np.uint32(int("300A0192", 16))
    FixationDeviceLabel = np.uint32(int("300A0194", 16))
    FixationDeviceDescription = np.uint32(int("300A0196", 16))
    FixationDevicePosition = np.uint32(int("300A0198", 16))
    FixationDevicePitchAngle = np.uint32(int("300A0199", 16))
    FixationDeviceRollAngle = np.uint32(int("300A019A", 16))
    ShieldingDeviceSequence = np.uint32(int("300A01A0", 16))
    ShieldingDeviceType = np.uint32(int("300A01A2", 16))
    ShieldingDeviceLabel = np.uint32(int("300A01A4", 16))
    ShieldingDeviceDescription = np.uint32(int("300A01A6", 16))
    ShieldingDevicePosition = np.uint32(int("300A01A8", 16))
    SetupTechnique = np.uint32(int("300A01B0", 16))
    SetupTechniqueDescription = np.uint32(int("300A01B2", 16))
    SetupDeviceSequence = np.uint32(int("300A01B4", 16))
    SetupDeviceType = np.uint32(int("300A01B6", 16))
    SetupDeviceLabel = np.uint32(int("300A01B8", 16))
    SetupDeviceDescription = np.uint32(int("300A01BA", 16))
    SetupDeviceParameter = np.uint32(int("300A01BC", 16))
    SetupReferenceDescription = np.uint32(int("300A01D0", 16))
    TableTopVerticalSetupDisplacement = np.uint32(int("300A01D2", 16))
    TableTopLongitudinalSetupDisplacement = np.uint32(int("300A01D4", 16))
    TableTopLateralSetupDisplacement = np.uint32(int("300A01D6", 16))
    BrachyTreatmentTechnique = np.uint32(int("300A0200", 16))
    BrachyTreatmentType = np.uint32(int("300A0202", 16))
    TreatmentMachineSequence = np.uint32(int("300A0206", 16))
    SourceSequence = np.uint32(int("300A0210", 16))
    SourceNumber = np.uint32(int("300A0212", 16))
    SourceType = np.uint32(int("300A0214", 16))
    SourceManufacturer = np.uint32(int("300A0216", 16))
    ActiveSourceDiameter = np.uint32(int("300A0218", 16))
    ActiveSourceLength = np.uint32(int("300A021A", 16))
    SourceModelID = np.uint32(int("300A021B", 16))
    SourceDescription = np.uint32(int("300A021C", 16))
    SourceEncapsulationNominalThickness = np.uint32(int("300A0222", 16))
    SourceEncapsulationNominalTransmission = np.uint32(int("300A0224", 16))
    SourceIsotopeName = np.uint32(int("300A0226", 16))
    SourceIsotopeHalfLife = np.uint32(int("300A0228", 16))
    SourceStrengthUnits = np.uint32(int("300A0229", 16))
    ReferenceAirKermaRate = np.uint32(int("300A022A", 16))
    SourceStrength = np.uint32(int("300A022B", 16))
    SourceStrengthReferenceDate = np.uint32(int("300A022C", 16))
    SourceStrengthReferenceTime = np.uint32(int("300A022E", 16))
    ApplicationSetupSequence = np.uint32(int("300A0230", 16))
    ApplicationSetupType = np.uint32(int("300A0232", 16))
    ApplicationSetupNumber = np.uint32(int("300A0234", 16))
    ApplicationSetupName = np.uint32(int("300A0236", 16))
    ApplicationSetupManufacturer = np.uint32(int("300A0238", 16))
    TemplateNumber = np.uint32(int("300A0240", 16))
    TemplateType = np.uint32(int("300A0242", 16))
    TemplateName = np.uint32(int("300A0244", 16))
    TotalReferenceAirKerma = np.uint32(int("300A0250", 16))
    BrachyAccessoryDeviceSequence = np.uint32(int("300A0260", 16))
    BrachyAccessoryDeviceNumber = np.uint32(int("300A0262", 16))
    BrachyAccessoryDeviceID = np.uint32(int("300A0263", 16))
    BrachyAccessoryDeviceType = np.uint32(int("300A0264", 16))
    BrachyAccessoryDeviceName = np.uint32(int("300A0266", 16))
    BrachyAccessoryDeviceNominalThickness = np.uint32(int("300A026A", 16))
    BrachyAccessoryDeviceNominalTransmission = np.uint32(int("300A026C", 16))
    ChannelSequence = np.uint32(int("300A0280", 16))
    ChannelNumber = np.uint32(int("300A0282", 16))
    ChannelLength = np.uint32(int("300A0284", 16))
    ChannelTotalTime = np.uint32(int("300A0286", 16))
    SourceMovementType = np.uint32(int("300A0288", 16))
    NumberofPulses = np.uint32(int("300A028A", 16))
    PulseRepetitionInterval = np.uint32(int("300A028C", 16))
    SourceApplicatorNumber = np.uint32(int("300A0290", 16))
    SourceApplicatorID = np.uint32(int("300A0291", 16))
    SourceApplicatorType = np.uint32(int("300A0292", 16))
    SourceApplicatorName = np.uint32(int("300A0294", 16))
    SourceApplicatorLength = np.uint32(int("300A0296", 16))
    SourceApplicatorManufacturer = np.uint32(int("300A0298", 16))
    SourceApplicatorWallNominalThickness = np.uint32(int("300A029C", 16))
    SourceApplicatorWallNominalTransmission = np.uint32(int("300A029E", 16))
    SourceApplicatorStepSize = np.uint32(int("300A02A0", 16))
    TransferTubeNumber = np.uint32(int("300A02A2", 16))
    TransferTubeLength = np.uint32(int("300A02A4", 16))
    ChannelShieldSequence = np.uint32(int("300A02B0", 16))
    ChannelShieldNumber = np.uint32(int("300A02B2", 16))
    ChannelShieldID = np.uint32(int("300A02B3", 16))
    ChannelShieldName = np.uint32(int("300A02B4", 16))
    ChannelShieldNominalThickness = np.uint32(int("300A02B8", 16))
    ChannelShieldNominalTransmission = np.uint32(int("300A02BA", 16))
    FinalCumulativeTimeWeight = np.uint32(int("300A02C8", 16))
    BrachyControlPointSequence = np.uint32(int("300A02D0", 16))
    ControlPointRelativePosition = np.uint32(int("300A02D2", 16))
    ControlPoint3DPosition = np.uint32(int("300A02D4", 16))
    CumulativeTimeWeight = np.uint32(int("300A02D6", 16))
    CompensatorDivergence = np.uint32(int("300A02E0", 16))
    CompensatorMountingPosition = np.uint32(int("300A02E1", 16))
    SourcetoCompensatorDistance = np.uint32(int("300A02E2", 16))
    TotalCompensatorTrayWaterEquivalentThickness = np.uint32(int("300A02E3", 16))
    IsocentertoCompensatorTrayDistance = np.uint32(int("300A02E4", 16))
    CompensatorColumnOffset = np.uint32(int("300A02E5", 16))
    IsocentertoCompensatorDistances = np.uint32(int("300A02E6", 16))
    CompensatorRelativeStoppingPowerRatio = np.uint32(int("300A02E7", 16))
    CompensatorMillingToolDiameter = np.uint32(int("300A02E8", 16))
    IonRangeCompensatorSequence = np.uint32(int("300A02EA", 16))
    CompensatorDescription = np.uint32(int("300A02EB", 16))
    RadiationMassNumber = np.uint32(int("300A0302", 16))
    RadiationAtomicNumber = np.uint32(int("300A0304", 16))
    RadiationChargeState = np.uint32(int("300A0306", 16))
    ScanMode = np.uint32(int("300A0308", 16))
    ModulatedScanModeType = np.uint32(int("300A0309", 16))
    VirtualSourceAxisDistances = np.uint32(int("300A030A", 16))
    SnoutSequence = np.uint32(int("300A030C", 16))
    SnoutPosition = np.uint32(int("300A030D", 16))
    SnoutID = np.uint32(int("300A030F", 16))
    NumberofRangeShifters = np.uint32(int("300A0312", 16))
    RangeShifterSequence = np.uint32(int("300A0314", 16))
    RangeShifterNumber = np.uint32(int("300A0316", 16))
    RangeShifterID = np.uint32(int("300A0318", 16))
    RangeShifterType = np.uint32(int("300A0320", 16))
    RangeShifterDescription = np.uint32(int("300A0322", 16))
    NumberofLateralSpreadingDevices = np.uint32(int("300A0330", 16))
    LateralSpreadingDeviceSequence = np.uint32(int("300A0332", 16))
    LateralSpreadingDeviceNumber = np.uint32(int("300A0334", 16))
    LateralSpreadingDeviceID = np.uint32(int("300A0336", 16))
    LateralSpreadingDeviceType = np.uint32(int("300A0338", 16))
    LateralSpreadingDeviceDescription = np.uint32(int("300A033A", 16))
    LateralSpreadingDeviceWaterEquivalentThickness = np.uint32(int("300A033C", 16))
    NumberofRangeModulators = np.uint32(int("300A0340", 16))
    RangeModulatorSequence = np.uint32(int("300A0342", 16))
    RangeModulatorNumber = np.uint32(int("300A0344", 16))
    RangeModulatorID = np.uint32(int("300A0346", 16))
    RangeModulatorType = np.uint32(int("300A0348", 16))
    RangeModulatorDescription = np.uint32(int("300A034A", 16))
    BeamCurrentModulationID = np.uint32(int("300A034C", 16))
    PatientSupportType = np.uint32(int("300A0350", 16))
    PatientSupportID = np.uint32(int("300A0352", 16))
    PatientSupportAccessoryCode = np.uint32(int("300A0354", 16))
    TrayAccessoryCode = np.uint32(int("300A0355", 16))
    FixationLightAzimuthalAngle = np.uint32(int("300A0356", 16))
    FixationLightPolarAngle = np.uint32(int("300A0358", 16))
    MetersetRate = np.uint32(int("300A035A", 16))
    RangeShifterSettingsSequence = np.uint32(int("300A0360", 16))
    RangeShifterSetting = np.uint32(int("300A0362", 16))
    IsocentertoRangeShifterDistance = np.uint32(int("300A0364", 16))
    RangeShifterWaterEquivalentThickness = np.uint32(int("300A0366", 16))
    LateralSpreadingDeviceSettingsSequence = np.uint32(int("300A0370", 16))
    LateralSpreadingDeviceSetting = np.uint32(int("300A0372", 16))
    IsocentertoLateralSpreadingDeviceDistance = np.uint32(int("300A0374", 16))
    RangeModulatorSettingsSequence = np.uint32(int("300A0380", 16))
    RangeModulatorGatingStartValue = np.uint32(int("300A0382", 16))
    RangeModulatorGatingStopValue = np.uint32(int("300A0384", 16))
    RangeModulatorGatingStartWaterEquivalentThickness = np.uint32(int("300A0386", 16))
    RangeModulatorGatingStopWaterEquivalentThickness = np.uint32(int("300A0388", 16))
    IsocentertoRangeModulatorDistance = np.uint32(int("300A038A", 16))
    ScanSpotTuneID = np.uint32(int("300A0390", 16))
    ScanSpotPrescribedIndices = np.uint32(int("300A0391", 16))
    NumberofScanSpotPositions = np.uint32(int("300A0392", 16))
    ScanSpotReordered = np.uint32(int("300A0393", 16))
    ScanSpotPositionMap = np.uint32(int("300A0394", 16))
    ScanSpotReorderingAllowed = np.uint32(int("300A0395", 16))
    ScanSpotMetersetWeights = np.uint32(int("300A0396", 16))
    ScanningSpotSize = np.uint32(int("300A0398", 16))
    NumberofPaintings = np.uint32(int("300A039A", 16))
    IonToleranceTableSequence = np.uint32(int("300A03A0", 16))
    IonBeamSequence = np.uint32(int("300A03A2", 16))
    IonBeamLimitingDeviceSequence = np.uint32(int("300A03A4", 16))
    IonBlockSequence = np.uint32(int("300A03A6", 16))
    IonControlPointSequence = np.uint32(int("300A03A8", 16))
    IonWedgeSequence = np.uint32(int("300A03AA", 16))
    IonWedgePositionSequence = np.uint32(int("300A03AC", 16))
    ReferencedSetupImageSequence = np.uint32(int("300A0401", 16))
    SetupImageComment = np.uint32(int("300A0402", 16))
    MotionSynchronizationSequence = np.uint32(int("300A0410", 16))
    ControlPointOrientation = np.uint32(int("300A0412", 16))
    GeneralAccessorySequence = np.uint32(int("300A0420", 16))
    GeneralAccessoryID = np.uint32(int("300A0421", 16))
    GeneralAccessoryDescription = np.uint32(int("300A0422", 16))
    GeneralAccessoryType = np.uint32(int("300A0423", 16))
    GeneralAccessoryNumber = np.uint32(int("300A0424", 16))
    SourcetoGeneralAccessoryDistance = np.uint32(int("300A0425", 16))
    ApplicatorGeometrySequence = np.uint32(int("300A0431", 16))
    ApplicatorApertureShape = np.uint32(int("300A0432", 16))
    ApplicatorOpening = np.uint32(int("300A0433", 16))
    ApplicatorOpeningX = np.uint32(int("300A0434", 16))
    ApplicatorOpeningY = np.uint32(int("300A0435", 16))
    SourcetoApplicatorMountingPositionDistance = np.uint32(int("300A0436", 16))
    NumberofBlockSlabItems = np.uint32(int("300A0440", 16))
    BlockSlabSequence = np.uint32(int("300A0441", 16))
    BlockSlabThickness = np.uint32(int("300A0442", 16))
    BlockSlabNumber = np.uint32(int("300A0443", 16))
    DeviceMotionControlSequence = np.uint32(int("300A0450", 16))
    DeviceMotionExecutionMode = np.uint32(int("300A0451", 16))
    DeviceMotionObservationMode = np.uint32(int("300A0452", 16))
    DeviceMotionParameterCodeSequence = np.uint32(int("300A0453", 16))
    DistalDepthFraction = np.uint32(int("300A0501", 16))
    DistalDepth = np.uint32(int("300A0502", 16))
    NominalRangeModulationFractions = np.uint32(int("300A0503", 16))
    NominalRangeModulatedRegionDepths = np.uint32(int("300A0504", 16))
    DepthDoseParametersSequence = np.uint32(int("300A0505", 16))
    DeliveredDepthDoseParametersSequence = np.uint32(int("300A0506", 16))
    DeliveredDistalDepthFraction = np.uint32(int("300A0507", 16))
    DeliveredDistalDepth = np.uint32(int("300A0508", 16))
    DeliveredNominalRangeModulationFractions = np.uint32(int("300A0509", 16))
    DeliveredNominalRangeModulatedRegionDepths = np.uint32(int("300A0510", 16))
    DeliveredReferenceDoseDefinition = np.uint32(int("300A0511", 16))
    ReferenceDoseDefinition = np.uint32(int("300A0512", 16))
    ReferencedRTPlanSequence = np.uint32(int("300C0002", 16))
    ReferencedBeamSequence = np.uint32(int("300C0004", 16))
    ReferencedBeamNumber = np.uint32(int("300C0006", 16))
    ReferencedReferenceImageNumber = np.uint32(int("300C0007", 16))
    StartCumulativeMetersetWeight = np.uint32(int("300C0008", 16))
    EndCumulativeMetersetWeight = np.uint32(int("300C0009", 16))
    ReferencedBrachyApplicationSetupSequence = np.uint32(int("300C000A", 16))
    ReferencedBrachyApplicationSetupNumber = np.uint32(int("300C000C", 16))
    ReferencedSourceNumber = np.uint32(int("300C000E", 16))
    ReferencedFractionGroupSequence = np.uint32(int("300C0020", 16))
    ReferencedFractionGroupNumber = np.uint32(int("300C0022", 16))
    ReferencedVerificationImageSequence = np.uint32(int("300C0040", 16))
    ReferencedReferenceImageSequence = np.uint32(int("300C0042", 16))
    ReferencedDoseReferenceSequence = np.uint32(int("300C0050", 16))
    ReferencedDoseReferenceNumber = np.uint32(int("300C0051", 16))
    BrachyReferencedDoseReferenceSequence = np.uint32(int("300C0055", 16))
    ReferencedStructureSetSequence = np.uint32(int("300C0060", 16))
    ReferencedPatientSetupNumber = np.uint32(int("300C006A", 16))
    ReferencedDoseSequence = np.uint32(int("300C0080", 16))
    ReferencedToleranceTableNumber = np.uint32(int("300C00A0", 16))
    ReferencedBolusSequence = np.uint32(int("300C00B0", 16))
    ReferencedWedgeNumber = np.uint32(int("300C00C0", 16))
    ReferencedCompensatorNumber = np.uint32(int("300C00D0", 16))
    ReferencedBlockNumber = np.uint32(int("300C00E0", 16))
    ReferencedControlPointIndex = np.uint32(int("300C00F0", 16))
    ReferencedControlPointSequence = np.uint32(int("300C00F2", 16))
    ReferencedStartControlPointIndex = np.uint32(int("300C00F4", 16))
    ReferencedStopControlPointIndex = np.uint32(int("300C00F6", 16))
    ReferencedRangeShifterNumber = np.uint32(int("300C0100", 16))
    ReferencedLateralSpreadingDeviceNumber = np.uint32(int("300C0102", 16))
    ReferencedRangeModulatorNumber = np.uint32(int("300C0104", 16))
    OmittedBeamTaskSequence = np.uint32(int("300C0111", 16))
    ReasonforOmission = np.uint32(int("300C0112", 16))
    ReasonforOmissionDescription = np.uint32(int("300C0113", 16))
    ApprovalStatus = np.uint32(int("300E0002", 16))
    ReviewDate = np.uint32(int("300E0004", 16))
    ReviewTime = np.uint32(int("300E0005", 16))
    ReviewerName = np.uint32(int("300E0008", 16))
    Arbitrary = np.uint32(int("40000010", 16))
    TextComments = np.uint32(int("40004000", 16))
    ResultsID = np.uint32(int("40080040", 16))
    ResultsIDIssuer = np.uint32(int("40080042", 16))
    ReferencedInterpretationSequence = np.uint32(int("40080050", 16))
    ReportProductionStatusTrial = np.uint32(int("400800FF", 16))
    InterpretationRecordedDate = np.uint32(int("40080100", 16))
    InterpretationRecordedTime = np.uint32(int("40080101", 16))
    InterpretationRecorder = np.uint32(int("40080102", 16))
    ReferencetoRecordedSound = np.uint32(int("40080103", 16))
    InterpretationTranscriptionDate = np.uint32(int("40080108", 16))
    InterpretationTranscriptionTime = np.uint32(int("40080109", 16))
    InterpretationTranscriber = np.uint32(int("4008010A", 16))
    InterpretationText = np.uint32(int("4008010B", 16))
    InterpretationAuthor = np.uint32(int("4008010C", 16))
    InterpretationApproverSequence = np.uint32(int("40080111", 16))
    InterpretationApprovalDate = np.uint32(int("40080112", 16))
    InterpretationApprovalTime = np.uint32(int("40080113", 16))
    PhysicianApprovingInterpretation = np.uint32(int("40080114", 16))
    InterpretationDiagnosisDescription = np.uint32(int("40080115", 16))
    InterpretationDiagnosisCodeSequence = np.uint32(int("40080117", 16))
    ResultsDistributionListSequence = np.uint32(int("40080118", 16))
    DistributionName = np.uint32(int("40080119", 16))
    DistributionAddress = np.uint32(int("4008011A", 16))
    InterpretationID = np.uint32(int("40080200", 16))
    InterpretationIDIssuer = np.uint32(int("40080202", 16))
    InterpretationTypeID = np.uint32(int("40080210", 16))
    InterpretationStatusID = np.uint32(int("40080212", 16))
    Impressions = np.uint32(int("40080300", 16))
    ResultsComments = np.uint32(int("40084000", 16))
    LowEnergyDetectors = np.uint32(int("40100001", 16))
    HighEnergyDetectors = np.uint32(int("40100002", 16))
    DetectorGeometrySequence = np.uint32(int("40100004", 16))
    ThreatROIVoxelSequence = np.uint32(int("40101001", 16))
    ThreatROIBase = np.uint32(int("40101004", 16))
    ThreatROIExtents = np.uint32(int("40101005", 16))
    ThreatROIBitmap = np.uint32(int("40101006", 16))
    RouteSegmentID = np.uint32(int("40101007", 16))
    GantryType = np.uint32(int("40101008", 16))
    OOIOwnerType = np.uint32(int("40101009", 16))
    RouteSegmentSequence = np.uint32(int("4010100A", 16))
    PotentialThreatObjectID = np.uint32(int("40101010", 16))
    ThreatSequence = np.uint32(int("40101011", 16))
    ThreatCategory = np.uint32(int("40101012", 16))
    ThreatCategoryDescription = np.uint32(int("40101013", 16))
    ATDAbilityAssessment = np.uint32(int("40101014", 16))
    ATDAssessmentFlag = np.uint32(int("40101015", 16))
    ATDAssessmentProbability = np.uint32(int("40101016", 16))
    Mass = np.uint32(int("40101017", 16))
    Density = np.uint32(int("40101018", 16))
    ZEffective = np.uint32(int("40101019", 16))
    BoardingPassID = np.uint32(int("4010101A", 16))
    CenterofMass = np.uint32(int("4010101B", 16))
    CenterofPTO = np.uint32(int("4010101C", 16))
    BoundingPolygon = np.uint32(int("4010101D", 16))
    RouteSegmentStartLocationID = np.uint32(int("4010101E", 16))
    RouteSegmentEndLocationID = np.uint32(int("4010101F", 16))
    RouteSegmentLocationIDType = np.uint32(int("40101020", 16))
    AbortReason = np.uint32(int("40101021", 16))
    VolumeofPTO = np.uint32(int("40101023", 16))
    AbortFlag = np.uint32(int("40101024", 16))
    RouteSegmentStartTime = np.uint32(int("40101025", 16))
    RouteSegmentEndTime = np.uint32(int("40101026", 16))
    TDRType = np.uint32(int("40101027", 16))
    InternationalRouteSegment = np.uint32(int("40101028", 16))
    ThreatDetectionAlgorithmandVersion = np.uint32(int("40101029", 16))
    AssignedLocation = np.uint32(int("4010102A", 16))
    AlarmDecisionTime = np.uint32(int("4010102B", 16))
    AlarmDecision = np.uint32(int("40101031", 16))
    NumberofTotalObjects = np.uint32(int("40101033", 16))
    NumberofAlarmObjects = np.uint32(int("40101034", 16))
    PTORepresentationSequence = np.uint32(int("40101037", 16))
    ATDAssessmentSequence = np.uint32(int("40101038", 16))
    TIPType = np.uint32(int("40101039", 16))
    Version = np.uint32(int("4010103A", 16))
    OOIOwnerCreationTime = np.uint32(int("40101041", 16))
    OOIType = np.uint32(int("40101042", 16))
    OOISize = np.uint32(int("40101043", 16))
    AcquisitionStatus = np.uint32(int("40101044", 16))
    BasisMaterialsCodeSequence = np.uint32(int("40101045", 16))
    PhantomType = np.uint32(int("40101046", 16))
    OOIOwnerSequence = np.uint32(int("40101047", 16))
    ScanType = np.uint32(int("40101048", 16))
    ItineraryID = np.uint32(int("40101051", 16))
    ItineraryIDType = np.uint32(int("40101052", 16))
    ItineraryIDAssigningAuthority = np.uint32(int("40101053", 16))
    RouteID = np.uint32(int("40101054", 16))
    RouteIDAssigningAuthority = np.uint32(int("40101055", 16))
    InboundArrivalType = np.uint32(int("40101056", 16))
    CarrierID = np.uint32(int("40101058", 16))
    CarrierIDAssigningAuthority = np.uint32(int("40101059", 16))
    SourceOrientation = np.uint32(int("40101060", 16))
    SourcePosition = np.uint32(int("40101061", 16))
    BeltHeight = np.uint32(int("40101062", 16))
    AlgorithmRoutingCodeSequence = np.uint32(int("40101064", 16))
    TransportClassification = np.uint32(int("40101067", 16))
    OOITypeDescriptor = np.uint32(int("40101068", 16))
    TotalProcessingTime = np.uint32(int("40101069", 16))
    DetectorCalibrationData = np.uint32(int("4010106C", 16))
    AdditionalScreeningPerformed = np.uint32(int("4010106D", 16))
    AdditionalInspectionSelectionCriteria = np.uint32(int("4010106E", 16))
    AdditionalInspectionMethodSequence = np.uint32(int("4010106F", 16))
    AITDeviceType = np.uint32(int("40101070", 16))
    QRMeasurementsSequence = np.uint32(int("40101071", 16))
    TargetMaterialSequence = np.uint32(int("40101072", 16))
    SNRThreshold = np.uint32(int("40101073", 16))
    ImageScaleRepresentation = np.uint32(int("40101075", 16))
    ReferencedPTOSequence = np.uint32(int("40101076", 16))
    ReferencedTDRInstanceSequence = np.uint32(int("40101077", 16))
    PTOLocationDescription = np.uint32(int("40101078", 16))
    AnomalyLocatorIndicatorSequence = np.uint32(int("40101079", 16))
    AnomalyLocatorIndicator = np.uint32(int("4010107A", 16))
    PTORegionSequence = np.uint32(int("4010107B", 16))
    InspectionSelectionCriteria = np.uint32(int("4010107C", 16))
    SecondaryInspectionMethodSequence = np.uint32(int("4010107D", 16))
    PRCStoRCSOrientation = np.uint32(int("4010107E", 16))
    MACParametersSequence = np.uint32(int("4FFE0001", 16))
    SharedFunctionalGroupsSequence = np.uint32(int("52009229", 16))
    PerframeFunctionalGroupsSequence = np.uint32(int("52009230", 16))
    WaveformSequence = np.uint32(int("54000100", 16))
    ChannelMinimumValue = np.uint32(int("54000110", 16))
    ChannelMaximumValue = np.uint32(int("54000112", 16))
    WaveformBitsAllocated = np.uint32(int("54001004", 16))
    WaveformSampleInterpretation = np.uint32(int("54001006", 16))
    WaveformPaddingValue = np.uint32(int("5400100A", 16))
    WaveformData = np.uint32(int("54001010", 16))
    FirstOrderPhaseCorrectionAngle = np.uint32(int("56000010", 16))
    SpectroscopyData = np.uint32(int("56000020", 16))
    FloatPixelData = np.uint32(int("7FE00008", 16))
    DoubleFloatPixelData = np.uint32(int("7FE00009", 16))
    PixelData = np.uint32(int("7FE00010", 16))
    CoefficientsSDVN = np.uint32(int("7FE00020", 16))
    CoefficientsSDHN = np.uint32(int("7FE00030", 16))
    CoefficientsSDDN = np.uint32(int("7FE00040", 16))
    DigitalSignaturesSequence = np.uint32(int("FFFAFFFA", 16))
    DataSetTrailingPadding = np.uint32(int("FFFCFFFC", 16))
    Item = np.uint32(int("FFFEE000", 16))
    ItemDelimitationItem = np.uint32(int("FFFEE00D", 16))
    SequenceDelimitationItem = np.uint32(int("FFFEE0DD", 16))
