import * as events from '@aws-cdk/aws-events';
import * as iam from '@aws-cdk/aws-iam';
import * as kms from '@aws-cdk/aws-kms';
import { IResource, RemovalPolicy, Resource } from '@aws-cdk/core';
import { Construct } from 'constructs';
import { BucketPolicy } from './bucket-policy';
import { IBucketNotificationDestination } from './destination';
import { LifecycleRule } from './rule';
/**
 * @stability stable
 */
export interface IBucket extends IResource {
    /**
     * The ARN of the bucket.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketArn: string;
    /**
     * The name of the bucket.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketName: string;
    /**
     * The URL of the static website.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketWebsiteUrl: string;
    /**
     * The Domain name of the static website.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketWebsiteDomainName: string;
    /**
     * The IPv4 DNS name of the specified bucket.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketDomainName: string;
    /**
     * The IPv6 DNS name of the specified bucket.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketDualStackDomainName: string;
    /**
     * The regional domain name of the specified bucket.
     *
     * @stability stable
     * @attribute true
     */
    readonly bucketRegionalDomainName: string;
    /**
     * If this bucket has been configured for static website hosting.
     *
     * @stability stable
     */
    readonly isWebsite?: boolean;
    /**
     * Optional KMS encryption key associated with this bucket.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     *
     * @stability stable
     */
    policy?: BucketPolicy;
    /**
     * Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @stability stable
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * The https URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key?: string): string;
    /**
     * The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key?: string, options?: VirtualHostedStyleUrlOptions): string;
    /**
     * The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key?: string): string;
    /**
     * Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * @stability stable
     */
    arnForObjects(keyPattern: string): string;
    /**
     * Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @returns The `iam.PolicyStatement` object, which can be used to apply e.g. conditions.
     * @stability stable
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    /**
     * Defines a CloudWatch event that triggers when something happens to this bucket.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
}
/**
 * A reference to a bucket.
 *
 * The easiest way to instantiate is to call
 * `bucket.export()`. Then, the consumer can use `Bucket.import(this, ref)` and
 * get a `Bucket`.
 *
 * @stability stable
 */
export interface BucketAttributes {
    /**
     * The ARN of the bucket.
     *
     * At least one of bucketArn or bucketName must be
     * defined in order to initialize a bucket ref.
     *
     * @stability stable
     */
    readonly bucketArn?: string;
    /**
     * The name of the bucket.
     *
     * If the underlying value of ARN is a string, the
     * name will be parsed from the ARN. Otherwise, the name is optional, but
     * some features that require the bucket name such as auto-creating a bucket
     * policy, won't work.
     *
     * @stability stable
     */
    readonly bucketName?: string;
    /**
     * The domain name of the bucket.
     *
     * @default Inferred from bucket name
     * @stability stable
     */
    readonly bucketDomainName?: string;
    /**
     * The website URL of the bucket (if static web hosting is enabled).
     *
     * @default Inferred from bucket name
     * @stability stable
     */
    readonly bucketWebsiteUrl?: string;
    /**
     * The regional domain name of the specified bucket.
     *
     * @stability stable
     */
    readonly bucketRegionalDomainName?: string;
    /**
     * The IPv6 DNS name of the specified bucket.
     *
     * @stability stable
     */
    readonly bucketDualStackDomainName?: string;
    /**
     * The format of the website URL of the bucket.
     *
     * This should be true for
     * regions launched since 2014.
     *
     * @default false
     * @stability stable
     */
    readonly bucketWebsiteNewUrlFormat?: boolean;
    /**
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     *
     * @default false
     * @stability stable
     */
    readonly isWebsite?: boolean;
    /**
     * The account this existing bucket belongs to.
     *
     * @default - it's assumed the bucket belongs to the same account as the scope it's being imported into
     * @stability stable
     */
    readonly account?: string;
    /**
     * The region this existing bucket is in.
     *
     * @default - it's assumed the bucket is in the same region as the scope it's being imported into
     * @stability stable
     */
    readonly region?: string;
}
/**
 * Represents an S3 Bucket.
 *
 * Buckets can be either defined within this stack:
 *
 *   new Bucket(this, 'MyBucket', { props });
 *
 * Or imported from an existing bucket:
 *
 *   Bucket.import(this, 'MyImportedBucket', { bucketArn: ... });
 *
 * You can also export a bucket and import it into another stack:
 *
 *   const ref = myBucket.export();
 *   Bucket.import(this, 'MyImportedBucket', ref);
 *
 */
declare abstract class BucketBase extends Resource implements IBucket {
    abstract readonly bucketArn: string;
    abstract readonly bucketName: string;
    abstract readonly bucketDomainName: string;
    abstract readonly bucketWebsiteUrl: string;
    abstract readonly bucketWebsiteDomainName: string;
    abstract readonly bucketRegionalDomainName: string;
    abstract readonly bucketDualStackDomainName: string;
    /**
     * Optional KMS encryption key associated with this bucket.
     */
    abstract readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     */
    abstract readonly isWebsite?: boolean;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     */
    abstract policy?: BucketPolicy;
    /**
     * Indicates if a bucket resource policy should automatically created upon
     * the first call to `addToResourcePolicy`.
     */
    protected abstract autoCreatePolicy: boolean;
    /**
     * Whether to disallow public access
     */
    protected abstract disallowPublicAccess?: boolean;
    /**
     * Define a CloudWatch event that triggers when something happens to this repository.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailEvent(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object is uploaded to the specified paths (keys) in this bucket using the PutObject API call.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using `onCloudTrailWriteObject` may be preferable.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailPutObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Defines an AWS CloudWatch event that triggers when an object at the specified paths (keys) in this bucket are written to.
     *
     * This includes
     * the events PutObject, CopyObject, and CompleteMultipartUpload.
     *
     * Note that some tools like `aws s3 cp` will automatically use either
     * PutObject or the multipart upload API depending on the file size,
     * so using this method may be preferable to `onCloudTrailPutObject`.
     *
     * Requires that there exists at least one CloudTrail Trail in your account
     * that captures the event. This method will not create the Trail.
     *
     * @param id The id of the rule.
     * @param options Options for adding the rule.
     * @stability stable
     */
    onCloudTrailWriteObject(id: string, options?: OnCloudTrailBucketEventOptions): events.Rule;
    /**
     * Adds a statement to the resource policy for a principal (i.e. account/role/service) to perform actions on this bucket and/or it's contents. Use `bucketArn` and `arnForObjects(keys)` to obtain ARNs for this bucket or objects.
     *
     * @stability stable
     */
    addToResourcePolicy(permission: iam.PolicyStatement): iam.AddToResourcePolicyResult;
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected validate(): string[];
    /**
     * The https URL of an S3 object.
     *
     * Specify `regional: false` at the options
     * for non-regional URLs. For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://s3.cn-north-1.amazonaws.com.cn/china-bucket/mykey
     */
    urlForObject(key?: string): string;
    /**
     * The virtual hosted-style URL of an S3 object.
     *
     * Specify `regional: false` at
     * the options for non-regional URL. For example:
     *
     * @param key The S3 key of the object.
     * @param options Options for generating URL.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * https://china-bucket.s3.cn-north-1.amazonaws.com.cn/mykey
     */
    virtualHostedUrlForObject(key?: string, options?: VirtualHostedStyleUrlOptions): string;
    /**
     * The S3 URL of an S3 object.
     *
     * For example:
     *
     * @param key The S3 key of the object.
     * @returns an ObjectS3Url token
     * @stability stable
     * @example
     *
     * s3://bucket/key
     */
    s3UrlForObject(key?: string): string;
    /**
     * Returns an ARN that represents all objects within the bucket that match the key pattern specified.
     *
     * To represent all keys, specify ``"*"``.
     *
     * If you need to specify a keyPattern with multiple components, concatenate them into a single string, e.g.:
     *
     *    arnForObjects(`home/${team}/${user}/*`)
     *
     * @stability stable
     */
    arnForObjects(keyPattern: string): string;
    /**
     * Grant read permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If encryption is used, permission to use the key to decrypt the contents
     * of the bucket will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantRead(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grant write permissions to this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:PutObject* and s3:Abort* permissions for this bucket to an IAM principal.
     *
     * If encryption is used, permission to use the key to encrypt the contents
     * of written files will also be granted to the same principal.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantPut(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants s3:DeleteObject* permission to an IAM pricipal for objects in this bucket.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantDelete(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Grants read/write permissions for this bucket and it's contents to an IAM principal (Role/Group/User).
     *
     * If an encryption key is used, permission to use the key for
     * encrypt/decrypt will also be granted.
     *
     * @param identity The principal.
     * @param objectsKeyPattern Restrict the permission to a certain key pattern (default '*').
     * @stability stable
     */
    grantReadWrite(identity: iam.IGrantable, objectsKeyPattern?: any): iam.Grant;
    /**
     * Allows unrestricted access to objects from this bucket.
     *
     * IMPORTANT: This permission allows anyone to perform actions on S3 objects
     * in this bucket, which is useful for when you configure your bucket as a
     * website and want everyone to be able to read objects in the bucket without
     * needing to authenticate.
     *
     * Without arguments, this method will grant read ("s3:GetObject") access to
     * all objects ("*") in the bucket.
     *
     * The method returns the `iam.Grant` object, which can then be modified
     * as needed. For example, you can add a condition that will restrict access only
     * to an IPv4 range like this:
     *
     *      const grant = bucket.grantPublicAccess();
     *      grant.resourceStatement!.addCondition(‘IpAddress’, { “aws:SourceIp”: “54.240.143.0/24” });
     *
     * @param keyPrefix the prefix of S3 object keys (e.g. `home/*`). Default is "*".
     * @param allowedActions the set of S3 actions to allow.
     * @stability stable
     */
    grantPublicAccess(keyPrefix?: string, ...allowedActions: string[]): iam.Grant;
    private urlJoin;
    private grant;
}
/**
 * @stability stable
 */
export interface BlockPublicAccessOptions {
    /**
     * Whether to block public ACLs.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @stability stable
     */
    readonly blockPublicAcls?: boolean;
    /**
     * Whether to block public policy.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @stability stable
     */
    readonly blockPublicPolicy?: boolean;
    /**
     * Whether to ignore public ACLs.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @stability stable
     */
    readonly ignorePublicAcls?: boolean;
    /**
     * Whether to restrict public access.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html#access-control-block-public-access-options
     * @stability stable
     */
    readonly restrictPublicBuckets?: boolean;
}
/**
 * @stability stable
 */
export declare class BlockPublicAccess {
    /**
     * @stability stable
     */
    static readonly BLOCK_ALL: BlockPublicAccess;
    /**
     * @stability stable
     */
    static readonly BLOCK_ACLS: BlockPublicAccess;
    /**
     * @stability stable
     */
    blockPublicAcls: boolean | undefined;
    /**
     * @stability stable
     */
    blockPublicPolicy: boolean | undefined;
    /**
     * @stability stable
     */
    ignorePublicAcls: boolean | undefined;
    /**
     * @stability stable
     */
    restrictPublicBuckets: boolean | undefined;
    /**
     * @stability stable
     */
    constructor(options: BlockPublicAccessOptions);
}
/**
 * Specifies a metrics configuration for the CloudWatch request metrics from an Amazon S3 bucket.
 *
 * @stability stable
 */
export interface BucketMetrics {
    /**
     * The ID used to identify the metrics configuration.
     *
     * @stability stable
     */
    readonly id: string;
    /**
     * The prefix that an object must have to be included in the metrics results.
     *
     * @stability stable
     */
    readonly prefix?: string;
    /**
     * Specifies a list of tag filters to use as a metrics configuration filter.
     *
     * The metrics configuration includes only objects that meet the filter's criteria.
     *
     * @stability stable
     */
    readonly tagFilters?: {
        [tag: string]: any;
    };
}
/**
 * All http request methods.
 *
 * @stability stable
 */
export declare enum HttpMethods {
    /**
     * The GET method requests a representation of the specified resource.
     *
     * @stability stable
     */
    GET = "GET",
    /**
     * The PUT method replaces all current representations of the target resource with the request payload.
     *
     * @stability stable
     */
    PUT = "PUT",
    /**
     * The HEAD method asks for a response identical to that of a GET request, but without the response body.
     *
     * @stability stable
     */
    HEAD = "HEAD",
    /**
     * The POST method is used to submit an entity to the specified resource, often causing a change in state or side effects on the server.
     *
     * @stability stable
     */
    POST = "POST",
    /**
     * The DELETE method deletes the specified resource.
     *
     * @stability stable
     */
    DELETE = "DELETE"
}
/**
 * Specifies a cross-origin access rule for an Amazon S3 bucket.
 *
 * @stability stable
 */
export interface CorsRule {
    /**
     * A unique identifier for this rule.
     *
     * @default - No id specified.
     * @stability stable
     */
    readonly id?: string;
    /**
     * The time in seconds that your browser is to cache the preflight response for the specified resource.
     *
     * @default - No caching.
     * @stability stable
     */
    readonly maxAge?: number;
    /**
     * Headers that are specified in the Access-Control-Request-Headers header.
     *
     * @default - No headers allowed.
     * @stability stable
     */
    readonly allowedHeaders?: string[];
    /**
     * An HTTP method that you allow the origin to execute.
     *
     * @stability stable
     */
    readonly allowedMethods: HttpMethods[];
    /**
     * One or more origins you want customers to be able to access the bucket from.
     *
     * @stability stable
     */
    readonly allowedOrigins: string[];
    /**
     * One or more headers in the response that you want customers to be able to access from their applications.
     *
     * @default - No headers exposed.
     * @stability stable
     */
    readonly exposedHeaders?: string[];
}
/**
 * All http request methods.
 *
 * @stability stable
 */
export declare enum RedirectProtocol {
    /**
     * @stability stable
     */
    HTTP = "http",
    /**
     * @stability stable
     */
    HTTPS = "https"
}
/**
 * Specifies a redirect behavior of all requests to a website endpoint of a bucket.
 *
 * @stability stable
 */
export interface RedirectTarget {
    /**
     * Name of the host where requests are redirected.
     *
     * @stability stable
     */
    readonly hostName: string;
    /**
     * Protocol to use when redirecting requests.
     *
     * @default - The protocol used in the original request.
     * @stability stable
     */
    readonly protocol?: RedirectProtocol;
}
/**
 * All supported inventory list formats.
 *
 * @stability stable
 */
export declare enum InventoryFormat {
    /**
     * Generate the inventory list as CSV.
     *
     * @stability stable
     */
    CSV = "CSV",
    /**
     * Generate the inventory list as Parquet.
     *
     * @stability stable
     */
    PARQUET = "Parquet",
    /**
     * Generate the inventory list as Parquet.
     *
     * @stability stable
     */
    ORC = "ORC"
}
/**
 * All supported inventory frequencies.
 *
 * @stability stable
 */
export declare enum InventoryFrequency {
    /**
     * A report is generated every day.
     *
     * @stability stable
     */
    DAILY = "Daily",
    /**
     * A report is generated every Sunday (UTC timezone) after the initial report.
     *
     * @stability stable
     */
    WEEKLY = "Weekly"
}
/**
 * Inventory version support.
 *
 * @stability stable
 */
export declare enum InventoryObjectVersion {
    /**
     * Includes all versions of each object in the report.
     *
     * @stability stable
     */
    ALL = "All",
    /**
     * Includes only the current version of each object in the report.
     *
     * @stability stable
     */
    CURRENT = "Current"
}
/**
 * The destination of the inventory.
 *
 * @stability stable
 */
export interface InventoryDestination {
    /**
     * Bucket where all inventories will be saved in.
     *
     * @stability stable
     */
    readonly bucket: IBucket;
    /**
     * The prefix to be used when saving the inventory.
     *
     * @default - No prefix.
     * @stability stable
     */
    readonly prefix?: string;
    /**
     * The account ID that owns the destination S3 bucket.
     *
     * If no account ID is provided, the owner is not validated before exporting data.
     * It's recommended to set an account ID to prevent problems if the destination bucket ownership changes.
     *
     * @default - No account ID.
     * @stability stable
     */
    readonly bucketOwner?: string;
}
/**
 * Specifies the inventory configuration of an S3 Bucket.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html
 * @stability stable
 */
export interface Inventory {
    /**
     * The destination of the inventory.
     *
     * @stability stable
     */
    readonly destination: InventoryDestination;
    /**
     * The inventory will only include objects that meet the prefix filter criteria.
     *
     * @default - No objects prefix
     * @stability stable
     */
    readonly objectsPrefix?: string;
    /**
     * The format of the inventory.
     *
     * @default InventoryFormat.CSV
     * @stability stable
     */
    readonly format?: InventoryFormat;
    /**
     * Whether the inventory is enabled or disabled.
     *
     * @default true
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * The inventory configuration ID.
     *
     * @default - generated ID.
     * @stability stable
     */
    readonly inventoryId?: string;
    /**
     * Frequency at which the inventory should be generated.
     *
     * @default InventoryFrequency.WEEKLY
     * @stability stable
     */
    readonly frequency?: InventoryFrequency;
    /**
     * If the inventory should contain all the object versions or only the current one.
     *
     * @default InventoryObjectVersion.ALL
     * @stability stable
     */
    readonly includeObjectVersions?: InventoryObjectVersion;
    /**
     * A list of optional fields to be included in the inventory result.
     *
     * @default - No optional fields.
     * @stability stable
     */
    readonly optionalFields?: string[];
}
/**
 * @stability stable
 */
export interface BucketProps {
    /**
     * The kind of server-side encryption to apply to this bucket.
     *
     * If you choose KMS, you can specify a KMS key via `encryptionKey`. If
     * encryption key is not specified, a key will automatically be created.
     *
     * @default - `Kms` if `encryptionKey` is specified, or `Unencrypted` otherwise.
     * @stability stable
     */
    readonly encryption?: BucketEncryption;
    /**
     * External KMS key to use for bucket encryption.
     *
     * The 'encryption' property must be either not specified or set to "Kms".
     * An error will be emitted if encryption is set to "Unencrypted" or
     * "Managed".
     *
     * @default - If encryption is set to "Kms" and this property is undefined,
     * a new KMS key will be created and associated with this bucket.
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * Physical name of this bucket.
     *
     * @default - Assigned by CloudFormation (recommended).
     * @stability stable
     */
    readonly bucketName?: string;
    /**
     * Policy to apply when the bucket is removed from this stack.
     *
     * @default - The bucket will be orphaned.
     * @stability stable
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * Whether this bucket should have versioning turned on or not.
     *
     * @default false
     * @stability stable
     */
    readonly versioned?: boolean;
    /**
     * Rules that define how Amazon S3 manages objects during their lifetime.
     *
     * @default - No lifecycle rules.
     * @stability stable
     */
    readonly lifecycleRules?: LifecycleRule[];
    /**
     * The name of the index document (e.g. "index.html") for the website. Enables static website hosting for this bucket.
     *
     * @default - No index document.
     * @stability stable
     */
    readonly websiteIndexDocument?: string;
    /**
     * The name of the error document (e.g. "404.html") for the website. `websiteIndexDocument` must also be set if this is set.
     *
     * @default - No error document.
     * @stability stable
     */
    readonly websiteErrorDocument?: string;
    /**
     * Specifies the redirect behavior of all requests to a website endpoint of a bucket.
     *
     * If you specify this property, you can't specify "websiteIndexDocument", "websiteErrorDocument" nor , "websiteRoutingRules".
     *
     * @default - No redirection.
     * @stability stable
     */
    readonly websiteRedirect?: RedirectTarget;
    /**
     * Rules that define when a redirect is applied and the redirect behavior.
     *
     * @default - No redirection rules.
     * @stability stable
     */
    readonly websiteRoutingRules?: RoutingRule[];
    /**
     * Specifies a canned ACL that grants predefined permissions to the bucket.
     *
     * @default BucketAccessControl.PRIVATE
     * @stability stable
     */
    readonly accessControl?: BucketAccessControl;
    /**
     * Grants public read access to all objects in the bucket.
     *
     * Similar to calling `bucket.grantPublicAccess()`
     *
     * @default false
     * @stability stable
     */
    readonly publicReadAccess?: boolean;
    /**
     * The block public access configuration of this bucket.
     *
     * @default false New buckets and objects don't allow public access, but users can modify bucket
     * policies or object permissions to allow public access.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/access-control-block-public-access.html
     * @stability stable
     */
    readonly blockPublicAccess?: BlockPublicAccess;
    /**
     * The metrics configuration of this bucket.
     *
     * @default - No metrics configuration.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-metricsconfiguration.html
     * @stability stable
     */
    readonly metrics?: BucketMetrics[];
    /**
     * The CORS configuration of this bucket.
     *
     * @default - No CORS configuration.
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket-cors.html
     * @stability stable
     */
    readonly cors?: CorsRule[];
    /**
     * Destination bucket for the server access logs.
     *
     * @default - If "serverAccessLogsPrefix" undefined - access logs disabled, otherwise - log to current bucket.
     * @stability stable
     */
    readonly serverAccessLogsBucket?: IBucket;
    /**
     * Optional log file prefix to use for the bucket's access logs.
     *
     * If defined without "serverAccessLogsBucket", enables access logs to current bucket with this prefix.
     *
     * @default - No log file prefix
     * @stability stable
     */
    readonly serverAccessLogsPrefix?: string;
    /**
     * The inventory configuration of the bucket.
     *
     * @default - No inventory configuration
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/storage-inventory.html
     * @stability stable
     */
    readonly inventories?: Inventory[];
}
/**
 * An S3 bucket with associated policy objects.
 *
 * This bucket does not yet have all features that exposed by the underlying
 * BucketResource.
 *
 * @stability stable
 */
export declare class Bucket extends BucketBase {
    /**
     * @stability stable
     */
    static fromBucketArn(scope: Construct, id: string, bucketArn: string): IBucket;
    /**
     * @stability stable
     */
    static fromBucketName(scope: Construct, id: string, bucketName: string): IBucket;
    /**
     * Creates a Bucket construct that represents an external bucket.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `BucketAttributes` object.
     * @stability stable
     */
    static fromBucketAttributes(scope: Construct, id: string, attrs: BucketAttributes): IBucket;
    /**
     * The ARN of the bucket.
     *
     * @stability stable
     */
    readonly bucketArn: string;
    /**
     * The name of the bucket.
     *
     * @stability stable
     */
    readonly bucketName: string;
    /**
     * The IPv4 DNS name of the specified bucket.
     *
     * @stability stable
     */
    readonly bucketDomainName: string;
    /**
     * The URL of the static website.
     *
     * @stability stable
     */
    readonly bucketWebsiteUrl: string;
    /**
     * The Domain name of the static website.
     *
     * @stability stable
     */
    readonly bucketWebsiteDomainName: string;
    /**
     * The IPv6 DNS name of the specified bucket.
     *
     * @stability stable
     */
    readonly bucketDualStackDomainName: string;
    /**
     * The regional domain name of the specified bucket.
     *
     * @stability stable
     */
    readonly bucketRegionalDomainName: string;
    /**
     * Optional KMS encryption key associated with this bucket.
     *
     * @stability stable
     */
    readonly encryptionKey?: kms.IKey;
    /**
     * If this bucket has been configured for static website hosting.
     *
     * @stability stable
     */
    readonly isWebsite?: boolean;
    /**
     * The resource policy associated with this bucket.
     *
     * If `autoCreatePolicy` is true, a `BucketPolicy` will be created upon the
     * first call to addToResourcePolicy(s).
     *
     * @stability stable
     */
    policy?: BucketPolicy;
    /**
     * Indicates if a bucket resource policy should automatically created upon the first call to `addToResourcePolicy`.
     *
     * @stability stable
     */
    protected autoCreatePolicy: boolean;
    /**
     * Whether to disallow public access.
     *
     * @stability stable
     */
    protected disallowPublicAccess?: boolean;
    private accessControl?;
    private readonly lifecycleRules;
    private readonly versioned?;
    private readonly notifications;
    private readonly metrics;
    private readonly cors;
    private readonly inventories;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: BucketProps);
    /**
     * Add a lifecycle rule to the bucket.
     *
     * @param rule The rule to add.
     * @stability stable
     */
    addLifecycleRule(rule: LifecycleRule): void;
    /**
     * Adds a metrics configuration for the CloudWatch request metrics from the bucket.
     *
     * @param metric The metric configuration to add.
     * @stability stable
     */
    addMetric(metric: BucketMetrics): void;
    /**
     * Adds a cross-origin access configuration for objects in an Amazon S3 bucket.
     *
     * @param rule The CORS configuration rule to add.
     * @stability stable
     */
    addCorsRule(rule: CorsRule): void;
    /**
     * Adds a bucket notification event destination.
     *
     * @param event The event to trigger the notification.
     * @param dest The notification destination (Lambda, SNS Topic or SQS Queue).
     * @param filters S3 object key filter rules to determine which objects trigger this event.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html
     * @stability stable
     * @example
     *
     *    bucket.addEventNotification(EventType.OnObjectCreated, myLambda, 'home/myusername/*')
     */
    addEventNotification(event: EventType, dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * Subscribes a destination to receive notifications when an object is created in the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectCreated)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectCreatedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * Subscribes a destination to receive notifications when an object is removed from the bucket.
     *
     * This is identical to calling
     * `onEvent(EventType.ObjectRemoved)`.
     *
     * @param dest The notification destination (see onEvent).
     * @param filters Filters (see onEvent).
     * @stability stable
     */
    addObjectRemovedNotification(dest: IBucketNotificationDestination, ...filters: NotificationKeyFilter[]): void;
    /**
     * Add an inventory configuration.
     *
     * @param inventory configuration to add.
     * @stability stable
     */
    addInventory(inventory: Inventory): void;
    private validateBucketName;
    /**
     * Set up key properties and return the Bucket encryption property from the
     * user's configuration.
     */
    private parseEncryption;
    /**
     * Parse the lifecycle configuration out of the uucket props
     * @param props Par
     */
    private parseLifecycleConfiguration;
    private parseServerAccessLogs;
    private parseMetricConfiguration;
    private parseCorsConfiguration;
    private parseTagFilters;
    private renderWebsiteConfiguration;
    /**
     * Allows the LogDelivery group to write, fails if ACL was set differently.
     *
     * @see
     * https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html#canned-acl
     */
    private allowLogDelivery;
    private parseInventoryConfiguration;
}
/**
 * What kind of server-side encryption to apply to this bucket.
 *
 * @stability stable
 */
export declare enum BucketEncryption {
    /**
     * Objects in the bucket are not encrypted.
     *
     * @stability stable
     */
    UNENCRYPTED = "NONE",
    /**
     * Server-side KMS encryption with a master key managed by KMS.
     *
     * @stability stable
     */
    KMS_MANAGED = "MANAGED",
    /**
     * Server-side encryption with a master key managed by S3.
     *
     * @stability stable
     */
    S3_MANAGED = "S3MANAGED",
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * @stability stable
     */
    KMS = "KMS"
}
/**
 * Notification event types.
 *
 * @stability stable
 */
export declare enum EventType {
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @stability stable
     */
    OBJECT_CREATED = "s3:ObjectCreated:*",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @stability stable
     */
    OBJECT_CREATED_PUT = "s3:ObjectCreated:Put",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @stability stable
     */
    OBJECT_CREATED_POST = "s3:ObjectCreated:Post",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @stability stable
     */
    OBJECT_CREATED_COPY = "s3:ObjectCreated:Copy",
    /**
     * Amazon S3 APIs such as PUT, POST, and COPY can create an object.
     *
     * Using
     * these event types, you can enable notification when an object is created
     * using a specific API, or you can use the s3:ObjectCreated:* event type to
     * request notification regardless of the API that was used to create an
     * object.
     *
     * @stability stable
     */
    OBJECT_CREATED_COMPLETE_MULTIPART_UPLOAD = "s3:ObjectCreated:CompleteMultipartUpload",
    /**
     * By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @stability stable
     */
    OBJECT_REMOVED = "s3:ObjectRemoved:*",
    /**
     * By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @stability stable
     */
    OBJECT_REMOVED_DELETE = "s3:ObjectRemoved:Delete",
    /**
     * By using the ObjectRemoved event types, you can enable notification when an object or a batch of objects is removed from a bucket.
     *
     * You can request notification when an object is deleted or a versioned
     * object is permanently deleted by using the s3:ObjectRemoved:Delete event
     * type. Or you can request notification when a delete marker is created for
     * a versioned object by using s3:ObjectRemoved:DeleteMarkerCreated. For
     * information about deleting versioned objects, see Deleting Object
     * Versions. You can also use a wildcard s3:ObjectRemoved:* to request
     * notification anytime an object is deleted.
     *
     * You will not receive event notifications from automatic deletes from
     * lifecycle policies or from failed operations.
     *
     * @stability stable
     */
    OBJECT_REMOVED_DELETE_MARKER_CREATED = "s3:ObjectRemoved:DeleteMarkerCreated",
    /**
     * Using restore object event types you can receive notifications for initiation and completion when restoring objects from the S3 Glacier storage class.
     *
     * You use s3:ObjectRestore:Post to request notification of object restoration
     * initiation.
     *
     * @stability stable
     */
    OBJECT_RESTORE_POST = "s3:ObjectRestore:Post",
    /**
     * Using restore object event types you can receive notifications for initiation and completion when restoring objects from the S3 Glacier storage class.
     *
     * You use s3:ObjectRestore:Completed to request notification of
     * restoration completion.
     *
     * @stability stable
     */
    OBJECT_RESTORE_COMPLETED = "s3:ObjectRestore:Completed",
    /**
     * You can use this event type to request Amazon S3 to send a notification message when Amazon S3 detects that an object of the RRS storage class is lost.
     *
     * @stability stable
     */
    REDUCED_REDUNDANCY_LOST_OBJECT = "s3:ReducedRedundancyLostObject",
    /**
     * You receive this notification event when an object that was eligible for replication using Amazon S3 Replication Time Control failed to replicate.
     *
     * @stability stable
     */
    REPLICATION_OPERATION_FAILED_REPLICATION = "s3:Replication:OperationFailedReplication",
    /**
     * You receive this notification event when an object that was eligible for replication using Amazon S3 Replication Time Control exceeded the 15-minute threshold for replication.
     *
     * @stability stable
     */
    REPLICATION_OPERATION_MISSED_THRESHOLD = "s3:Replication:OperationMissedThreshold",
    /**
     * You receive this notification event for an object that was eligible for replication using the Amazon S3 Replication Time Control feature replicated after the 15-minute threshold.
     *
     * @stability stable
     */
    REPLICATION_OPERATION_REPLICATED_AFTER_THRESHOLD = "s3:Replication:OperationReplicatedAfterThreshold",
    /**
     * You receive this notification event for an object that was eligible for replication using Amazon S3 Replication Time Control but is no longer tracked by replication metrics.
     *
     * @stability stable
     */
    REPLICATION_OPERATION_NOT_TRACKED = "s3:Replication:OperationNotTracked"
}
/**
 * @stability stable
 */
export interface NotificationKeyFilter {
    /**
     * S3 keys must have the specified prefix.
     *
     * @stability stable
     */
    readonly prefix?: string;
    /**
     * S3 keys must have the specified suffix.
     *
     * @stability stable
     */
    readonly suffix?: string;
}
/**
 * Options for the onCloudTrailPutObject method.
 *
 * @stability stable
 */
export interface OnCloudTrailBucketEventOptions extends events.OnEventOptions {
    /**
     * Only watch changes to these object paths.
     *
     * @default - Watch changes to all objects
     * @stability stable
     */
    readonly paths?: string[];
}
/**
 * Default bucket access control types.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/acl-overview.html
 * @stability stable
 */
export declare enum BucketAccessControl {
    /**
     * Owner gets FULL_CONTROL.
     *
     * No one else has access rights.
     *
     * @stability stable
     */
    PRIVATE = "Private",
    /**
     * Owner gets FULL_CONTROL.
     *
     * The AllUsers group gets READ access.
     *
     * @stability stable
     */
    PUBLIC_READ = "PublicRead",
    /**
     * Owner gets FULL_CONTROL.
     *
     * The AllUsers group gets READ and WRITE access.
     * Granting this on a bucket is generally not recommended.
     *
     * @stability stable
     */
    PUBLIC_READ_WRITE = "PublicReadWrite",
    /**
     * Owner gets FULL_CONTROL.
     *
     * The AuthenticatedUsers group gets READ access.
     *
     * @stability stable
     */
    AUTHENTICATED_READ = "AuthenticatedRead",
    /**
     * The LogDelivery group gets WRITE and READ_ACP permissions on the bucket.
     *
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerLogs.html
     * @stability stable
     */
    LOG_DELIVERY_WRITE = "LogDeliveryWrite",
    /**
     * Object owner gets FULL_CONTROL.
     *
     * Bucket owner gets READ access.
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     *
     * @stability stable
     */
    BUCKET_OWNER_READ = "BucketOwnerRead",
    /**
     * Both the object owner and the bucket owner get FULL_CONTROL over the object.
     *
     * If you specify this canned ACL when creating a bucket, Amazon S3 ignores it.
     *
     * @stability stable
     */
    BUCKET_OWNER_FULL_CONTROL = "BucketOwnerFullControl",
    /**
     * Owner gets FULL_CONTROL.
     *
     * Amazon EC2 gets READ access to GET an Amazon Machine Image (AMI) bundle from Amazon S3.
     *
     * @stability stable
     */
    AWS_EXEC_READ = "AwsExecRead"
}
/**
 * @stability stable
 */
export interface RoutingRuleCondition {
    /**
     * The HTTP error code when the redirect is applied.
     *
     * In the event of an error, if the error code equals this value, then the specified redirect is applied.
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The HTTP error code will not be verified
     * @stability stable
     */
    readonly httpErrorCodeReturnedEquals?: string;
    /**
     * The object key name prefix when the redirect is applied.
     *
     * If both condition properties are specified, both must be true for the redirect to be applied.
     *
     * @default - The object key name will not be verified
     * @stability stable
     */
    readonly keyPrefixEquals?: string;
}
/**
 * @stability stable
 */
export declare class ReplaceKey {
    readonly withKey?: string | undefined;
    readonly prefixWithKey?: string | undefined;
    /**
     * The specific object key to use in the redirect request.
     *
     * @stability stable
     */
    static with(keyReplacement: string): ReplaceKey;
    /**
     * The object key prefix to use in the redirect request.
     *
     * @stability stable
     */
    static prefixWith(keyReplacement: string): ReplaceKey;
    private constructor();
}
/**
 * Rule that define when a redirect is applied and the redirect behavior.
 *
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/how-to-page-redirect.html
 * @stability stable
 */
export interface RoutingRule {
    /**
     * The host name to use in the redirect request.
     *
     * @default - The host name used in the original request.
     * @stability stable
     */
    readonly hostName?: string;
    /**
     * The HTTP redirect code to use on the response.
     *
     * @default "301" - Moved Permanently
     * @stability stable
     */
    readonly httpRedirectCode?: string;
    /**
     * Protocol to use when redirecting requests.
     *
     * @default - The protocol used in the original request.
     * @stability stable
     */
    readonly protocol?: RedirectProtocol;
    /**
     * Specifies the object key prefix to use in the redirect request.
     *
     * @default - The key will not be replaced
     * @stability stable
     */
    readonly replaceKey?: ReplaceKey;
    /**
     * Specifies a condition that must be met for the specified redirect to apply.
     *
     * @default - No condition
     * @stability stable
     */
    readonly condition?: RoutingRuleCondition;
}
/**
 * Options for creating Virtual-Hosted style URL.
 *
 * @stability stable
 */
export interface VirtualHostedStyleUrlOptions {
    /**
     * Specifies the URL includes the region.
     *
     * @default - true
     * @stability stable
     */
    readonly regional?: boolean;
}
export {};
