"""hammad.ai.embeddings.types"""

from typing import List, Literal

from ...base.model import Model

__all__ = (
    "Embedding",
    "EmbeddingUsage",
    "EmbeddingResponse",
)


class Embedding(Model):
    embedding: List[float]
    """The embedding vector, which is a list of floats.

    The length of vector depends on the model as listed in the
    [embedding guide](https://platform.openai.com/docs/guides/embeddings).
    """

    index: int
    """The index of the embedding in the list of embeddings."""

    object: Literal["embedding"]
    """The object type, which is always "embedding"."""

    @property
    def dimensions(self) -> int:
        """The dimensions of the embedding."""
        return len(self.embedding)


class EmbeddingUsage(Model):
    """Usage statistics for embedding requests."""

    prompt_tokens: int
    """The number of tokens used by the prompt."""

    total_tokens: int
    """The total number of tokens used by the request."""


class EmbeddingResponse(Model):
    data: List[Embedding]
    """The list of embeddings generated by the model."""

    model: str
    """The name of the model used to generate the embedding."""

    object: Literal["list"]
    """The object type, which is always "list"."""

    usage: EmbeddingUsage
    """The usage information for the request."""

    @property
    def dimensions(self) -> int:
        """The dimensions of the embedding."""
        return len(self.data[0].embedding)

    def __str__(self) -> str:
        return (
            "Embedding Response:\n"
            f">>> Model: {self.model}\n"
            f">>> Dimensions: {self.dimensions}\n"
            f">>> Usage: {self.usage}\n"
            f">>> Number of Generated Embeddings: {len(self.data)}\n"
        )
