import os
from pathlib import Path

import yaml

from src.mb_cruise_migration.framework.consts.error_consts import ErrorConsts


class OracleDbConfig(object):
    def __init__(self, config):
        self.server = config['server']
        self.port = config['port']
        self.user = config['user']
        self.password = config['password']
        self.service = config['service'] if config['service'] else None
        self.sid = config['sid'] if config['sid'] else None


class MbDatabaseProperties(OracleDbConfig):
    def __init__(self, config):
        self.pagesize = config['pagesize']
        super().__init__(config)


class CruiseDatabaseProperties(OracleDbConfig):
    def __init__(self, config):
        self.pooled = config['pooled']
        super().__init__(config)


class LogConfig(object):
    def __init__(self, config):
        self.log_path = config['log_path']
        self.report_path = config['report_path']
        self.level = config['level']


class RunParameters(object):
    def __init__(self, config):
        self.migrate_shapes = config['migrate_shapes']
        self.max_queue_size = config['max_queue_size']
        self.file_processing_thread_count = config['file_processing_thread_count']
        self.batch_insertion_size = config['batch_insertion_size']


class Migrate(object):
    def __init__(self, config):
        self.extraneous = config['extraneous']
        self.legs = config['legs']
        self.zones = config['zones']
        self.regions = config['regions']
        self.survey_metadata = config['survey_metadata']
        self.standard = config['standard']


class Manifest(object):
    def __init__(self, config):
        self.default_query = config['default_query']
        self.use_list = config['use_target_list']
        self.target_migration_surveys = config['target_migration_surveys']
        self.survey_blacklist = config['survey_blacklist']


class MigrationProperties(object):
    SURVEY_QUERY = None
    PROJECT_ROOT = None
    SRC_DIR = None
    TESTS_DIR = None
    mb_db_config = None
    cruise_db_config = None
    log_config = None
    run_parameters = None
    migrate = None
    manifest = None

    def __init__(self, filename):
        with open(filename, 'r') as yaml_data_file:
            config = yaml.safe_load(yaml_data_file)

        MigrationProperties.mb_db_config = MbDatabaseProperties(config['multibeam_db'])
        MigrationProperties.cruise_db_config = CruiseDatabaseProperties(config['cruise_db'])
        MigrationProperties.log_config = LogConfig(config['log'])
        MigrationProperties.run_parameters = RunParameters(config['run_parameters'])
        MigrationProperties.migrate = Migrate(config['migrate'])
        MigrationProperties.manifest = Manifest(config['migration_manifest'])
        self.__set_project_dirs()

    @staticmethod
    def get_project_root():
        return MigrationProperties.PROJECT_ROOT

    @staticmethod
    def get_src_dir():
        return MigrationProperties.SRC_DIR

    @staticmethod
    def get_tests_dir():
        return MigrationProperties.TESTS_DIR

    @staticmethod
    def get_survey_query():
        if MigrationProperties.SURVEY_QUERY is None:
            MigrationProperties.set_survey_query()
        return MigrationProperties.SURVEY_QUERY

    @staticmethod
    def __set_project_dirs():
        MigrationProperties.PROJECT_ROOT = Path(__file__).absolute().parent.parent.parent
        MigrationProperties.SRC_DIR = os.path.join(MigrationProperties.PROJECT_ROOT, 'src')
        MigrationProperties.TESTS_DIR = os.path.join(MigrationProperties.PROJECT_ROOT, 'tests')

    @staticmethod
    def set_survey_query():
        if MigrationProperties.manifest.use_list:
            surveys = MigrationProperties.manifest.target_migration_surveys
            if not surveys:
                raise ValueError(ErrorConsts.NO_TARGET_SURVEYS)
            print(type(surveys))
            where_conditions = f"SURVEY_NAME = {surveys.pop(0)}"
            if surveys:
                for survey in surveys:
                    where_conditions += f" OR SURVEY_NAME = {survey}"

            MigrationProperties.SURVEY_QUERY = f"SELECT * FROM MB.SURVEY WHERE {where_conditions}"
        else:
            MigrationProperties.SURVEY_QUERY = MigrationProperties.manifest.default_query
