"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractDependencies = exports.tryGetModuleVersionFromPkg = exports.tryGetModuleVersionFromRequire = exports.exec = exports.findUpMultiple = exports.findUp = exports.callsites = void 0;
const child_process_1 = require("child_process");
const fs = require("fs");
const path = require("path");
/**
 * Get callsites from the V8 stack trace API
 *
 * https://github.com/sindresorhus/callsites
 */
function callsites() {
    var _a;
    const _prepareStackTrace = Error.prepareStackTrace;
    Error.prepareStackTrace = (_, stack) => stack;
    const stack = (_a = new Error().stack) === null || _a === void 0 ? void 0 : _a.slice(1);
    Error.prepareStackTrace = _prepareStackTrace;
    return stack;
}
exports.callsites = callsites;
/**
 * Find a file by walking up parent directories
 */
function findUp(name, directory = process.cwd()) {
    return findUpMultiple([name], directory)[0];
}
exports.findUp = findUp;
/**
 * Find the lowest of multiple files by walking up parent directories. If
 * multiple files exist at the same level, they will all be returned.
 */
function findUpMultiple(names, directory = process.cwd()) {
    const absoluteDirectory = path.resolve(directory);
    const files = [];
    for (const name of names) {
        const file = path.join(directory, name);
        if (fs.existsSync(file)) {
            files.push(file);
        }
    }
    if (files.length > 0) {
        return files;
    }
    const { root } = path.parse(absoluteDirectory);
    if (absoluteDirectory === root) {
        return [];
    }
    return findUpMultiple(names, path.dirname(absoluteDirectory));
}
exports.findUpMultiple = findUpMultiple;
/**
 * Spawn sync with error handling
 */
function exec(cmd, args, options) {
    var _a, _b;
    const proc = child_process_1.spawnSync(cmd, args, options);
    if (proc.error) {
        throw proc.error;
    }
    if (proc.status !== 0) {
        if (proc.stdout || proc.stderr) {
            throw new Error(`[Status ${proc.status}] stdout: ${(_a = proc.stdout) === null || _a === void 0 ? void 0 : _a.toString().trim()}\n\n\nstderr: ${(_b = proc.stderr) === null || _b === void 0 ? void 0 : _b.toString().trim()}`);
        }
        throw new Error(`${cmd} ${args.join(' ')} ${(options === null || options === void 0 ? void 0 : options.cwd) ? `run in directory ${options.cwd}` : ''} exited with status ${proc.status}`);
    }
    return proc;
}
exports.exec = exec;
/**
 * Returns a module version by requiring its package.json file
 */
function tryGetModuleVersionFromRequire(mod) {
    try {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require(`${mod}/package.json`).version;
    }
    catch (err) {
        return undefined;
    }
}
exports.tryGetModuleVersionFromRequire = tryGetModuleVersionFromRequire;
/**
 * Gets module version from package.json content
 */
function tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath) {
    var _a, _b, _c;
    const dependencies = {
        ...(_a = pkgJson.dependencies) !== null && _a !== void 0 ? _a : {},
        ...(_b = pkgJson.devDependencies) !== null && _b !== void 0 ? _b : {},
        ...(_c = pkgJson.peerDependencies) !== null && _c !== void 0 ? _c : {},
    };
    if (!dependencies[mod]) {
        return undefined;
    }
    // If it's a "file:" version, make it absolute
    const fileMatch = dependencies[mod].match(/file:(.+)/);
    if (fileMatch && !path.isAbsolute(fileMatch[1])) {
        const absoluteFilePath = path.join(path.dirname(pkgPath), fileMatch[1]);
        return `file:${absoluteFilePath}`;
    }
    ;
    return dependencies[mod];
}
exports.tryGetModuleVersionFromPkg = tryGetModuleVersionFromPkg;
/**
 * Extract versions for a list of modules.
 *
 * First lookup the version in the package.json and then fallback to requiring
 * the module's package.json. The fallback is needed for transitive dependencies.
 */
function extractDependencies(pkgPath, modules) {
    var _a;
    const dependencies = {};
    // Use require for cache
    const pkgJson = require(pkgPath); // eslint-disable-line @typescript-eslint/no-require-imports
    for (const mod of modules) {
        const version = (_a = tryGetModuleVersionFromPkg(mod, pkgJson, pkgPath)) !== null && _a !== void 0 ? _a : tryGetModuleVersionFromRequire(mod);
        if (!version) {
            throw new Error(`Cannot extract version for module '${mod}'. Check that it's referenced in your package.json or installed.`);
        }
        dependencies[mod] = version;
    }
    return dependencies;
}
exports.extractDependencies = extractDependencies;
//# sourceMappingURL=data:application/json;base64,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