from __future__ import annotations

from pathlib import Path

from canopen.objectdictionary import Array, ObjectDictionary, Record, Variable

from .._yaml_to_od import TPDO_COMM_START, TPDO_PARA_START, DataType, gen_od
from ..configs.od_config import OdConfig
from . import INDENT4, OTHER_STD_OBJS_START, __version__, snake_to_camel


def make_enum_lines(name: str, enums: dict[int, str]) -> list[str]:
    lines = ["\n\n"]
    class_name = snake_to_camel(name)
    lines.append(f"class {class_name}(Enum):\n")
    for value, e_name in enums.items():
        lines.append(f"    {e_name.upper()} = {value}\n")
    return lines


def make_bitfield_lines(name: str, bitfields: dict[str, int | list[int]]) -> list[str]:
    lines = ["\n\n"]
    class_name = snake_to_camel(name) + "BitField"
    lines.append(f"class {class_name}(EntryBitField):\n")
    for b_name, value in bitfields.items():
        values = [value] if isinstance(value, int) else value
        bits = len(values)
        offset = min(values)
        lines.append(f"    {b_name.upper()} = {offset}, {bits}\n")
    return lines


def write_cand_od(
    name: str, od: ObjectDictionary, dir_path: str | Path | None = None, add_tpdos: bool = True
) -> None:
    enums = {}
    bitfields = {}
    entries = {}
    tpdos = []

    if dir_path is None:
        dir_path = Path().cwd()
    elif isinstance(dir_path, str):
        dir_path = Path(dir_path)

    for index in sorted(od.indices):
        obj = od[index]

        if TPDO_COMM_START <= index < TPDO_PARA_START:
            tpdos.append(index - TPDO_COMM_START)

        if index < OTHER_STD_OBJS_START:
            continue

        if isinstance(obj, Variable):
            obj_name = obj.name
            entries[obj_name] = obj

            obj_name = f"{name}_{obj.name}"
            if obj.value_descriptions:
                enums[obj_name] = obj.value_descriptions
            if obj.bit_definitions:
                bitfields[obj_name] = obj.bit_definitions
        elif isinstance(obj, Array):
            sub1 = list(obj.subindices.values())[1]
            obj_name = f"{name}_{obj.name}"
            if sub1.value_descriptions:
                enums[obj_name] = sub1.value_descriptions
            if sub1.bit_definitions:
                bitfields[obj_name] = sub1.bit_definitions

            for sub_obj in obj.subindices.values():
                if sub_obj.subindex == 0:
                    continue

                obj_name = f"{obj.name}_{sub_obj.name}"
                entries[obj_name] = sub_obj
        else:  # Record
            for sub_obj in obj.subindices.values():
                if sub_obj.subindex == 0:
                    continue

                obj_name = f"{obj.name}_{sub_obj.name}"
                entries[obj_name] = sub_obj

                obj_name = f"{name}_{obj_name}"
                if sub_obj.value_descriptions:
                    enums[obj_name] = sub_obj.value_descriptions
                if sub_obj.bit_definitions:
                    bitfields[obj_name] = sub_obj.bit_definitions

    lines = [
        f'"""generated by oresat-configs v{__version__}"""\n\n',
        "from enum import Enum\n\n",
    ]

    line = "from oresat_cand import DataType, Entry"
    if bitfields:
        line += ", EntryBitField"
    line += "\n"
    lines.append(line)

    for e_name, values in enums.items():
        lines += make_enum_lines(e_name, values)

    for b_name, values in bitfields.items():
        lines += make_bitfield_lines(b_name, values)

    lines.append("\n")
    lines.append("\n")
    node_name = snake_to_camel(name)
    lines.append(f"class {node_name}Entry(Entry):\n")
    for entry_name, obj in entries.items():
        dt = DataType(obj.data_type)

        class_name = obj.parent.name if isinstance(obj.parent, Array) else entry_name
        class_name = snake_to_camel(f"{name}_{class_name}")

        e_enum = None
        if obj.value_descriptions:
            e_enum = class_name

        bitfield = None
        if obj.bit_definitions:
            bitfield = f"{class_name}BitField"

        line = f"    {entry_name.upper()} = 0x{obj.index:X}, 0x{obj.subindex:X}, DataType.{dt.name}"
        default = obj.default
        if isinstance(default, str):
            default = f'"{default}"'
        line += f", {default}"

        if obj.min or obj.max or e_enum or bitfield:
            line += f", {obj.min}, {obj.max}, {e_enum}, {bitfield}"

        lines.append(line + "\n")

    if add_tpdos and len(tpdos) > 0:
        lines.append(f"\n\nclass {snake_to_camel(name)}Tpdo(Enum):\n")
        for i in range(len(tpdos)):
            lines.append(f"{INDENT4}TPDO_{tpdos[i] + 1} = {i}\n")

    dir_path.mkdir(parents=True, exist_ok=True)
    output_file = dir_path / f"{name}_od.py"
    with output_file.open("w") as f:
        f.writelines(lines)


def write_cand_od_config(od: ObjectDictionary, dir_path: str | Path) -> None:
    if isinstance(dir_path, str):
        dir_path = Path(dir_path)

    def make_var_line(var: Variable, subindex: int | None) -> str:
        subindex_str = "" if subindex is None else f"0x{subindex:X}"
        default = ""
        if var.default:  # optional
            if var.data_type == DataType.BYTES.value:
                default = var.default.hex()
            elif var.data_type == DataType.BOOL.value:
                default = int(var.default)
            else:
                default = var.default
        access_type = var.access_type.lower()
        return f"7,0x{var.index:X},{subindex_str},{access_type},{var.data_type},{default}\n"

    lines = [
        ";generated by oresat-configs\n",
        f";oresat-configs=v{__version__}\n",
        f";objects={len(od.indices)}\n",
        "object_type,index,subindex,access_type,data_type,default\n",
    ]
    for i in sorted(od.indices):
        obj = od[i]
        if isinstance(obj, Variable):
            lines.append(make_var_line(obj, subindex=None))
        elif isinstance(obj, Array):
            lines.append(f"8,0x{obj.index:X},,,,{len(obj)}\n")
            for s in sorted(obj.subindices):
                lines.extend(make_var_line(obj[s], subindex=s))
        elif isinstance(obj, Record):
            lines.append(f"9,0x{obj.index:X},,,,{len(obj)}\n")
            for s in sorted(obj.subindices):
                lines.extend(make_var_line(obj[s], subindex=s))

    dir_path.mkdir(parents=True, exist_ok=True)
    output_file = dir_path / "od.csv"
    with output_file.open("w") as f:
        f.writelines(lines)


def gen_cand_files(od_config_path: str | Path, dir_path: str | Path) -> None:
    if isinstance(od_config_path, str):
        od_config_path = Path(od_config_path)
    if isinstance(dir_path, str):
        dir_path = Path(dir_path)

    od_config = OdConfig.from_yaml(od_config_path)
    od = gen_od([od_config])

    dir_path.mkdir(parents=True, exist_ok=True)
    init_file = dir_path / "__init__.py"
    init_file.touch()

    write_cand_od(od_config.name, od, dir_path)
    write_cand_od_config(od, dir_path)
