from typing import List, Any, Dict
from followthemoney.types import registry

from nomenklatura.dataset.util import type_check, type_require, cleanup


class DataCoverage(object):
    """Details on the temporal and geographic scope of a dataset."""

    # Derived from Aleph
    FREQUENCIES = (
        "unknown",
        "never",
        "daily",
        "weekly",
        "monthly",
        "annual",
    )

    def __init__(self, data: Dict[str, Any]) -> None:
        self.start = type_check(registry.date, data.get("start"))
        self.end = type_check(registry.date, data.get("end"))
        self.countries: List[str] = []
        for country in data.get("countries", []):
            self.countries.append(type_require(registry.country, country))
        freq = data.get("frequency", "unknown")
        self.frequency = type_check(registry.string, freq, self.FREQUENCIES)

    def to_dict(self) -> Dict[str, Any]:
        data = {
            "start": self.start,
            "end": self.end,
            "countries": self.countries,
            "frequency": self.frequency,
        }
        return cleanup(data)

    def __repr__(self) -> str:
        return f"<DataCoverage({self.start!r}>{self.end!r}, {self.countries!r})>"
