import { JupyterFrontEnd } from '@jupyterlab/application';
import { IChangedArgs, ISettingRegistry } from '@jupyterlab/coreutils';
import { CommandRegistry } from '@phosphor/commands';
import { JSONObject } from '@phosphor/coreutils';
import { IDisposable } from '@phosphor/disposable';
import { ISignal } from '@phosphor/signaling';
import { IGitExtension, Git } from './tokens';
/** Main extension class */
export declare class GitExtension implements IGitExtension, IDisposable {
    constructor(app?: JupyterFrontEnd, settings?: ISettingRegistry.ISettings);
    /**
     * A signal emitted when the HEAD of the git repository changes.
     */
    readonly headChanged: ISignal<IGitExtension, void>;
    /**
     * Git Repository path
     *
     * This is the top-level folder fullpath.
     * null if not defined.
     */
    pathRepository: string | null;
    /**
     * A signal emitted when the current git repository changes.
     */
    readonly repositoryChanged: ISignal<IGitExtension, IChangedArgs<string | null>>;
    readonly status: Git.IStatusFileResult[];
    protected _setStatus(v: Git.IStatusFileResult[]): void;
    /**
     * A signal emitted when the current status of the git repository changes.
     */
    readonly statusChanged: ISignal<IGitExtension, Git.IStatusFileResult[]>;
    /**
     * A signal emitted when the current marking of the git repository changes.
     */
    readonly markChanged: ISignal<IGitExtension, void>;
    readonly commands: CommandRegistry | null;
    readonly shell: JupyterFrontEnd.IShell;
    /**
     * Get whether the model is disposed.
     */
    readonly isDisposed: boolean;
    /**
     * Dispose of the resources held by the model.
     */
    dispose(): void;
    /**
     * Gets the path of the file relative to the Jupyter server root.
     *
     * If no path is provided, returns the Git repository top folder relative path.
     * If no Git repository selected, return null
     *
     * @param path the file path relative to Git repository top folder
     */
    getRelativeFilePath(path?: string): string | null;
    /** Make request for the Git Pull API. */
    pull(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /** Make request for the Git Push API. */
    push(auth?: Git.IAuth): Promise<Git.IPushPullResult>;
    /** Make request for the Git Clone API. */
    clone(path: string, url: string, auth?: Git.IAuth): Promise<Git.ICloneResult>;
    /** Make request for all git info of the repository
     * (This API is also implicitly used to check if the current repo is a Git repo)
     */
    allHistory(historyCount?: number): Promise<Git.IAllHistory>;
    /** Make request for top level path of repository 'path' */
    showTopLevel(path: string): Promise<Git.IShowTopLevelResult>;
    /** Make request for the prefix path of a directory 'path',
     * with respect to the root directory of repository
     */
    showPrefix(path: string): Promise<Git.IShowPrefixResult>;
    refresh(): Promise<void>;
    refreshStatus(): Promise<void>;
    /** Refresh the git repository status */
    protected _refreshStatus(): Promise<void>;
    /** Make request for git commit logs of repository */
    log(historyCount?: number): Promise<Git.ILogResult>;
    /** Make request for detailed git commit info of
     * commit 'hash' in the repository
     */
    detailedLog(hash: string): Promise<Git.ISingleCommitFilePathInfo>;
    /** Make request for a list of all git branches in the repository */
    protected _branch(): Promise<Git.IBranchResult>;
    refreshBranch(): Promise<void>;
    readonly branches: Git.IBranch[];
    readonly currentBranch: Git.IBranch;
    /** Make request to add one or all files into
     * the staging area in repository
     */
    add(...filename: string[]): Promise<Response>;
    /** Make request to add all unstaged files into
     * the staging area in repository 'path'
     */
    addAllUnstaged(): Promise<Response>;
    /** Make request to add all untracked files into
     * the staging area in the repository
     */
    addAllUntracked(): Promise<Response>;
    /**
     * Make request to switch current working branch,
     * create new branch if needed,
     * or discard a specific file change or all changes
     * TODO: Refactor into seperate endpoints for each kind of checkout request
     *
     * If a branch name is provided, check it out (with or without creating it)
     * If a filename is provided, check the file out
     * If nothing is provided, check all files out
     */
    checkout(options?: Git.ICheckoutOptions): Promise<Git.ICheckoutResult>;
    /** Make request to commit all staged files in the repository */
    commit(message: string): Promise<Response>;
    /**
     * Get or set Git configuration options
     *
     * @param options Configuration options to set (undefined to get)
     */
    config(options?: JSONObject): Promise<Response>;
    /**
     * Make request to move one or all files from the staged to the unstaged area
     *
     * @param filename - Path to a file to be reset. Leave blank to reset all
     *
     * @returns a promise that resolves when the request is complete.
     */
    reset(filename?: string): Promise<Response>;
    /** Make request to delete changes from selected commit */
    deleteCommit(message: string, commitId: string): Promise<Response>;
    /**
     * Make request to reset to selected commit
     *
     * @param commitId - Git commit specification. Leave blank to reset to HEAD
     *
     * @returns a promise that resolves when the request is complete.
     */
    resetToCommit(commitId?: string): Promise<Response>;
    /** Make request to initialize a  new git repository at path 'path' */
    init(path: string): Promise<Response>;
    registerDiffProvider(filetypes: string[], callback: Git.IDiffCallback): void;
    performDiff(filename: string, revisionA: string, revisionB: string): void;
    /**
     * Test whether the model is ready.
     */
    readonly isReady: boolean;
    /**
     * A promise that fulfills when the model is ready.
     */
    readonly ready: Promise<void>;
    /**
     * Add file named fname to current marker obj
     */
    addMark(fname: string, mark: boolean): void;
    /**
     * get current mark of fname
     */
    getMark(fname: string): boolean;
    /**
     * Toggle mark for file named fname in current marker obj
     */
    toggleMark(fname: string): void;
    private _getServerRoot;
    /**
     * set marker obj for repo path/branch combination
     */
    private _setMarker;
    private _status;
    private _pathRepository;
    private _branches;
    private _currentBranch;
    private _serverRoot;
    private _app;
    private _diffProviders;
    private _isDisposed;
    private _markerCache;
    private _currentMarker;
    private _readyPromise;
    private _pendingReadyPromise;
    private _poll;
    private _headChanged;
    private _markChanged;
    private _repositoryChanged;
    private _statusChanged;
}
export declare class BranchMarker implements Git.IBranchMarker {
    private _refresh;
    constructor(_refresh: () => void);
    add(fname: string, mark?: boolean): void;
    get(fname: string): boolean;
    set(fname: string, mark: boolean): void;
    toggle(fname: string): void;
    private _marks;
}
export declare class Markers {
    private _refresh;
    constructor(_refresh: () => void);
    get(path: string, branch: string): BranchMarker;
    static markerKey(path: string, branch: string): string;
    private _branchMarkers;
}
