import { PathExt, Poll } from '@jupyterlab/coreutils';
import { ServerConnection } from '@jupyterlab/services';
import { Signal } from '@phosphor/signaling';
import { httpGitRequest } from './git';
// Default refresh interval (in milliseconds) for polling the current Git status (NOTE: this value should be the same value as in the plugin settings schema):
const DEFAULT_REFRESH_INTERVAL = 3000; // ms
/** Main extension class */
export class GitExtension {
    constructor(app = null, settings) {
        this._status = [];
        this._pathRepository = null;
        this._diffProviders = {};
        this._isDisposed = false;
        this._markerCache = new Markers(() => this._markChanged.emit());
        this._currentMarker = null;
        this._readyPromise = Promise.resolve();
        this._pendingReadyPromise = 0;
        this._headChanged = new Signal(this);
        this._markChanged = new Signal(this);
        this._repositoryChanged = new Signal(this);
        this._statusChanged = new Signal(this);
        const model = this;
        this._app = app;
        // Load the server root path
        this._getServerRoot()
            .then(root => {
            this._serverRoot = root;
        })
            .catch(reason => {
            console.error(`Fail to get the server root path.\n${reason}`);
        });
        let interval;
        if (settings) {
            interval = settings.composite.refreshInterval;
            settings.changed.connect(onSettingsChange, this);
        }
        else {
            interval = DEFAULT_REFRESH_INTERVAL;
        }
        const poll = new Poll({
            factory: () => model._refreshStatus(),
            frequency: {
                interval: interval,
                backoff: true,
                max: 300 * 1000
            },
            standby: 'when-hidden'
        });
        this._poll = poll;
        /**
         * Callback invoked upon a change to plugin settings.
         *
         * @private
         * @param settings - settings registry
         */
        function onSettingsChange(settings) {
            const freq = poll.frequency;
            poll.frequency = {
                interval: settings.composite.refreshInterval,
                backoff: freq.backoff,
                max: freq.max
            };
        }
    }
    /**
     * A signal emitted when the HEAD of the git repository changes.
     */
    get headChanged() {
        return this._headChanged;
    }
    /**
     * Git Repository path
     *
     * This is the top-level folder fullpath.
     * null if not defined.
     */
    get pathRepository() {
        return this._pathRepository;
    }
    set pathRepository(v) {
        const change = {
            name: 'pathRepository',
            newValue: null,
            oldValue: this._pathRepository
        };
        if (v === null) {
            this._pendingReadyPromise += 1;
            this._readyPromise.then(() => {
                this._pathRepository = null;
                this._pendingReadyPromise -= 1;
                if (change.newValue !== change.oldValue) {
                    this.refresh().then(() => this._repositoryChanged.emit(change));
                }
            });
        }
        else {
            const currentReady = this._readyPromise;
            this._pendingReadyPromise += 1;
            this._readyPromise = Promise.all([currentReady, this.showTopLevel(v)])
                .then(r => {
                const results = r[1];
                if (results.code === 0) {
                    this._pathRepository = results.top_repo_path;
                    change.newValue = results.top_repo_path;
                }
                else {
                    this._pathRepository = null;
                }
                if (change.newValue !== change.oldValue) {
                    this.refresh().then(() => this._repositoryChanged.emit(change));
                }
            })
                .catch(reason => {
                console.error(`Fail to find git top level for path ${v}.\n${reason}`);
            });
            void this._readyPromise.then(() => {
                this._pendingReadyPromise -= 1;
            });
        }
    }
    /**
     * A signal emitted when the current git repository changes.
     */
    get repositoryChanged() {
        return this._repositoryChanged;
    }
    get status() {
        return this._status;
    }
    _setStatus(v) {
        this._status = v;
        this._statusChanged.emit(this._status);
    }
    /**
     * A signal emitted when the current status of the git repository changes.
     */
    get statusChanged() {
        return this._statusChanged;
    }
    /**
     * A signal emitted when the current marking of the git repository changes.
     */
    get markChanged() {
        return this._markChanged;
    }
    get commands() {
        return this._app ? this._app.commands : null;
    }
    get shell() {
        return this._app ? this._app.shell : null;
    }
    /**
     * Get whether the model is disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Dispose of the resources held by the model.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._isDisposed = true;
        this._poll.dispose();
        Signal.clearData(this);
    }
    /**
     * Gets the path of the file relative to the Jupyter server root.
     *
     * If no path is provided, returns the Git repository top folder relative path.
     * If no Git repository selected, return null
     *
     * @param path the file path relative to Git repository top folder
     */
    getRelativeFilePath(path) {
        if (this.pathRepository === null || this._serverRoot === undefined) {
            return null;
        }
        return PathExt.join(PathExt.relative(this._serverRoot, this.pathRepository), path || '');
    }
    /** Make request for the Git Pull API. */
    async pull(auth) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let obj = {
                current_path: path,
                auth
            };
            let response = await httpGitRequest('/git/pull', 'POST', obj);
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            this._headChanged.emit();
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for the Git Push API. */
    async push(auth) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let obj = {
                current_path: path,
                auth
            };
            let response = await httpGitRequest('/git/push', 'POST', obj);
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            this._headChanged.emit();
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for the Git Clone API. */
    async clone(path, url, auth) {
        try {
            let obj = {
                current_path: path,
                clone_url: url,
                auth
            };
            let response = await httpGitRequest('/git/clone', 'POST', obj);
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for all git info of the repository
     * (This API is also implicitly used to check if the current repo is a Git repo)
     */
    async allHistory(historyCount = 25) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let response = await httpGitRequest('/git/all_history', 'POST', {
                current_path: path,
                history_count: historyCount
            });
            if (response.status !== 200) {
                const data = await response.text();
                throw new ServerConnection.ResponseError(response, data);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for top level path of repository 'path' */
    async showTopLevel(path) {
        try {
            let response = await httpGitRequest('/git/show_top_level', 'POST', {
                current_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for the prefix path of a directory 'path',
     * with respect to the root directory of repository
     */
    async showPrefix(path) {
        try {
            let response = await httpGitRequest('/git/show_prefix', 'POST', {
                current_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    async refresh() {
        await this.refreshBranch();
        await this.refreshStatus();
    }
    async refreshStatus() {
        await this._poll.refresh();
        await this._poll.tick;
    }
    /** Refresh the git repository status */
    async _refreshStatus() {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            this._setStatus([]);
            return Promise.resolve();
        }
        try {
            let response = await httpGitRequest('/git/status', 'POST', {
                current_path: path
            });
            const data = await response.json();
            if (response.status !== 200) {
                console.error(data.message);
                // TODO should we notify the user
                this._setStatus([]);
            }
            this._setStatus(data.files);
        }
        catch (err) {
            console.error(err);
            // TODO should we notify the user
            this._setStatus([]);
        }
    }
    /** Make request for git commit logs of repository */
    async log(historyCount = 25) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let response = await httpGitRequest('/git/log', 'POST', {
                current_path: path,
                history_count: historyCount
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for detailed git commit info of
     * commit 'hash' in the repository
     */
    async detailedLog(hash) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let response = await httpGitRequest('/git/detailed_log', 'POST', {
                selected_hash: hash,
                current_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request for a list of all git branches in the repository */
    async _branch() {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        try {
            let response = await httpGitRequest('/git/branch', 'POST', {
                current_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    async refreshBranch() {
        const response = await this._branch();
        if (response.code === 0) {
            this._branches = response.branches;
            this._currentBranch = response.current_branch;
            if (this._currentBranch) {
                // set up the marker obj for the current (valid) repo/branch combination
                this._setMarker(this.pathRepository, this._currentBranch.name);
            }
        }
        else {
            this._branches = [];
            this._currentBranch = null;
        }
    }
    get branches() {
        return this._branches;
    }
    get currentBranch() {
        return this._currentBranch;
    }
    /** Make request to add one or all files into
     * the staging area in repository
     */
    async add(...filename) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        const response = await httpGitRequest('/git/add', 'POST', {
            add_all: !filename,
            filename: filename || '',
            top_repo_path: path
        });
        this.refreshStatus();
        return Promise.resolve(response);
    }
    /** Make request to add all unstaged files into
     * the staging area in repository 'path'
     */
    async addAllUnstaged() {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/add_all_unstaged', 'POST', {
                top_repo_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request to add all untracked files into
     * the staging area in the repository
     */
    async addAllUntracked() {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/add_all_untracked', 'POST', {
                top_repo_path: path
            });
            if (response.status !== 200) {
                const data = await response.json();
                throw new ServerConnection.ResponseError(response, data.message);
            }
            this.refreshStatus();
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /**
     * Make request to switch current working branch,
     * create new branch if needed,
     * or discard a specific file change or all changes
     * TODO: Refactor into seperate endpoints for each kind of checkout request
     *
     * If a branch name is provided, check it out (with or without creating it)
     * If a filename is provided, check the file out
     * If nothing is provided, check all files out
     */
    async checkout(options) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve({
                code: -1,
                message: 'Not in a git repository.'
            });
        }
        const body = {
            checkout_branch: false,
            new_check: false,
            branchname: '',
            checkout_all: true,
            filename: '',
            top_repo_path: path
        };
        if (options !== undefined) {
            if (options.branchname) {
                body.branchname = options.branchname;
                body.checkout_branch = true;
                body.new_check = options.newBranch === true;
            }
            else if (options.filename) {
                body.filename = options.filename;
                body.checkout_all = false;
            }
        }
        try {
            let response = await httpGitRequest('/git/checkout', 'POST', body);
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            if (body.checkout_branch) {
                await this.refreshBranch();
                this._headChanged.emit();
            }
            else {
                this.refreshStatus();
            }
            return response.json();
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request to commit all staged files in the repository */
    async commit(message) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/commit', 'POST', {
                commit_msg: message,
                top_repo_path: path
            });
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            this.refreshStatus();
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /**
     * Get or set Git configuration options
     *
     * @param options Configuration options to set (undefined to get)
     */
    async config(options) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let method = 'POST';
            let body = { path, options };
            let response = await httpGitRequest('/git/config', method, body);
            if (!response.ok) {
                const jsonData = await response.json();
                throw new ServerConnection.ResponseError(response, jsonData.message);
            }
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /**
     * Make request to move one or all files from the staged to the unstaged area
     *
     * @param filename - Path to a file to be reset. Leave blank to reset all
     *
     * @returns a promise that resolves when the request is complete.
     */
    async reset(filename) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/reset', 'POST', {
                reset_all: filename === undefined,
                filename: filename === undefined ? null : filename,
                top_repo_path: path
            });
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            this.refreshStatus();
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request to delete changes from selected commit */
    async deleteCommit(message, commitId) {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/delete_commit', 'POST', {
                commit_id: commitId,
                top_repo_path: path
            });
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            await this.commit(message);
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /**
     * Make request to reset to selected commit
     *
     * @param commitId - Git commit specification. Leave blank to reset to HEAD
     *
     * @returns a promise that resolves when the request is complete.
     */
    async resetToCommit(commitId = '') {
        await this.ready;
        const path = this.pathRepository;
        if (path === null) {
            return Promise.resolve(new Response(JSON.stringify({
                code: -1,
                message: 'Not in a git repository.'
            })));
        }
        try {
            let response = await httpGitRequest('/git/reset_to_commit', 'POST', {
                commit_id: commitId,
                top_repo_path: path
            });
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            await this.refreshBranch();
            this._headChanged.emit();
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    /** Make request to initialize a  new git repository at path 'path' */
    async init(path) {
        try {
            let response = await httpGitRequest('/git/init', 'POST', {
                current_path: path
            });
            if (response.status !== 200) {
                return response.json().then((data) => {
                    throw new ServerConnection.ResponseError(response, data.message);
                });
            }
            return response;
        }
        catch (err) {
            throw new ServerConnection.NetworkError(err);
        }
    }
    registerDiffProvider(filetypes, callback) {
        filetypes.forEach(fileType => {
            this._diffProviders[fileType] = callback;
        });
    }
    performDiff(filename, revisionA, revisionB) {
        let extension = PathExt.extname(filename).toLocaleLowerCase();
        if (this._diffProviders[extension] !== undefined) {
            this._diffProviders[extension](filename, revisionA, revisionB);
        }
        else if (this.commands) {
            this.commands.execute('git:terminal-cmd', {
                cmd: 'git diff ' + revisionA + ' ' + revisionB
            });
        }
    }
    /**
     * Test whether the model is ready.
     */
    get isReady() {
        return this._pendingReadyPromise === 0;
    }
    /**
     * A promise that fulfills when the model is ready.
     */
    get ready() {
        return this._readyPromise;
    }
    /**
     * Add file named fname to current marker obj
     */
    addMark(fname, mark) {
        this._currentMarker.add(fname, mark);
    }
    /**
     * get current mark of fname
     */
    getMark(fname) {
        return this._currentMarker.get(fname);
    }
    /**
     * Toggle mark for file named fname in current marker obj
     */
    toggleMark(fname) {
        this._currentMarker.toggle(fname);
    }
    async _getServerRoot() {
        try {
            const response = await httpGitRequest('/git/server_root', 'GET', null);
            const data = await response.json();
            return data['server_root'];
        }
        catch (reason) {
            throw new Error(reason);
        }
    }
    /**
     * set marker obj for repo path/branch combination
     */
    _setMarker(path, branch) {
        this._currentMarker = this._markerCache.get(path, branch);
        return this._currentMarker;
    }
}
export class BranchMarker {
    constructor(_refresh) {
        this._refresh = _refresh;
        this._marks = {};
    }
    add(fname, mark = true) {
        if (!(fname in this._marks)) {
            this.set(fname, mark);
        }
    }
    get(fname) {
        return this._marks[fname];
    }
    set(fname, mark) {
        this._marks[fname] = mark;
        this._refresh();
    }
    toggle(fname) {
        this.set(fname, !this._marks[fname]);
    }
}
export class Markers {
    constructor(_refresh) {
        this._refresh = _refresh;
        this._branchMarkers = {};
    }
    get(path, branch) {
        const key = Markers.markerKey(path, branch);
        if (key in this._branchMarkers) {
            return this._branchMarkers[key];
        }
        let marker = new BranchMarker(this._refresh);
        this._branchMarkers[key] = marker;
        return marker;
    }
    static markerKey(path, branch) {
        return [path, branch].join(':');
    }
}
//# sourceMappingURL=model.js.map