/*
 * Copyright 2018-2020 IBM Corporation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
import { showDialog, Dialog } from '@jupyterlab/apputils';
import { URLExt } from '@jupyterlab/coreutils';
import { ServerConnection } from '@jupyterlab/services';
import * as React from 'react';
const MESSAGE_DISPLAY = 'elyra-pipelineSubmission-messageDisplay';
const ERROR_DISPLAY_BUTTON = 'elyra-pipelineSubmission-errDisplayButton';
const DOWN_ICON_CLASS = 'elyra-pipelineSubmission-errDisplayButton-down';
const UP_ICON_CLASS = 'elyra-pipelineSubmission-errDisplayButton-up';
const ERROR_DETAILS_VISIBLE = 'elyra-pipelineSubmission-error-visible';
const ERROR_DETAILS_HIDDEN = 'elyra-pipelineSubmission-error-hidden';
export class SubmissionHandler {
    static handleError(response, submissionType) {
        const reason = response.reason ? response.reason : '';
        const message = response.message ? response.message : '';
        const timestamp = response.timestamp ? response.timestamp : '';
        const traceback = response.traceback ? response.traceback : '';
        const default_body = response.timestamp
            ? 'Check the JupyterLab log for more details at ' + response.timestamp
            : 'Check the JupyterLab log for more details';
        return showDialog({
            title: 'Error submitting ' + submissionType,
            body: reason || message ? (React.createElement(ErrorDialogContent, { reason: reason, message: message, timestamp: timestamp, traceback: traceback, default_msg: default_body })) : (React.createElement("p", null, default_body)),
            buttons: [Dialog.okButton()]
        });
    }
    static handle404(submissionType) {
        return showDialog({
            title: 'Error submitting ' + submissionType,
            body: React.createElement("p", null, "Elyra service endpoint not found."),
            buttons: [Dialog.okButton()]
        });
    }
    static makeGetRequest(requestExt, submissionType, dialogCallback) {
        this.makeServerRequest(requestExt, { method: 'GET' }, submissionType, dialogCallback);
    }
    static makePostRequest(requestExt, requestBody, submissionType, dialogCallback) {
        this.makeServerRequest(requestExt, { method: 'POST', body: requestBody }, submissionType, dialogCallback);
    }
    static makeServerRequest(requestExt, requestOptions, submissionType, dialogCallback) {
        // use ServerConnection utility to make calls to Jupyter Based services
        // which in this case are the in the extension installed by this package
        const settings = ServerConnection.makeSettings();
        const requestUrl = URLExt.join(settings.baseUrl, requestExt);
        console.log('Submitting a ' + requestOptions.method + ' request to ' + requestUrl);
        // Note: a button is required to resolve the dialog below
        const waitDialog = new Dialog({
            title: 'Submitting request...',
            body: 'This may take some time',
            buttons: [Dialog.okButton()]
        });
        waitDialog.launch();
        ServerConnection.makeRequest(requestUrl, requestOptions, settings).then((response) => {
            waitDialog.resolve();
            response.json().then((result) => {
                if (response.status !== 200) {
                    return this.handleError(result, submissionType);
                }
                return dialogCallback(result);
            }, (reason) => {
                // handle 404 if elyra server extension is not found
                return this.handle404(submissionType);
            });
        });
    }
    static submitPipeline(pipeline, runtime_config, submissionType) {
        console.log('Pipeline definition:');
        console.log(pipeline);
        this.makePostRequest('api/scheduler', JSON.stringify(pipeline), submissionType, (data) => {
            const dialogTitle = 'Job submission to ' + runtime_config + ' succeeded';
            const dialogBody = (React.createElement("p", null,
                "Check the status of your run at",
                ' ',
                React.createElement("a", { href: data.url, target: "_blank", rel: "noopener noreferrer" }, "Run Details")));
            return showDialog({
                title: dialogTitle,
                body: dialogBody,
                buttons: [Dialog.okButton()]
            });
        });
    }
}
class ErrorDialogContent extends React.Component {
    constructor(props) {
        super(props);
        this.state = { expanded: false };
    }
    toggleMsgDisplay() {
        // Switch expanded flag
        const expanded = !this.state.expanded;
        this.setState({ expanded: expanded });
    }
    render() {
        const details = this.props.traceback ? (React.createElement("div", null,
            React.createElement("br", null),
            React.createElement("div", null,
                React.createElement("button", { className: ERROR_DISPLAY_BUTTON +
                        ' ' +
                        (this.state.expanded ? UP_ICON_CLASS : DOWN_ICON_CLASS), onClick: () => {
                        this.toggleMsgDisplay();
                    } }),
                'Error details: '),
            React.createElement("br", null),
            React.createElement("div", { className: this.state.expanded ? ERROR_DETAILS_VISIBLE : ERROR_DETAILS_HIDDEN }, this.props.traceback))) : null;
        return (React.createElement("div", { className: MESSAGE_DISPLAY },
            this.props.message,
            React.createElement("br", null),
            details,
            React.createElement("br", null),
            React.createElement("div", null, this.props.default_msg)));
    }
}
//# sourceMappingURL=submission.js.map