import * as React from 'react';
import { findRepoButtonStyle, panelContainerStyle, panelWarningStyle } from '../style/GitPanelStyle';
import { decodeStage } from '../utils';
import { BranchHeader } from './BranchHeader';
import { FileList } from './FileList';
import { HistorySideBar } from './HistorySideBar';
import { PathHeader } from './PathHeader';
/** A React component for the git extension's main display */
export class GitPanel extends React.Component {
    constructor(props) {
        super(props);
        this.refreshBranch = async () => {
            const { currentBranch } = this.props.model;
            this.setState({
                branches: this.props.model.branches,
                currentBranch: currentBranch ? currentBranch.name : 'master',
                upstreamBranch: currentBranch ? currentBranch.upstream : ''
            });
        };
        this.refreshHistory = async () => {
            if (this.props.model.pathRepository !== null) {
                // Get git log for current branch
                let logData = await this.props.model.log(this.props.settings.composite['historyCount']);
                let pastCommits = new Array();
                if (logData.code === 0) {
                    pastCommits = logData.commits;
                }
                this.setState({
                    pastCommits: pastCommits
                });
            }
        };
        this.setStatus = () => {
            if (this.props.model.pathRepository !== null) {
                // Get git status for current branch
                let stagedFiles = new Array();
                let unstagedFiles = new Array();
                let untrackedFiles = new Array();
                let statusFiles = this.props.model.status;
                if (statusFiles.length > 0) {
                    for (let i = 0; i < statusFiles.length; i++) {
                        const file = statusFiles[i];
                        const { x, y } = file;
                        const stage = decodeStage(x, y);
                        // If file is untracked
                        if (stage === 'untracked') {
                            untrackedFiles.push(file);
                        }
                        else if (stage === 'unstaged') {
                            unstagedFiles.push(file);
                        }
                        else if (stage === 'staged') {
                            stagedFiles.push(file);
                        }
                    }
                }
                this.setState({
                    stagedFiles: stagedFiles,
                    unstagedFiles: unstagedFiles,
                    untrackedFiles: untrackedFiles
                });
            }
        };
        this.refreshStatus = async () => {
            await this.props.model.refreshStatus();
        };
        /**
         * Refresh widget, update all content
         */
        this.refresh = async () => {
            if (this.props.model.pathRepository !== null) {
                await this.refreshBranch();
                await this.refreshHistory();
                await this.refreshStatus();
            }
        };
        this.toggleSidebar = () => {
            if (!this.state.isHistoryVisible) {
                this.refreshHistory();
            }
            this.setState({ isHistoryVisible: !this.state.isHistoryVisible });
        };
        this.state = {
            inGitRepository: false,
            branches: [],
            currentBranch: '',
            upstreamBranch: '',
            pastCommits: [],
            stagedFiles: [],
            unstagedFiles: [],
            untrackedFiles: [],
            isHistoryVisible: false
        };
        props.model.repositoryChanged.connect((_, args) => {
            this.setState({
                inGitRepository: args.newValue !== null
            });
            this.refresh();
        }, this);
        props.model.statusChanged.connect(() => {
            this.setStatus();
        }, this);
        props.model.headChanged.connect(async () => {
            await this.refreshBranch();
            if (this.state.isHistoryVisible) {
                this.refreshHistory();
            }
            else {
                this.refreshStatus();
            }
        }, this);
        props.model.markChanged.connect(() => this.forceUpdate());
        props.settings.changed.connect(this.refresh, this);
    }
    render() {
        let main;
        let sub;
        if (this.state.isHistoryVisible) {
            sub = (React.createElement(HistorySideBar, { isExpanded: this.state.isHistoryVisible, branches: this.state.branches, pastCommits: this.state.pastCommits, model: this.props.model, renderMime: this.props.renderMime }));
        }
        else {
            sub = (React.createElement(FileList, { stagedFiles: this.state.stagedFiles, unstagedFiles: this.state.unstagedFiles, untrackedFiles: this.state.untrackedFiles, model: this.props.model, renderMime: this.props.renderMime, settings: this.props.settings }));
        }
        if (this.state.inGitRepository) {
            const disableBranchOps = Boolean(this.props.settings.composite['disableBranchWithChanges'] &&
                ((this.state.unstagedFiles && this.state.unstagedFiles.length) ||
                    (this.state.stagedFiles && this.state.stagedFiles.length)));
            main = (React.createElement(React.Fragment, null,
                React.createElement(BranchHeader, { model: this.props.model, refresh: this.refreshBranch, currentBranch: this.state.currentBranch, upstreamBranch: this.state.upstreamBranch, stagedFiles: this.state.stagedFiles, data: this.state.branches, disabled: disableBranchOps, toggleSidebar: this.toggleSidebar, sideBarExpanded: this.state.isHistoryVisible }),
                sub));
        }
        else {
            main = (React.createElement("div", { className: panelWarningStyle },
                React.createElement("div", null, "You aren\u2019t in a git repository."),
                React.createElement("button", { className: findRepoButtonStyle, onClick: () => this.props.model.commands.execute('filebrowser:toggle-main') }, "Go find a repo")));
        }
        return (React.createElement("div", { className: panelContainerStyle },
            React.createElement(PathHeader, { model: this.props.model, refresh: async () => {
                    await this.refreshBranch();
                    if (this.state.isHistoryVisible) {
                        this.refreshHistory();
                    }
                    else {
                        this.refreshStatus();
                    }
                } }),
            main));
    }
}
//# sourceMappingURL=GitPanel.js.map