"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const elb = require("aws-cdk-lib/aws-elasticloadbalancingv2");
const cdk = require("aws-cdk-lib");
const defaults = require("@aws-solutions-constructs/core");
const assertions_1 = require("aws-cdk-lib/assertions");
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        LoadBalancerAttributes: [
            {
                Key: "deletion_protection.enabled",
                Value: "false"
            },
            {
                Key: "access_logs.s3.enabled",
                Value: "true"
            },
            {
                Key: "access_logs.s3.bucket",
                Value: {
                    Ref: "testoneE6ACFBB6"
                }
            },
            {
                Key: "access_logs.s3.prefix",
                Value: ""
            }
        ],
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test new load balancer and new lambda function for HTTP api', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::Lambda::Function', 1);
});
test('Test existing load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::Lambda::Function', 1);
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internet-facing",
    });
});
test('Test new load balancer and existing lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: testFunctionName,
        vpc: testExistingVpc
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true,
        existingVpc: testExistingVpc
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
});
test("Test existing load balancer and existing lambda function", () => {
    const testFunctionName = "fixed-name";
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: "us-east-1" },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, "test-alb", {
        vpc: testExistingVpc,
    });
    const lambdaFunction = new lambda.Function(stack, "existing-function", {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: "index.handler",
        functionName: testFunctionName,
        vpc: testExistingVpc,
    });
    const props = {
        existingLambdaObj: lambdaFunction,
        existingLoadBalancerObj: existingAlb,
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
        existingVpc: testExistingVpc,
    };
    new lib_1.AlbToLambda(stack, "test-one", props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::LoadBalancer", {
        Scheme: "internal",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTP",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::Listener", {
        Protocol: "HTTPS",
    });
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::ListenerCertificate", {});
    template.hasResourceProperties("AWS::ElasticLoadBalancingV2::TargetGroup", {
        TargetType: "lambda",
    });
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
    });
});
test('Test new load balancer and new lambda function', () => {
    const testFunctionName = 'fixed-name';
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            functionName: testFunctionName,
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
        },
        publicApi: true,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::Lambda::Function', {
        FunctionName: testFunctionName
    });
    template.resourceCountIs('AWS::EC2::VPC', 1);
});
test('Test HTTPS adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test HTTP adding 2 lambda targets, second with rules', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.resourceCountIs('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    defaults.expectNonexistence(stack, 'AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        "*admin*"
                    ]
                }
            }
        ],
    });
});
test('Test new load balancer and new lambda function', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda',
        Name: 'different-name'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test logging turned off', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs('AWS::S3::Bucket', 0);
});
test('Check Properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const target = new lib_1.AlbToLambda(stack, 'test-one', props);
    expect(target.loadBalancer);
    expect(target.vpc);
    expect(target.lambdaFunction);
    expect(target.listener);
});
test('Test custom ALB properties', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        loadBalancerProps: {
            loadBalancerName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
        Name: 'custom-name',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
});
test('Test custom logging bucket', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'custom-name',
        }
    };
    new lib_1.AlbToLambda(stack, 'test-one', props);
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: 'internet-facing',
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: 'lambda'
    });
    template.hasResourceProperties('AWS::ElasticLoadBalancingV2::ListenerCertificate', {});
    template.hasResourceProperties('AWS::S3::Bucket', {
        BucketName: 'custom-name'
    });
});
test('Test providing certificateArns is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificateArns: ['	arn:aws:acm:us-east-1:123456789012:certificate/11112222-3333-1234-1234-123456789012']
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates/);
});
test('Test logging off with logBucketProperties provided is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        targetProps: {
            targetGroupName: 'different-name'
        },
        publicApi: true,
        logAlbAccessLogs: false,
        albLoggingBucketProps: {
            bucketName: 'should-fail'
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/Error - If logAlbAccessLogs is false, supplying albLoggingBucketProps is invalid./);
});
test('Test certificate with HTTP is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")],
            protocol: 'HTTP',
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    expect(app).toThrowError(/HTTP listeners cannot use a certificate/);
});
test('Test new ALB with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with no listener with no listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP/);
});
test('Test existing ALB with a listener with listenerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        ruleProps: {
            conditions: [elb.ListenerCondition.pathPatterns(["*admin*"])],
            priority: 10
        },
        listenerProps: {
            protocol: 'HTTPS'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/This load balancer already has a listener, listenerProps may not be specified/);
});
test('Test second target with no rules is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: true
    };
    const firstConstruct = new lib_1.AlbToLambda(stack, 'test-one', props);
    const secondProps = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        existingVpc: firstConstruct.vpc,
        existingLoadBalancerObj: firstConstruct.loadBalancer,
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-two', secondProps);
    };
    // Assertion
    expect(app).toThrowError(/When adding a second target to an existing listener, there must be rules provided/);
});
test('Test no cert for an HTTPS listener is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {},
        publicApi: true
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/A listener using HTTPS protocol requires a certificate/);
});
test('Test existingLoadBalancerObj and loadBalancerProps is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        existingVpc: testExistingVpc,
        publicApi: true,
        loadBalancerProps: {
            internetFacing: true
        }
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/Error - Either provide loadBalancerProps or existingLoadBalancerObj, but not both./);
});
test('Test existingLoadBalancerObj and no existingVpc is an error', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: testExistingVpc,
        internetFacing: true
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            protocol: 'HTTP'
        },
        existingLoadBalancerObj: existingAlb,
        publicApi: true,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'test-one', props);
    };
    // Assertion
    expect(app).toThrowError(/An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use./);
});
test('Confirm that CheckLambdaProps is called', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testExistingVpc = defaults.getTestVpc(stack);
    const lambdaFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        functionName: 'name',
        vpc: testExistingVpc
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        existingLambdaObj: lambdaFunction,
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: false,
        existingVpc: testExistingVpc,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'new-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
test('Confirm that CheckVpcProps is called', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const props = {
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: false,
        vpcProps: {},
        existingVpc: defaults.getTestVpc(stack),
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'new-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm that CheckAlbProps is called', () => {
    const stack = new cdk.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    const testName = 'test-value';
    const existingVpc = defaults.getTestVpc(stack);
    const existingAlb = new elb.ApplicationLoadBalancer(stack, 'test-alb', {
        vpc: existingVpc,
        internetFacing: true,
        loadBalancerName: testName,
    });
    const props = {
        existingVpc,
        lambdaFunctionProps: {
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler'
        },
        listenerProps: {
            certificates: [defaults.getFakeCertificate(stack, "fake-cert")]
        },
        publicApi: false,
        vpcProps: {},
        loadBalancerProps: {
            loadBalancerName: 'new-loadbalancer',
            internetFacing: true,
        },
        existingLoadBalancerObj: existingAlb,
    };
    const app = () => {
        new lib_1.AlbToLambda(stack, 'new-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide loadBalancerProps or existingLoadBalancerObj, but not both.\n');
});
test('Test sending VPC in loadBalancerProps error', () => {
    const props = {
        loadBalancerProps: {
            vpc: { val: 'placeholder' }
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("Any existing VPC must be defined in the construct props (props.existingVpc). A VPC specified in the loadBalancerProps must be the same VPC");
});
test('WHen providing VPC in construct and resource props, the vpcId must match', () => {
    const fakeVpcOne = { vpcId: 'one' };
    const fakeVpcTwo = { vpcId: 'two' };
    const props = {
        existingVpc: fakeVpcOne,
        loadBalancerProps: {
            vpc: fakeVpcTwo
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("Any existing VPC must be defined in the construct props (props.existingVpc). A VPC specified in the loadBalancerProps must be the same VPC");
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxiLWxhbWJkYS50ZXN0LmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYWxiLWxhbWJkYS50ZXN0LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQTs7Ozs7Ozs7Ozs7R0FXRzs7QUFFSCxnQ0FBdUQ7QUFDdkQsaURBQWlEO0FBQ2pELDhEQUE4RDtBQUM5RCxtQ0FBbUM7QUFDbkMsMkRBQTJEO0FBQzNELHVEQUFrRDtBQUVsRCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO0lBQzFELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO1FBQ3pCLHNCQUFzQixFQUFFO1lBQ3RCO2dCQUNFLEdBQUcsRUFBRSw2QkFBNkI7Z0JBQ2xDLEtBQUssRUFBRSxPQUFPO2FBQ2Y7WUFDRDtnQkFDRSxHQUFHLEVBQUUsd0JBQXdCO2dCQUM3QixLQUFLLEVBQUUsTUFBTTthQUNkO1lBQ0Q7Z0JBQ0UsR0FBRyxFQUFFLHVCQUF1QjtnQkFDNUIsS0FBSyxFQUFFO29CQUNMLEdBQUcsRUFBRSxpQkFBaUI7aUJBQ3ZCO2FBQ0Y7WUFDRDtnQkFDRSxHQUFHLEVBQUUsdUJBQXVCO2dCQUM1QixLQUFLLEVBQUUsRUFBRTthQUNWO1NBQ0Y7S0FDRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE1BQU07S0FDakIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1FBQ3RFLFFBQVEsRUFBRSxPQUFPO0tBQ2xCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywwQ0FBMEMsRUFBRTtRQUN6RSxVQUFVLEVBQUUsUUFBUTtLQUNyQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsa0RBQWtELEVBQUUsRUFDbEYsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkRBQTZELEVBQUUsR0FBRyxFQUFFO0lBQ3ZFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixRQUFRLEVBQUUsTUFBTTtTQUNqQjtRQUNELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUUxQyxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsMkNBQTJDLEVBQUU7UUFDMUUsTUFBTSxFQUFFLGlCQUFpQjtLQUMxQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE1BQU07S0FDakIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSx1Q0FBdUMsRUFBRTtRQUMxRSxRQUFRLEVBQUUsT0FBTztLQUNsQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLGtEQUFrRCxFQUFFLEVBQ3RGLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywwQ0FBMEMsRUFBRTtRQUN6RSxVQUFVLEVBQUUsUUFBUTtLQUNyQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsZUFBZSxDQUFDLHVCQUF1QixFQUFFLENBQUMsQ0FBQyxDQUFDO0FBRXZELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHFEQUFxRCxFQUFFLEdBQUcsRUFBRTtJQUMvRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxlQUFlLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVuRCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ3JFLEdBQUcsRUFBRSxlQUFlO1FBQ3BCLGNBQWMsRUFBRSxJQUFJO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFFBQVEsRUFBRSxNQUFNO1NBQ2pCO1FBQ0QsdUJBQXVCLEVBQUUsV0FBVztRQUNwQyxXQUFXLEVBQUUsZUFBZTtRQUM1QixTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFMUMsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1FBQ3RFLFFBQVEsRUFBRSxNQUFNO0tBQ2pCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsdUNBQXVDLEVBQUU7UUFDMUUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFN0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxlQUFlLENBQUMsdUJBQXVCLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFckQsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDJDQUEyQyxFQUFFO1FBQzFFLE1BQU0sRUFBRSxpQkFBaUI7S0FDMUIsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMscURBQXFELEVBQUUsR0FBRyxFQUFFO0lBQy9ELE1BQU0sZ0JBQWdCLEdBQUcsWUFBWSxDQUFDO0lBRXRDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRW5ELE1BQU0sY0FBYyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDckUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7UUFDdkQsT0FBTyxFQUFFLGVBQWU7UUFDeEIsWUFBWSxFQUFFLGdCQUFnQjtRQUM5QixHQUFHLEVBQUUsZUFBZTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsaUJBQWlCLEVBQUUsY0FBYztRQUNqQyxhQUFhLEVBQUU7WUFDYixRQUFRLEVBQUUsTUFBTTtTQUNqQjtRQUNELFNBQVMsRUFBRSxJQUFJO1FBQ2YsV0FBVyxFQUFFLGVBQWU7S0FDN0IsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO0tBQzFCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsTUFBTTtLQUNqQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLHVDQUF1QyxFQUFFO1FBQzFFLFFBQVEsRUFBRSxPQUFPO0tBQ2xCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsa0RBQWtELEVBQUUsRUFDdEYsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1QkFBdUIsRUFBRTtRQUN0RCxZQUFZLEVBQUUsZ0JBQWdCO0tBQy9CLENBQUMsQ0FBQztBQUVMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBEQUEwRCxFQUFFLEdBQUcsRUFBRTtJQUNwRSxNQUFNLGdCQUFnQixHQUFHLFlBQVksQ0FBQztJQUV0QyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxlQUFlLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVuRCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ3JFLEdBQUcsRUFBRSxlQUFlO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sY0FBYyxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxLQUFLLEVBQUUsbUJBQW1CLEVBQUU7UUFDckUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7UUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7UUFDdkQsT0FBTyxFQUFFLGVBQWU7UUFDeEIsWUFBWSxFQUFFLGdCQUFnQjtRQUM5QixHQUFHLEVBQUUsZUFBZTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsaUJBQWlCLEVBQUUsY0FBYztRQUNqQyx1QkFBdUIsRUFBRSxXQUFXO1FBQ3BDLGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxTQUFTLEVBQUUsSUFBSTtRQUNmLFdBQVcsRUFBRSxlQUFlO0tBQzdCLENBQUM7SUFDRixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUUxQyxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsMkNBQTJDLEVBQUU7UUFDMUUsTUFBTSxFQUFFLFVBQVU7S0FDbkIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1FBQ3RFLFFBQVEsRUFBRSxNQUFNO0tBQ2pCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsT0FBTztLQUNsQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQzVCLGtEQUFrRCxFQUNsRCxFQUFFLENBQ0gsQ0FBQztJQUVGLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywwQ0FBMEMsRUFBRTtRQUN6RSxVQUFVLEVBQUUsUUFBUTtLQUNyQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUJBQXVCLEVBQUU7UUFDdEQsWUFBWSxFQUFFLGdCQUFnQjtLQUMvQixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFDMUQsTUFBTSxnQkFBZ0IsR0FBRyxZQUFZLENBQUM7SUFFdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtZQUN4QixZQUFZLEVBQUUsZ0JBQWdCO1NBQy9CO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsWUFBWSxFQUFFLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztTQUNoRTtRQUNELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUUxQyxNQUFNLFFBQVEsR0FBRyxxQkFBUSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUMzQyxRQUFRLENBQUMscUJBQXFCLENBQUMsMkNBQTJDLEVBQUU7UUFDMUUsTUFBTSxFQUFFLGlCQUFpQjtLQUMxQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE1BQU07S0FDakIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1FBQ3RFLFFBQVEsRUFBRSxPQUFPO0tBQ2xCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrREFBa0QsRUFBRSxFQUNsRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsMENBQTBDLEVBQUU7UUFDekUsVUFBVSxFQUFFLFFBQVE7S0FDckIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVCQUF1QixFQUFFO1FBQ3RELFlBQVksRUFBRSxnQkFBZ0I7S0FDL0IsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLGVBQWUsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFFL0MsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsdURBQXVELEVBQUUsR0FBRyxFQUFFO0lBQ2pFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLE1BQU0sY0FBYyxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRWpFLE1BQU0sV0FBVyxHQUFxQjtRQUNwQyxtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELFNBQVMsRUFBRTtZQUNULFVBQVUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxpQkFBaUIsQ0FBQyxZQUFZLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDO1lBQzdELFFBQVEsRUFBRSxFQUFFO1NBQ2I7UUFDRCxXQUFXLEVBQUUsY0FBYyxDQUFDLEdBQUc7UUFDL0IsdUJBQXVCLEVBQUUsY0FBYyxDQUFDLFlBQVk7UUFDcEQsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBRWhELE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO0tBQzFCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsTUFBTTtLQUNqQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxlQUFlLENBQUMsMENBQTBDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFeEUsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtEQUFrRCxFQUFFLEVBQ2xGLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxVQUFVLEVBQUU7WUFDVjtnQkFDRSxLQUFLLEVBQUUsY0FBYztnQkFDckIsaUJBQWlCLEVBQUU7b0JBQ2pCLE1BQU0sRUFBRTt3QkFDTixTQUFTO3FCQUNWO2lCQUNGO2FBQ0Y7U0FDRjtLQUNGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHNEQUFzRCxFQUFFLEdBQUcsRUFBRTtJQUNoRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsUUFBUSxFQUFFLE1BQU07U0FDakI7UUFDRCxTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsTUFBTSxjQUFjLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFakUsTUFBTSxXQUFXLEdBQXFCO1FBQ3BDLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsU0FBUyxFQUFFO1lBQ1QsVUFBVSxFQUFFLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFDN0QsUUFBUSxFQUFFLEVBQUU7U0FDYjtRQUNELFdBQVcsRUFBRSxjQUFjLENBQUMsR0FBRztRQUMvQix1QkFBdUIsRUFBRSxjQUFjLENBQUMsWUFBWTtRQUNwRCxTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsV0FBVyxDQUFDLENBQUM7SUFFaEQsTUFBTSxRQUFRLEdBQUcscUJBQVEsQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDM0MsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDJDQUEyQyxFQUFFO1FBQzFFLE1BQU0sRUFBRSxpQkFBaUI7S0FDMUIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLHVDQUF1QyxFQUFFO1FBQ3RFLFFBQVEsRUFBRSxNQUFNO0tBQ2pCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsdUNBQXVDLEVBQUU7UUFDMUUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxlQUFlLENBQUMsMENBQTBDLEVBQUUsQ0FBQyxDQUFDLENBQUM7SUFFeEUsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxrREFBa0QsRUFBRSxFQUN0RixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsMkNBQTJDLEVBQUU7UUFDMUUsVUFBVSxFQUFFO1lBQ1Y7Z0JBQ0UsS0FBSyxFQUFFLGNBQWM7Z0JBQ3JCLGlCQUFpQixFQUFFO29CQUNqQixNQUFNLEVBQUU7d0JBQ04sU0FBUztxQkFDVjtpQkFDRjthQUNGO1NBQ0Y7S0FDRixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnREFBZ0QsRUFBRSxHQUFHLEVBQUU7SUFDMUQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxXQUFXLEVBQUU7WUFDWCxlQUFlLEVBQUUsZ0JBQWdCO1NBQ2xDO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO0tBQzFCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsTUFBTTtLQUNqQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO1FBQ3BCLElBQUksRUFBRSxnQkFBZ0I7S0FDdkIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLGtEQUFrRCxFQUFFLEVBQ2xGLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLHlCQUF5QixFQUFFLEdBQUcsRUFBRTtJQUNuQyxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsWUFBWSxFQUFFLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztTQUNoRTtRQUNELFdBQVcsRUFBRTtZQUNYLGVBQWUsRUFBRSxnQkFBZ0I7U0FDbEM7UUFDRCxTQUFTLEVBQUUsSUFBSTtRQUNmLGdCQUFnQixFQUFFLEtBQUs7S0FDeEIsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxlQUFlLENBQUMsaUJBQWlCLEVBQUUsQ0FBQyxDQUFDLENBQUM7QUFFakQsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsa0JBQWtCLEVBQUUsR0FBRyxFQUFFO0lBQzVCLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLE1BQU0sTUFBTSxHQUFHLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRXpELE1BQU0sQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDNUIsTUFBTSxDQUFDLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNuQixNQUFNLENBQUMsTUFBTSxDQUFDLGNBQWMsQ0FBQyxDQUFDO0lBQzlCLE1BQU0sQ0FBQyxNQUFNLENBQUMsUUFBUSxDQUFDLENBQUM7QUFFMUIsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNEJBQTRCLEVBQUUsR0FBRyxFQUFFO0lBQ3RDLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixZQUFZLEVBQUUsQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1NBQ2hFO1FBQ0QsU0FBUyxFQUFFLElBQUk7UUFDZixpQkFBaUIsRUFBRTtZQUNqQixnQkFBZ0IsRUFBRSxhQUFhO1NBQ2hDO0tBQ0YsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO1FBQ3pCLElBQUksRUFBRSxhQUFhO0tBQ3BCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsTUFBTTtLQUNqQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrREFBa0QsRUFBRSxFQUNsRixDQUFDLENBQUM7QUFFTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw0QkFBNEIsRUFBRSxHQUFHLEVBQUU7SUFDdEMsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxTQUFTLEVBQUUsSUFBSTtRQUNmLHFCQUFxQixFQUFFO1lBQ3JCLFVBQVUsRUFBRSxhQUFhO1NBQzFCO0tBQ0YsQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBRTFDLE1BQU0sUUFBUSxHQUFHLHFCQUFRLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzNDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQywyQ0FBMkMsRUFBRTtRQUMxRSxNQUFNLEVBQUUsaUJBQWlCO0tBQzFCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyx1Q0FBdUMsRUFBRTtRQUN0RSxRQUFRLEVBQUUsTUFBTTtLQUNqQixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsdUNBQXVDLEVBQUU7UUFDdEUsUUFBUSxFQUFFLE9BQU87S0FDbEIsQ0FBQyxDQUFDO0lBRUgsUUFBUSxDQUFDLHFCQUFxQixDQUFDLDBDQUEwQyxFQUFFO1FBQ3pFLFVBQVUsRUFBRSxRQUFRO0tBQ3JCLENBQUMsQ0FBQztJQUVILFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQyxrREFBa0QsRUFBRSxFQUNsRixDQUFDLENBQUM7SUFFSCxRQUFRLENBQUMscUJBQXFCLENBQUMsaUJBQWlCLEVBQUU7UUFDaEQsVUFBVSxFQUFFLGFBQWE7S0FDMUIsQ0FBQyxDQUFDO0FBRUwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNENBQTRDLEVBQUUsR0FBRyxFQUFFO0lBQ3RELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUU7WUFDYixlQUFlLEVBQUUsQ0FBQyxzRkFBc0YsQ0FBQztTQUMxRztRQUNELFdBQVcsRUFBRTtZQUNYLGVBQWUsRUFBRSxnQkFBZ0I7U0FDbEM7UUFDRCxTQUFTLEVBQUUsSUFBSTtRQUNmLHFCQUFxQixFQUFFO1lBQ3JCLFVBQVUsRUFBRSxhQUFhO1NBQzFCO0tBQ0YsQ0FBQztJQUNGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVDLENBQUMsQ0FBQztJQUNGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsa0dBQWtHLENBQUMsQ0FBQztBQUMvSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxnRUFBZ0UsRUFBRSxHQUFHLEVBQUU7SUFDMUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxXQUFXLEVBQUU7WUFDWCxlQUFlLEVBQUUsZ0JBQWdCO1NBQ2xDO1FBQ0QsU0FBUyxFQUFFLElBQUk7UUFDZixnQkFBZ0IsRUFBRSxLQUFLO1FBQ3ZCLHFCQUFxQixFQUFFO1lBQ3JCLFVBQVUsRUFBRSxhQUFhO1NBQzFCO0tBQ0YsQ0FBQztJQUNGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVDLENBQUMsQ0FBQztJQUNGLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsbUZBQW1GLENBQUMsQ0FBQztBQUNoSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3Q0FBd0MsRUFBRSxHQUFHLEVBQUU7SUFDbEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7WUFDL0QsUUFBUSxFQUFFLE1BQU07U0FDakI7UUFDRCxTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDNUMsQ0FBQyxDQUFDO0lBQ0YsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO0FBQ3RFLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtJQUMxRCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUVGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVDLENBQUMsQ0FBQztJQUNGLFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLG9KQUFvSixDQUFDLENBQUM7QUFDakwsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsc0VBQXNFLEVBQUUsR0FBRyxFQUFFO0lBQ2hGLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRW5ELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDckUsR0FBRyxFQUFFLGVBQWU7UUFDcEIsY0FBYyxFQUFFLElBQUk7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsdUJBQXVCLEVBQUUsV0FBVztRQUNwQyxXQUFXLEVBQUUsZUFBZTtRQUM1QixTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDNUMsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsb0pBQW9KLENBQUMsQ0FBQztBQUNqTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxrRUFBa0UsRUFBRSxHQUFHLEVBQUU7SUFDNUUsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxTQUFTLEVBQUUsSUFBSTtLQUNoQixDQUFDO0lBQ0YsTUFBTSxjQUFjLEdBQUcsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFFakUsTUFBTSxXQUFXLEdBQXFCO1FBQ3BDLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsU0FBUyxFQUFFO1lBQ1QsVUFBVSxFQUFFLENBQUMsR0FBRyxDQUFDLGlCQUFpQixDQUFDLFlBQVksQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7WUFDN0QsUUFBUSxFQUFFLEVBQUU7U0FDYjtRQUNELGFBQWEsRUFBRTtZQUNiLFFBQVEsRUFBRSxPQUFPO1NBQ2xCO1FBQ0QsV0FBVyxFQUFFLGNBQWMsQ0FBQyxHQUFHO1FBQy9CLHVCQUF1QixFQUFFLGNBQWMsQ0FBQyxZQUFZO1FBQ3BELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxXQUFXLENBQUMsQ0FBQztJQUNsRCxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQywrRUFBK0UsQ0FBQyxDQUFDO0FBRTVHLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDhDQUE4QyxFQUFFLEdBQUcsRUFBRTtJQUN4RCxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsWUFBWSxFQUFFLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztTQUNoRTtRQUNELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixNQUFNLGNBQWMsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUVqRSxNQUFNLFdBQVcsR0FBcUI7UUFDcEMsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxXQUFXLEVBQUUsY0FBYyxDQUFDLEdBQUc7UUFDL0IsdUJBQXVCLEVBQUUsY0FBYyxDQUFDLFlBQVk7UUFDcEQsU0FBUyxFQUFFLElBQUk7S0FDaEIsQ0FBQztJQUNGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBQ2xELENBQUMsQ0FBQztJQUNGLFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLG1GQUFtRixDQUFDLENBQUM7QUFDaEgsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsZ0RBQWdELEVBQUUsR0FBRyxFQUFFO0lBQzFELE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxhQUFhLEVBQUUsRUFDZDtRQUNELFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM1QyxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyx3REFBd0QsQ0FBQyxDQUFDO0FBQ3JGLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLGdFQUFnRSxFQUFFLEdBQUcsRUFBRTtJQUMxRSxNQUFNLEtBQUssR0FBRyxJQUFJLEdBQUcsQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLFNBQVMsRUFBRTtRQUNoRCxHQUFHLEVBQUUsRUFBRSxPQUFPLEVBQUUsY0FBYyxFQUFFLE1BQU0sRUFBRSxXQUFXLEVBQUU7S0FDdEQsQ0FBQyxDQUFDO0lBRUgsTUFBTSxlQUFlLEdBQUcsUUFBUSxDQUFDLFVBQVUsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUVuRCxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyx1QkFBdUIsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1FBQ3JFLEdBQUcsRUFBRSxlQUFlO1FBQ3BCLGNBQWMsRUFBRSxJQUFJO0tBQ3JCLENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFFBQVEsRUFBRSxNQUFNO1NBQ2pCO1FBQ0QsdUJBQXVCLEVBQUUsV0FBVztRQUNwQyxXQUFXLEVBQUUsZUFBZTtRQUM1QixTQUFTLEVBQUUsSUFBSTtRQUNmLGlCQUFpQixFQUFFO1lBQ2pCLGNBQWMsRUFBRSxJQUFJO1NBQ3JCO0tBQ0YsQ0FBQztJQUNGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzVDLENBQUMsQ0FBQztJQUNGLFlBQVk7SUFDWixNQUFNLENBQUMsR0FBRyxDQUFDLENBQUMsWUFBWSxDQUFDLG9GQUFvRixDQUFDLENBQUM7QUFDakgsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkRBQTZELEVBQUUsR0FBRyxFQUFFO0lBQ3ZFLE1BQU0sS0FBSyxHQUFHLElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxFQUFFO1FBQ2hELEdBQUcsRUFBRSxFQUFFLE9BQU8sRUFBRSxjQUFjLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRTtLQUN0RCxDQUFDLENBQUM7SUFFSCxNQUFNLGVBQWUsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRW5ELE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDckUsR0FBRyxFQUFFLGVBQWU7UUFDcEIsY0FBYyxFQUFFLElBQUk7S0FDckIsQ0FBQyxDQUFDO0lBRUgsTUFBTSxLQUFLLEdBQXFCO1FBQzlCLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsUUFBUSxFQUFFLE1BQU07U0FDakI7UUFDRCx1QkFBdUIsRUFBRSxXQUFXO1FBQ3BDLFNBQVMsRUFBRSxJQUFJO0tBQ2hCLENBQUM7SUFDRixNQUFNLEdBQUcsR0FBRyxHQUFHLEVBQUU7UUFDZixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLFVBQVUsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUM1QyxDQUFDLENBQUM7SUFDRixZQUFZO0lBQ1osTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FDdEIsMkhBQTJILENBQUMsQ0FBQztBQUNqSSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx5Q0FBeUMsRUFBRSxHQUFHLEVBQUU7SUFDbkQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUNILE1BQU0sZUFBZSxHQUFHLFFBQVEsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLENBQUM7SUFFbkQsTUFBTSxjQUFjLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLEtBQUssRUFBRSxtQkFBbUIsRUFBRTtRQUNyRSxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztRQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztRQUN2RCxPQUFPLEVBQUUsZUFBZTtRQUN4QixZQUFZLEVBQUUsTUFBTTtRQUNwQixHQUFHLEVBQUUsZUFBZTtLQUNyQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsbUJBQW1CLEVBQUU7WUFDbkIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7WUFDbEQsT0FBTyxFQUFFLFFBQVEsQ0FBQyxxQ0FBcUM7WUFDdkQsT0FBTyxFQUFFLGVBQWU7U0FDekI7UUFDRCxpQkFBaUIsRUFBRSxjQUFjO1FBQ2pDLGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxTQUFTLEVBQUUsS0FBSztRQUNoQixXQUFXLEVBQUUsZUFBZTtLQUM3QixDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDakQsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsa0ZBQWtGLENBQUMsQ0FBQztBQUMvRyxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUVILE1BQU0sS0FBSyxHQUFxQjtRQUM5QixtQkFBbUIsRUFBRTtZQUNuQixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUNsRCxPQUFPLEVBQUUsUUFBUSxDQUFDLHFDQUFxQztZQUN2RCxPQUFPLEVBQUUsZUFBZTtTQUN6QjtRQUNELGFBQWEsRUFBRTtZQUNiLFlBQVksRUFBRSxDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxLQUFLLEVBQUUsV0FBVyxDQUFDLENBQUM7U0FDaEU7UUFDRCxTQUFTLEVBQUUsS0FBSztRQUNoQixRQUFRLEVBQUUsRUFBRTtRQUNaLFdBQVcsRUFBRSxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQztLQUN4QyxDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDakQsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsc0dBQXNHLENBQUMsQ0FBQztBQUNuSSxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyxzQ0FBc0MsRUFBRSxHQUFHLEVBQUU7SUFDaEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLEVBQUU7UUFDaEQsR0FBRyxFQUFFLEVBQUUsT0FBTyxFQUFFLGNBQWMsRUFBRSxNQUFNLEVBQUUsV0FBVyxFQUFFO0tBQ3RELENBQUMsQ0FBQztJQUNILE1BQU0sUUFBUSxHQUFHLFlBQVksQ0FBQztJQUU5QixNQUFNLFdBQVcsR0FBRyxRQUFRLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBRS9DLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxDQUFDLHVCQUF1QixDQUFDLEtBQUssRUFBRSxVQUFVLEVBQUU7UUFDckUsR0FBRyxFQUFFLFdBQVc7UUFDaEIsY0FBYyxFQUFFLElBQUk7UUFDcEIsZ0JBQWdCLEVBQUUsUUFBUTtLQUMzQixDQUFDLENBQUM7SUFFSCxNQUFNLEtBQUssR0FBcUI7UUFDOUIsV0FBVztRQUNYLG1CQUFtQixFQUFFO1lBQ25CLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1lBQ2xELE9BQU8sRUFBRSxRQUFRLENBQUMscUNBQXFDO1lBQ3ZELE9BQU8sRUFBRSxlQUFlO1NBQ3pCO1FBQ0QsYUFBYSxFQUFFO1lBQ2IsWUFBWSxFQUFFLENBQUMsUUFBUSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztTQUNoRTtRQUNELFNBQVMsRUFBRSxLQUFLO1FBQ2hCLFFBQVEsRUFBRSxFQUFFO1FBQ1osaUJBQWlCLEVBQUU7WUFDakIsZ0JBQWdCLEVBQUUsa0JBQWtCO1lBQ3BDLGNBQWMsRUFBRSxJQUFJO1NBQ3JCO1FBQ0QsdUJBQXVCLEVBQUUsV0FBVztLQUNyQyxDQUFDO0lBQ0YsTUFBTSxHQUFHLEdBQUcsR0FBRyxFQUFFO1FBQ2YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDakQsQ0FBQyxDQUFDO0lBQ0YsWUFBWTtJQUNaLE1BQU0sQ0FBQyxHQUFHLENBQUMsQ0FBQyxZQUFZLENBQUMsc0ZBQXNGLENBQUMsQ0FBQztBQUNuSCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw2Q0FBNkMsRUFBRSxHQUFHLEVBQUU7SUFDdkQsTUFBTSxLQUFLLEdBQUc7UUFDWixpQkFBaUIsRUFBRTtZQUNqQixHQUFHLEVBQUUsRUFBRSxHQUFHLEVBQUUsYUFBYSxFQUFFO1NBQzVCO0tBQ0YsQ0FBQztJQUVGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDaEMsQ0FBQyxDQUFDO0lBRUYsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyw0SUFBNEksQ0FBQyxDQUFDO0FBQ3pLLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDBFQUEwRSxFQUFFLEdBQUcsRUFBRTtJQUNwRixNQUFNLFVBQVUsR0FBRyxFQUFDLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQztJQUNsQyxNQUFNLFVBQVUsR0FBRyxFQUFDLEtBQUssRUFBRSxLQUFLLEVBQUMsQ0FBQztJQUVsQyxNQUFNLEtBQUssR0FBRztRQUNaLFdBQVcsRUFBRSxVQUFVO1FBQ3ZCLGlCQUFpQixFQUFFO1lBQ2pCLEdBQUcsRUFBRSxVQUFVO1NBQ2hCO0tBQ0YsQ0FBQztJQUVGLE1BQU0sR0FBRyxHQUFHLEdBQUcsRUFBRTtRQUNmLFFBQVEsQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDaEMsQ0FBQyxDQUFDO0lBRUYsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLFlBQVksQ0FBQyw0SUFBNEksQ0FBQyxDQUFDO0FBQ3pLLENBQUMsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgQWxiVG9MYW1iZGEsIEFsYlRvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBlbGIgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCJhd3MtY2RrLWxpYlwiO1xuaW1wb3J0ICogYXMgZGVmYXVsdHMgZnJvbSAnQGF3cy1zb2x1dGlvbnMtY29uc3RydWN0cy9jb3JlJztcbmltcG9ydCB7IFRlbXBsYXRlIH0gZnJvbSAnYXdzLWNkay1saWIvYXNzZXJ0aW9ucyc7XG5cbnRlc3QoJ1Rlc3QgbmV3IGxvYWQgYmFsYW5jZXIgYW5kIG5ldyBsYW1iZGEgZnVuY3Rpb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgU2NoZW1lOiAnaW50ZXJuZXQtZmFjaW5nJyxcbiAgICBMb2FkQmFsYW5jZXJBdHRyaWJ1dGVzOiBbXG4gICAgICB7XG4gICAgICAgIEtleTogXCJkZWxldGlvbl9wcm90ZWN0aW9uLmVuYWJsZWRcIixcbiAgICAgICAgVmFsdWU6IFwiZmFsc2VcIlxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgS2V5OiBcImFjY2Vzc19sb2dzLnMzLmVuYWJsZWRcIixcbiAgICAgICAgVmFsdWU6IFwidHJ1ZVwiXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBLZXk6IFwiYWNjZXNzX2xvZ3MuczMuYnVja2V0XCIsXG4gICAgICAgIFZhbHVlOiB7XG4gICAgICAgICAgUmVmOiBcInRlc3RvbmVFNkFDRkJCNlwiXG4gICAgICAgIH1cbiAgICAgIH0sXG4gICAgICB7XG4gICAgICAgIEtleTogXCJhY2Nlc3NfbG9ncy5zMy5wcmVmaXhcIixcbiAgICAgICAgVmFsdWU6IFwiXCJcbiAgICAgIH1cbiAgICBdLFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQUydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgIFRhcmdldFR5cGU6ICdsYW1iZGEnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lckNlcnRpZmljYXRlJywge1xuICB9KTtcblxufSk7XG5cbnRlc3QoJ1Rlc3QgbmV3IGxvYWQgYmFsYW5jZXIgYW5kIG5ldyBsYW1iZGEgZnVuY3Rpb24gZm9yIEhUVFAgYXBpJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIHByb3RvY29sOiAnSFRUUCdcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgU2NoZW1lOiAnaW50ZXJuZXQtZmFjaW5nJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQJ1xuICB9KTtcblxuICBkZWZhdWx0cy5leHBlY3ROb25leGlzdGVuY2Uoc3RhY2ssICdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUFMnXG4gIH0pO1xuXG4gIGRlZmF1bHRzLmV4cGVjdE5vbmV4aXN0ZW5jZShzdGFjaywgJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXJDZXJ0aWZpY2F0ZScsIHtcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgIFRhcmdldFR5cGU6ICdsYW1iZGEnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgMSk7XG5cbn0pO1xuXG50ZXN0KCdUZXN0IGV4aXN0aW5nIGxvYWQgYmFsYW5jZXIgYW5kIG5ldyBsYW1iZGEgZnVuY3Rpb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlc3RFeGlzdGluZ1ZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2spO1xuXG4gIGNvbnN0IGV4aXN0aW5nQWxiID0gbmV3IGVsYi5BcHBsaWNhdGlvbkxvYWRCYWxhbmNlcihzdGFjaywgJ3Rlc3QtYWxiJywge1xuICAgIHZwYzogdGVzdEV4aXN0aW5nVnBjLFxuICAgIGludGVybmV0RmFjaW5nOiB0cnVlXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgcHJvdG9jb2w6ICdIVFRQJ1xuICAgIH0sXG4gICAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGV4aXN0aW5nQWxiLFxuICAgIGV4aXN0aW5nVnBjOiB0ZXN0RXhpc3RpbmdWcGMsXG4gICAgcHVibGljQXBpOiB0cnVlLFxuICB9O1xuICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICBQcm90b2NvbDogJ0hUVFAnXG4gIH0pO1xuXG4gIGRlZmF1bHRzLmV4cGVjdE5vbmV4aXN0ZW5jZShzdGFjaywgJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQUydcbiAgfSk7XG5cbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVDMjo6VlBDJywgMSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgIFRhcmdldFR5cGU6ICdsYW1iZGEnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpMYW1iZGE6OkZ1bmN0aW9uJywgMSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXJcIiwge1xuICAgIFNjaGVtZTogXCJpbnRlcm5ldC1mYWNpbmdcIixcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCBuZXcgbG9hZCBiYWxhbmNlciBhbmQgZXhpc3RpbmcgbGFtYmRhIGZ1bmN0aW9uJywgKCkgPT4ge1xuICBjb25zdCB0ZXN0RnVuY3Rpb25OYW1lID0gJ2ZpeGVkLW5hbWUnO1xuXG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHRlc3RFeGlzdGluZ1ZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2spO1xuXG4gIGNvbnN0IGxhbWJkYUZ1bmN0aW9uID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ2V4aXN0aW5nLWZ1bmN0aW9uJywge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIGZ1bmN0aW9uTmFtZTogdGVzdEZ1bmN0aW9uTmFtZSxcbiAgICB2cGM6IHRlc3RFeGlzdGluZ1ZwY1xuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBleGlzdGluZ0xhbWJkYU9iajogbGFtYmRhRnVuY3Rpb24sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgcHJvdG9jb2w6ICdIVFRQJ1xuICAgIH0sXG4gICAgcHVibGljQXBpOiB0cnVlLFxuICAgIGV4aXN0aW5nVnBjOiB0ZXN0RXhpc3RpbmdWcGNcbiAgfTtcbiAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyJywge1xuICAgIFNjaGVtZTogJ2ludGVybmV0LWZhY2luZydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUCdcbiAgfSk7XG5cbiAgZGVmYXVsdHMuZXhwZWN0Tm9uZXhpc3RlbmNlKHN0YWNrLCAnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICBQcm90b2NvbDogJ0hUVFBTJ1xuICB9KTtcblxuICBkZWZhdWx0cy5leHBlY3ROb25leGlzdGVuY2Uoc3RhY2ssICdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyQ2VydGlmaWNhdGUnLCB7XG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpUYXJnZXRHcm91cCcsIHtcbiAgICBUYXJnZXRUeXBlOiAnbGFtYmRhJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6TGFtYmRhOjpGdW5jdGlvbicsIHtcbiAgICBGdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWVcbiAgfSk7XG5cbn0pO1xuXG50ZXN0KFwiVGVzdCBleGlzdGluZyBsb2FkIGJhbGFuY2VyIGFuZCBleGlzdGluZyBsYW1iZGEgZnVuY3Rpb25cIiwgKCkgPT4ge1xuICBjb25zdCB0ZXN0RnVuY3Rpb25OYW1lID0gXCJmaXhlZC1uYW1lXCI7XG5cbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogXCJ1cy1lYXN0LTFcIiB9LFxuICB9KTtcblxuICBjb25zdCB0ZXN0RXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuICBjb25zdCBleGlzdGluZ0FsYiA9IG5ldyBlbGIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIoc3RhY2ssIFwidGVzdC1hbGJcIiwge1xuICAgIHZwYzogdGVzdEV4aXN0aW5nVnBjLFxuICB9KTtcblxuICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssIFwiZXhpc3RpbmctZnVuY3Rpb25cIiwge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgaGFuZGxlcjogXCJpbmRleC5oYW5kbGVyXCIsXG4gICAgZnVuY3Rpb25OYW1lOiB0ZXN0RnVuY3Rpb25OYW1lLFxuICAgIHZwYzogdGVzdEV4aXN0aW5nVnBjLFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBleGlzdGluZ0xhbWJkYU9iajogbGFtYmRhRnVuY3Rpb24sXG4gICAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGV4aXN0aW5nQWxiLFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIGNlcnRpZmljYXRlczogW2RlZmF1bHRzLmdldEZha2VDZXJ0aWZpY2F0ZShzdGFjaywgXCJmYWtlLWNlcnRcIildLFxuICAgIH0sXG4gICAgcHVibGljQXBpOiB0cnVlLFxuICAgIGV4aXN0aW5nVnBjOiB0ZXN0RXhpc3RpbmdWcGMsXG4gIH07XG4gIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgXCJ0ZXN0LW9uZVwiLCBwcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlclwiLCB7XG4gICAgU2NoZW1lOiBcImludGVybmFsXCIsXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcyhcIkFXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXJcIiwge1xuICAgIFByb3RvY29sOiBcIkhUVFBcIixcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFwiQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lclwiLCB7XG4gICAgUHJvdG9jb2w6IFwiSFRUUFNcIixcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKFxuICAgIFwiQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lckNlcnRpZmljYXRlXCIsXG4gICAge31cbiAgKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwXCIsIHtcbiAgICBUYXJnZXRUeXBlOiBcImxhbWJkYVwiLFxuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoXCJBV1M6OkxhbWJkYTo6RnVuY3Rpb25cIiwge1xuICAgIEZ1bmN0aW9uTmFtZTogdGVzdEZ1bmN0aW9uTmFtZSxcbiAgfSk7XG59KTtcblxudGVzdCgnVGVzdCBuZXcgbG9hZCBiYWxhbmNlciBhbmQgbmV3IGxhbWJkYSBmdW5jdGlvbicsICgpID0+IHtcbiAgY29uc3QgdGVzdEZ1bmN0aW9uTmFtZSA9ICdmaXhlZC1uYW1lJztcblxuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBmdW5jdGlvbk5hbWU6IHRlc3RGdW5jdGlvbk5hbWUsXG4gICAgfSxcbiAgICBsaXN0ZW5lclByb3BzOiB7XG4gICAgICBjZXJ0aWZpY2F0ZXM6IFtkZWZhdWx0cy5nZXRGYWtlQ2VydGlmaWNhdGUoc3RhY2ssIFwiZmFrZS1jZXJ0XCIpXSxcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgfTtcbiAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyJywge1xuICAgIFNjaGVtZTogJ2ludGVybmV0LWZhY2luZydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUCdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUFMnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lckNlcnRpZmljYXRlJywge1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6VGFyZ2V0R3JvdXAnLCB7XG4gICAgVGFyZ2V0VHlwZTogJ2xhbWJkYSdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkxhbWJkYTo6RnVuY3Rpb24nLCB7XG4gICAgRnVuY3Rpb25OYW1lOiB0ZXN0RnVuY3Rpb25OYW1lXG4gIH0pO1xuXG4gIHRlbXBsYXRlLnJlc291cmNlQ291bnRJcygnQVdTOjpFQzI6OlZQQycsIDEpO1xuXG59KTtcblxudGVzdCgnVGVzdCBIVFRQUyBhZGRpbmcgMiBsYW1iZGEgdGFyZ2V0cywgc2Vjb25kIHdpdGggcnVsZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBjb25zdCBmaXJzdENvbnN0cnVjdCA9IG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG4gIGNvbnN0IHNlY29uZFByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgcnVsZVByb3BzOiB7XG4gICAgICBjb25kaXRpb25zOiBbZWxiLkxpc3RlbmVyQ29uZGl0aW9uLnBhdGhQYXR0ZXJucyhbXCIqYWRtaW4qXCJdKV0sXG4gICAgICBwcmlvcml0eTogMTBcbiAgICB9LFxuICAgIGV4aXN0aW5nVnBjOiBmaXJzdENvbnN0cnVjdC52cGMsXG4gICAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGZpcnN0Q29uc3RydWN0LmxvYWRCYWxhbmNlcixcbiAgICBwdWJsaWNBcGk6IHRydWVcbiAgfTtcbiAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC10d28nLCBzZWNvbmRQcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyJywge1xuICAgIFNjaGVtZTogJ2ludGVybmV0LWZhY2luZydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUCdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUFMnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpUYXJnZXRHcm91cCcsIHtcbiAgICBUYXJnZXRUeXBlOiAnbGFtYmRhJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5yZXNvdXJjZUNvdW50SXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6VGFyZ2V0R3JvdXAnLCAyKTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXJDZXJ0aWZpY2F0ZScsIHtcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyUnVsZScsIHtcbiAgICBDb25kaXRpb25zOiBbXG4gICAgICB7XG4gICAgICAgIEZpZWxkOiBcInBhdGgtcGF0dGVyblwiLFxuICAgICAgICBQYXRoUGF0dGVybkNvbmZpZzoge1xuICAgICAgICAgIFZhbHVlczogW1xuICAgICAgICAgICAgXCIqYWRtaW4qXCJcbiAgICAgICAgICBdXG4gICAgICAgIH1cbiAgICAgIH1cbiAgICBdLFxuICB9KTtcbn0pO1xuXG50ZXN0KCdUZXN0IEhUVFAgYWRkaW5nIDIgbGFtYmRhIHRhcmdldHMsIHNlY29uZCB3aXRoIHJ1bGVzJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIHByb3RvY29sOiAnSFRUUCdcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBjb25zdCBmaXJzdENvbnN0cnVjdCA9IG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG4gIGNvbnN0IHNlY29uZFByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgcnVsZVByb3BzOiB7XG4gICAgICBjb25kaXRpb25zOiBbZWxiLkxpc3RlbmVyQ29uZGl0aW9uLnBhdGhQYXR0ZXJucyhbXCIqYWRtaW4qXCJdKV0sXG4gICAgICBwcmlvcml0eTogMTBcbiAgICB9LFxuICAgIGV4aXN0aW5nVnBjOiBmaXJzdENvbnN0cnVjdC52cGMsXG4gICAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGZpcnN0Q29uc3RydWN0LmxvYWRCYWxhbmNlcixcbiAgICBwdWJsaWNBcGk6IHRydWVcbiAgfTtcbiAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC10d28nLCBzZWNvbmRQcm9wcyk7XG5cbiAgY29uc3QgdGVtcGxhdGUgPSBUZW1wbGF0ZS5mcm9tU3RhY2soc3RhY2spO1xuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TG9hZEJhbGFuY2VyJywge1xuICAgIFNjaGVtZTogJ2ludGVybmV0LWZhY2luZydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUCdcbiAgfSk7XG5cbiAgZGVmYXVsdHMuZXhwZWN0Tm9uZXhpc3RlbmNlKHN0YWNrLCAnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICBQcm90b2NvbDogJ0hUVFBTJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6VGFyZ2V0R3JvdXAnLCB7XG4gICAgVGFyZ2V0VHlwZTogJ2xhbWJkYSdcbiAgfSk7XG5cbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywgMik7XG5cbiAgZGVmYXVsdHMuZXhwZWN0Tm9uZXhpc3RlbmNlKHN0YWNrLCAnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lckNlcnRpZmljYXRlJywge1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXJSdWxlJywge1xuICAgIENvbmRpdGlvbnM6IFtcbiAgICAgIHtcbiAgICAgICAgRmllbGQ6IFwicGF0aC1wYXR0ZXJuXCIsXG4gICAgICAgIFBhdGhQYXR0ZXJuQ29uZmlnOiB7XG4gICAgICAgICAgVmFsdWVzOiBbXG4gICAgICAgICAgICBcIiphZG1pbipcIlxuICAgICAgICAgIF1cbiAgICAgICAgfVxuICAgICAgfVxuICAgIF0sXG4gIH0pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgbmV3IGxvYWQgYmFsYW5jZXIgYW5kIG5ldyBsYW1iZGEgZnVuY3Rpb24nLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHRhcmdldFByb3BzOiB7XG4gICAgICB0YXJnZXRHcm91cE5hbWU6ICdkaWZmZXJlbnQtbmFtZSdcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgU2NoZW1lOiAnaW50ZXJuZXQtZmFjaW5nJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXInLCB7XG4gICAgUHJvdG9jb2w6ICdIVFRQUydcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OlRhcmdldEdyb3VwJywge1xuICAgIFRhcmdldFR5cGU6ICdsYW1iZGEnLFxuICAgIE5hbWU6ICdkaWZmZXJlbnQtbmFtZSdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyQ2VydGlmaWNhdGUnLCB7XG4gIH0pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgbG9nZ2luZyB0dXJuZWQgb2ZmJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIGNlcnRpZmljYXRlczogW2RlZmF1bHRzLmdldEZha2VDZXJ0aWZpY2F0ZShzdGFjaywgXCJmYWtlLWNlcnRcIildXG4gICAgfSxcbiAgICB0YXJnZXRQcm9wczoge1xuICAgICAgdGFyZ2V0R3JvdXBOYW1lOiAnZGlmZmVyZW50LW5hbWUnXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IHRydWUsXG4gICAgbG9nQWxiQWNjZXNzTG9nczogZmFsc2UsXG4gIH07XG4gIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUucmVzb3VyY2VDb3VudElzKCdBV1M6OlMzOjpCdWNrZXQnLCAwKTtcblxufSk7XG5cbnRlc3QoJ0NoZWNrIFByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBjb25zdCB0YXJnZXQgPSBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBleHBlY3QodGFyZ2V0LmxvYWRCYWxhbmNlcik7XG4gIGV4cGVjdCh0YXJnZXQudnBjKTtcbiAgZXhwZWN0KHRhcmdldC5sYW1iZGFGdW5jdGlvbik7XG4gIGV4cGVjdCh0YXJnZXQubGlzdGVuZXIpO1xuXG59KTtcblxudGVzdCgnVGVzdCBjdXN0b20gQUxCIHByb3BlcnRpZXMnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBsb2FkQmFsYW5jZXJQcm9wczoge1xuICAgICAgbG9hZEJhbGFuY2VyTmFtZTogJ2N1c3RvbS1uYW1lJyxcbiAgICB9XG4gIH07XG4gIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG4gIGNvbnN0IHRlbXBsYXRlID0gVGVtcGxhdGUuZnJvbVN0YWNrKHN0YWNrKTtcbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6OkxvYWRCYWxhbmNlcicsIHtcbiAgICBTY2hlbWU6ICdpbnRlcm5ldC1mYWNpbmcnLFxuICAgIE5hbWU6ICdjdXN0b20tbmFtZScsXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICBQcm90b2NvbDogJ0hUVFAnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMaXN0ZW5lcicsIHtcbiAgICBQcm90b2NvbDogJ0hUVFBTJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6VGFyZ2V0R3JvdXAnLCB7XG4gICAgVGFyZ2V0VHlwZTogJ2xhbWJkYSdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyQ2VydGlmaWNhdGUnLCB7XG4gIH0pO1xuXG59KTtcblxudGVzdCgnVGVzdCBjdXN0b20gbG9nZ2luZyBidWNrZXQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBhbGJMb2dnaW5nQnVja2V0UHJvcHM6IHtcbiAgICAgIGJ1Y2tldE5hbWU6ICdjdXN0b20tbmFtZScsXG4gICAgfVxuICB9O1xuICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCB0ZW1wbGF0ZSA9IFRlbXBsYXRlLmZyb21TdGFjayhzdGFjayk7XG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpMb2FkQmFsYW5jZXInLCB7XG4gICAgU2NoZW1lOiAnaW50ZXJuZXQtZmFjaW5nJyxcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUCdcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OkVsYXN0aWNMb2FkQmFsYW5jaW5nVjI6Okxpc3RlbmVyJywge1xuICAgIFByb3RvY29sOiAnSFRUUFMnXG4gIH0pO1xuXG4gIHRlbXBsYXRlLmhhc1Jlc291cmNlUHJvcGVydGllcygnQVdTOjpFbGFzdGljTG9hZEJhbGFuY2luZ1YyOjpUYXJnZXRHcm91cCcsIHtcbiAgICBUYXJnZXRUeXBlOiAnbGFtYmRhJ1xuICB9KTtcblxuICB0ZW1wbGF0ZS5oYXNSZXNvdXJjZVByb3BlcnRpZXMoJ0FXUzo6RWxhc3RpY0xvYWRCYWxhbmNpbmdWMjo6TGlzdGVuZXJDZXJ0aWZpY2F0ZScsIHtcbiAgfSk7XG5cbiAgdGVtcGxhdGUuaGFzUmVzb3VyY2VQcm9wZXJ0aWVzKCdBV1M6OlMzOjpCdWNrZXQnLCB7XG4gICAgQnVja2V0TmFtZTogJ2N1c3RvbS1uYW1lJ1xuICB9KTtcblxufSk7XG5cbnRlc3QoJ1Rlc3QgcHJvdmlkaW5nIGNlcnRpZmljYXRlQXJucyBpcyBhbiBlcnJvcicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG5cbiAgY29uc3QgcHJvcHM6IEFsYlRvTGFtYmRhUHJvcHMgPSB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gICAgfSxcbiAgICBsaXN0ZW5lclByb3BzOiB7XG4gICAgICBjZXJ0aWZpY2F0ZUFybnM6IFsnXHRhcm46YXdzOmFjbTp1cy1lYXN0LTE6MTIzNDU2Nzg5MDEyOmNlcnRpZmljYXRlLzExMTEyMjIyLTMzMzMtMTIzNC0xMjM0LTEyMzQ1Njc4OTAxMiddXG4gICAgfSxcbiAgICB0YXJnZXRQcm9wczoge1xuICAgICAgdGFyZ2V0R3JvdXBOYW1lOiAnZGlmZmVyZW50LW5hbWUnXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IHRydWUsXG4gICAgYWxiTG9nZ2luZ0J1Y2tldFByb3BzOiB7XG4gICAgICBidWNrZXROYW1lOiAnc2hvdWxkLWZhaWwnXG4gICAgfVxuICB9O1xuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG4gIH07XG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigvY2VydGlmaWNhdGVBcm5zIGlzIGRlcHJlY2F0ZWQuIFBsZWFzZSBzdXBwbHkgY2VydGlmaWNhdGVzIHVzaW5nIHByb3BzLmxpc3RlbmVyUHJvcHMuY2VydGlmaWNhdGVzLyk7XG59KTtcblxudGVzdCgnVGVzdCBsb2dnaW5nIG9mZiB3aXRoIGxvZ0J1Y2tldFByb3BlcnRpZXMgcHJvdmlkZWQgaXMgYW4gZXJyb3InLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHRhcmdldFByb3BzOiB7XG4gICAgICB0YXJnZXRHcm91cE5hbWU6ICdkaWZmZXJlbnQtbmFtZSdcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBsb2dBbGJBY2Nlc3NMb2dzOiBmYWxzZSxcbiAgICBhbGJMb2dnaW5nQnVja2V0UHJvcHM6IHtcbiAgICAgIGJ1Y2tldE5hbWU6ICdzaG91bGQtZmFpbCdcbiAgICB9XG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcbiAgfTtcbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKC9FcnJvciAtIElmIGxvZ0FsYkFjY2Vzc0xvZ3MgaXMgZmFsc2UsIHN1cHBseWluZyBhbGJMb2dnaW5nQnVja2V0UHJvcHMgaXMgaW52YWxpZC4vKTtcbn0pO1xuXG50ZXN0KCdUZXN0IGNlcnRpZmljYXRlIHdpdGggSFRUUCBpcyBhbiBlcnJvcicsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG5cbiAgY29uc3QgcHJvcHM6IEFsYlRvTGFtYmRhUHJvcHMgPSB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYCksXG4gICAgICBydW50aW1lOiBkZWZhdWx0cy5DT01NRVJDSUFMX1JFR0lPTl9MQU1CREFfTk9ERV9SVU5USU1FLFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInXG4gICAgfSxcbiAgICBsaXN0ZW5lclByb3BzOiB7XG4gICAgICBjZXJ0aWZpY2F0ZXM6IFtkZWZhdWx0cy5nZXRGYWtlQ2VydGlmaWNhdGUoc3RhY2ssIFwiZmFrZS1jZXJ0XCIpXSxcbiAgICAgIHByb3RvY29sOiAnSFRUUCcsXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IHRydWVcbiAgfTtcbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuICB9O1xuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoL0hUVFAgbGlzdGVuZXJzIGNhbm5vdCB1c2UgYSBjZXJ0aWZpY2F0ZS8pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgbmV3IEFMQiB3aXRoIG5vIGxpc3RlbmVyUHJvcHMgaXMgYW4gZXJyb3InLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgcHVibGljQXBpOiB0cnVlXG4gIH07XG5cbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuICB9O1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKC9XaGVuIGFkZGluZyB0aGUgZmlyc3QgbGlzdGVuZXIgYW5kIHRhcmdldCB0byBhIGxvYWQgYmFsYW5jZXIsIGxpc3RlbmVyUHJvcHMgbXVzdCBiZSBzcGVjaWZpZWQgYW5kIGluY2x1ZGUgYXQgbGVhc3QgYSBjZXJ0aWZpY2F0ZSBvciBwcm90b2NvbDogSFRUUC8pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgZXhpc3RpbmcgQUxCIHdpdGggbm8gbGlzdGVuZXIgd2l0aCBubyBsaXN0ZW5lclByb3BzIGlzIGFuIGVycm9yJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCB0ZXN0RXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuICBjb25zdCBleGlzdGluZ0FsYiA9IG5ldyBlbGIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIoc3RhY2ssICd0ZXN0LWFsYicsIHtcbiAgICB2cGM6IHRlc3RFeGlzdGluZ1ZwYyxcbiAgICBpbnRlcm5ldEZhY2luZzogdHJ1ZVxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBleGlzdGluZ0FsYixcbiAgICBleGlzdGluZ1ZwYzogdGVzdEV4aXN0aW5nVnBjLFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgfTtcbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuICB9O1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKC9XaGVuIGFkZGluZyB0aGUgZmlyc3QgbGlzdGVuZXIgYW5kIHRhcmdldCB0byBhIGxvYWQgYmFsYW5jZXIsIGxpc3RlbmVyUHJvcHMgbXVzdCBiZSBzcGVjaWZpZWQgYW5kIGluY2x1ZGUgYXQgbGVhc3QgYSBjZXJ0aWZpY2F0ZSBvciBwcm90b2NvbDogSFRUUC8pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgZXhpc3RpbmcgQUxCIHdpdGggYSBsaXN0ZW5lciB3aXRoIGxpc3RlbmVyUHJvcHMgaXMgYW4gZXJyb3InLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBjb25zdCBmaXJzdENvbnN0cnVjdCA9IG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtb25lJywgcHJvcHMpO1xuXG4gIGNvbnN0IHNlY29uZFByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgcnVsZVByb3BzOiB7XG4gICAgICBjb25kaXRpb25zOiBbZWxiLkxpc3RlbmVyQ29uZGl0aW9uLnBhdGhQYXR0ZXJucyhbXCIqYWRtaW4qXCJdKV0sXG4gICAgICBwcmlvcml0eTogMTBcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIHByb3RvY29sOiAnSFRUUFMnXG4gICAgfSxcbiAgICBleGlzdGluZ1ZwYzogZmlyc3RDb25zdHJ1Y3QudnBjLFxuICAgIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBmaXJzdENvbnN0cnVjdC5sb2FkQmFsYW5jZXIsXG4gICAgcHVibGljQXBpOiB0cnVlXG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LXR3bycsIHNlY29uZFByb3BzKTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigvVGhpcyBsb2FkIGJhbGFuY2VyIGFscmVhZHkgaGFzIGEgbGlzdGVuZXIsIGxpc3RlbmVyUHJvcHMgbWF5IG5vdCBiZSBzcGVjaWZpZWQvKTtcblxufSk7XG5cbnRlc3QoJ1Rlc3Qgc2Vjb25kIHRhcmdldCB3aXRoIG5vIHJ1bGVzIGlzIGFuIGVycm9yJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIGNlcnRpZmljYXRlczogW2RlZmF1bHRzLmdldEZha2VDZXJ0aWZpY2F0ZShzdGFjaywgXCJmYWtlLWNlcnRcIildXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IHRydWVcbiAgfTtcbiAgY29uc3QgZmlyc3RDb25zdHJ1Y3QgPSBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcblxuICBjb25zdCBzZWNvbmRQcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGV4aXN0aW5nVnBjOiBmaXJzdENvbnN0cnVjdC52cGMsXG4gICAgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmo6IGZpcnN0Q29uc3RydWN0LmxvYWRCYWxhbmNlcixcbiAgICBwdWJsaWNBcGk6IHRydWVcbiAgfTtcbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ3Rlc3QtdHdvJywgc2Vjb25kUHJvcHMpO1xuICB9O1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKC9XaGVuIGFkZGluZyBhIHNlY29uZCB0YXJnZXQgdG8gYW4gZXhpc3RpbmcgbGlzdGVuZXIsIHRoZXJlIG11c3QgYmUgcnVsZXMgcHJvdmlkZWQvKTtcbn0pO1xuXG50ZXN0KCdUZXN0IG5vIGNlcnQgZm9yIGFuIEhUVFBTIGxpc3RlbmVyIGlzIGFuIGVycm9yJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICB9LFxuICAgIHB1YmxpY0FwaTogdHJ1ZVxuICB9O1xuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgbmV3IEFsYlRvTGFtYmRhKHN0YWNrLCAndGVzdC1vbmUnLCBwcm9wcyk7XG4gIH07XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoL0EgbGlzdGVuZXIgdXNpbmcgSFRUUFMgcHJvdG9jb2wgcmVxdWlyZXMgYSBjZXJ0aWZpY2F0ZS8pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmogYW5kIGxvYWRCYWxhbmNlclByb3BzIGlzIGFuIGVycm9yJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCB0ZXN0RXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuICBjb25zdCBleGlzdGluZ0FsYiA9IG5ldyBlbGIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIoc3RhY2ssICd0ZXN0LWFsYicsIHtcbiAgICB2cGM6IHRlc3RFeGlzdGluZ1ZwYyxcbiAgICBpbnRlcm5ldEZhY2luZzogdHJ1ZVxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIHByb3RvY29sOiAnSFRUUCdcbiAgICB9LFxuICAgIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBleGlzdGluZ0FsYixcbiAgICBleGlzdGluZ1ZwYzogdGVzdEV4aXN0aW5nVnBjLFxuICAgIHB1YmxpY0FwaTogdHJ1ZSxcbiAgICBsb2FkQmFsYW5jZXJQcm9wczoge1xuICAgICAgaW50ZXJuZXRGYWNpbmc6IHRydWVcbiAgICB9XG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcigvRXJyb3IgLSBFaXRoZXIgcHJvdmlkZSBsb2FkQmFsYW5jZXJQcm9wcyBvciBleGlzdGluZ0xvYWRCYWxhbmNlck9iaiwgYnV0IG5vdCBib3RoLi8pO1xufSk7XG5cbnRlc3QoJ1Rlc3QgZXhpc3RpbmdMb2FkQmFsYW5jZXJPYmogYW5kIG5vIGV4aXN0aW5nVnBjIGlzIGFuIGVycm9yJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcblxuICBjb25zdCB0ZXN0RXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuICBjb25zdCBleGlzdGluZ0FsYiA9IG5ldyBlbGIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIoc3RhY2ssICd0ZXN0LWFsYicsIHtcbiAgICB2cGM6IHRlc3RFeGlzdGluZ1ZwYyxcbiAgICBpbnRlcm5ldEZhY2luZzogdHJ1ZVxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIHByb3RvY29sOiAnSFRUUCdcbiAgICB9LFxuICAgIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqOiBleGlzdGluZ0FsYixcbiAgICBwdWJsaWNBcGk6IHRydWUsXG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICd0ZXN0LW9uZScsIHByb3BzKTtcbiAgfTtcbiAgLy8gQXNzZXJ0aW9uXG4gIGV4cGVjdChhcHApLnRvVGhyb3dFcnJvcihcbiAgICAvQW4gZXhpc3RpbmcgQUxCIGlzIGFscmVhZHkgaW4gYSBWUEMsIHRoYXQgVlBDIG11c3QgYmUgcHJvdmlkZWQgaW4gcHJvcHMuZXhpc3RpbmdWcGMgZm9yIHRoZSByZXN0IG9mIHRoZSBjb25zdHJ1Y3QgdG8gdXNlLi8pO1xufSk7XG5cbnRlc3QoJ0NvbmZpcm0gdGhhdCBDaGVja0xhbWJkYVByb3BzIGlzIGNhbGxlZCcsICgpID0+IHtcbiAgY29uc3Qgc3RhY2sgPSBuZXcgY2RrLlN0YWNrKHVuZGVmaW5lZCwgdW5kZWZpbmVkLCB7XG4gICAgZW52OiB7IGFjY291bnQ6IFwiMTIzNDU2Nzg5MDEyXCIsIHJlZ2lvbjogJ3VzLWVhc3QtMScgfSxcbiAgfSk7XG4gIGNvbnN0IHRlc3RFeGlzdGluZ1ZwYyA9IGRlZmF1bHRzLmdldFRlc3RWcGMoc3RhY2spO1xuXG4gIGNvbnN0IGxhbWJkYUZ1bmN0aW9uID0gbmV3IGxhbWJkYS5GdW5jdGlvbihzdGFjaywgJ2V4aXN0aW5nLWZ1bmN0aW9uJywge1xuICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgIGZ1bmN0aW9uTmFtZTogJ25hbWUnLFxuICAgIHZwYzogdGVzdEV4aXN0aW5nVnBjXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgZXhpc3RpbmdMYW1iZGFPYmo6IGxhbWJkYUZ1bmN0aW9uLFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIGNlcnRpZmljYXRlczogW2RlZmF1bHRzLmdldEZha2VDZXJ0aWZpY2F0ZShzdGFjaywgXCJmYWtlLWNlcnRcIildXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IGZhbHNlLFxuICAgIGV4aXN0aW5nVnBjOiB0ZXN0RXhpc3RpbmdWcGMsXG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICduZXctY29uc3RydWN0JywgcHJvcHMpO1xuICB9O1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKCdFcnJvciAtIEVpdGhlciBwcm92aWRlIGxhbWJkYUZ1bmN0aW9uUHJvcHMgb3IgZXhpc3RpbmdMYW1iZGFPYmosIGJ1dCBub3QgYm90aC5cXG4nKTtcbn0pO1xuXG50ZXN0KCdDb25maXJtIHRoYXQgQ2hlY2tWcGNQcm9wcyBpcyBjYWxsZWQnLCAoKSA9PiB7XG4gIGNvbnN0IHN0YWNrID0gbmV3IGNkay5TdGFjayh1bmRlZmluZWQsIHVuZGVmaW5lZCwge1xuICAgIGVudjogeyBhY2NvdW50OiBcIjEyMzQ1Njc4OTAxMlwiLCByZWdpb246ICd1cy1lYXN0LTEnIH0sXG4gIH0pO1xuXG4gIGNvbnN0IHByb3BzOiBBbGJUb0xhbWJkYVByb3BzID0ge1xuICAgIGxhbWJkYUZ1bmN0aW9uUHJvcHM6IHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgcnVudGltZTogZGVmYXVsdHMuQ09NTUVSQ0lBTF9SRUdJT05fTEFNQkRBX05PREVfUlVOVElNRSxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJ1xuICAgIH0sXG4gICAgbGlzdGVuZXJQcm9wczoge1xuICAgICAgY2VydGlmaWNhdGVzOiBbZGVmYXVsdHMuZ2V0RmFrZUNlcnRpZmljYXRlKHN0YWNrLCBcImZha2UtY2VydFwiKV1cbiAgICB9LFxuICAgIHB1YmxpY0FwaTogZmFsc2UsXG4gICAgdnBjUHJvcHM6IHt9LFxuICAgIGV4aXN0aW5nVnBjOiBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKSxcbiAgfTtcbiAgY29uc3QgYXBwID0gKCkgPT4ge1xuICAgIG5ldyBBbGJUb0xhbWJkYShzdGFjaywgJ25ldy1jb25zdHJ1Y3QnLCBwcm9wcyk7XG4gIH07XG4gIC8vIEFzc2VydGlvblxuICBleHBlY3QoYXBwKS50b1Rocm93RXJyb3IoJ0Vycm9yIC0gRWl0aGVyIHByb3ZpZGUgYW4gZXhpc3RpbmdWcGMgb3Igc29tZSBjb21iaW5hdGlvbiBvZiBkZXBsb3lWcGMgYW5kIHZwY1Byb3BzLCBidXQgbm90IGJvdGguXFxuJyk7XG59KTtcblxudGVzdCgnQ29uZmlybSB0aGF0IENoZWNrQWxiUHJvcHMgaXMgY2FsbGVkJywgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBjZGsuU3RhY2sodW5kZWZpbmVkLCB1bmRlZmluZWQsIHtcbiAgICBlbnY6IHsgYWNjb3VudDogXCIxMjM0NTY3ODkwMTJcIiwgcmVnaW9uOiAndXMtZWFzdC0xJyB9LFxuICB9KTtcbiAgY29uc3QgdGVzdE5hbWUgPSAndGVzdC12YWx1ZSc7XG5cbiAgY29uc3QgZXhpc3RpbmdWcGMgPSBkZWZhdWx0cy5nZXRUZXN0VnBjKHN0YWNrKTtcblxuICBjb25zdCBleGlzdGluZ0FsYiA9IG5ldyBlbGIuQXBwbGljYXRpb25Mb2FkQmFsYW5jZXIoc3RhY2ssICd0ZXN0LWFsYicsIHtcbiAgICB2cGM6IGV4aXN0aW5nVnBjLFxuICAgIGludGVybmV0RmFjaW5nOiB0cnVlLFxuICAgIGxvYWRCYWxhbmNlck5hbWU6IHRlc3ROYW1lLFxuICB9KTtcblxuICBjb25zdCBwcm9wczogQWxiVG9MYW1iZGFQcm9wcyA9IHtcbiAgICBleGlzdGluZ1ZwYyxcbiAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoYCR7X19kaXJuYW1lfS9sYW1iZGFgKSxcbiAgICAgIHJ1bnRpbWU6IGRlZmF1bHRzLkNPTU1FUkNJQUxfUkVHSU9OX0xBTUJEQV9OT0RFX1JVTlRJTUUsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcidcbiAgICB9LFxuICAgIGxpc3RlbmVyUHJvcHM6IHtcbiAgICAgIGNlcnRpZmljYXRlczogW2RlZmF1bHRzLmdldEZha2VDZXJ0aWZpY2F0ZShzdGFjaywgXCJmYWtlLWNlcnRcIildXG4gICAgfSxcbiAgICBwdWJsaWNBcGk6IGZhbHNlLFxuICAgIHZwY1Byb3BzOiB7fSxcbiAgICBsb2FkQmFsYW5jZXJQcm9wczoge1xuICAgICAgbG9hZEJhbGFuY2VyTmFtZTogJ25ldy1sb2FkYmFsYW5jZXInLFxuICAgICAgaW50ZXJuZXRGYWNpbmc6IHRydWUsXG4gICAgfSxcbiAgICBleGlzdGluZ0xvYWRCYWxhbmNlck9iajogZXhpc3RpbmdBbGIsXG4gIH07XG4gIGNvbnN0IGFwcCA9ICgpID0+IHtcbiAgICBuZXcgQWxiVG9MYW1iZGEoc3RhY2ssICduZXctY29uc3RydWN0JywgcHJvcHMpO1xuICB9O1xuICAvLyBBc3NlcnRpb25cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKCdFcnJvciAtIEVpdGhlciBwcm92aWRlIGxvYWRCYWxhbmNlclByb3BzIG9yIGV4aXN0aW5nTG9hZEJhbGFuY2VyT2JqLCBidXQgbm90IGJvdGguXFxuJyk7XG59KTtcblxudGVzdCgnVGVzdCBzZW5kaW5nIFZQQyBpbiBsb2FkQmFsYW5jZXJQcm9wcyBlcnJvcicsICgpID0+IHtcbiAgY29uc3QgcHJvcHMgPSB7XG4gICAgbG9hZEJhbGFuY2VyUHJvcHM6IHtcbiAgICAgIHZwYzogeyB2YWw6ICdwbGFjZWhvbGRlcicgfVxuICAgIH1cbiAgfTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgZGVmYXVsdHMuQ2hlY2tBbGJQcm9wcyhwcm9wcyk7XG4gIH07XG5cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKFwiQW55IGV4aXN0aW5nIFZQQyBtdXN0IGJlIGRlZmluZWQgaW4gdGhlIGNvbnN0cnVjdCBwcm9wcyAocHJvcHMuZXhpc3RpbmdWcGMpLiBBIFZQQyBzcGVjaWZpZWQgaW4gdGhlIGxvYWRCYWxhbmNlclByb3BzIG11c3QgYmUgdGhlIHNhbWUgVlBDXCIpO1xufSk7XG5cbnRlc3QoJ1dIZW4gcHJvdmlkaW5nIFZQQyBpbiBjb25zdHJ1Y3QgYW5kIHJlc291cmNlIHByb3BzLCB0aGUgdnBjSWQgbXVzdCBtYXRjaCcsICgpID0+IHtcbiAgY29uc3QgZmFrZVZwY09uZSA9IHt2cGNJZDogJ29uZSd9O1xuICBjb25zdCBmYWtlVnBjVHdvID0ge3ZwY0lkOiAndHdvJ307XG5cbiAgY29uc3QgcHJvcHMgPSB7XG4gICAgZXhpc3RpbmdWcGM6IGZha2VWcGNPbmUsXG4gICAgbG9hZEJhbGFuY2VyUHJvcHM6IHtcbiAgICAgIHZwYzogZmFrZVZwY1R3b1xuICAgIH1cbiAgfTtcblxuICBjb25zdCBhcHAgPSAoKSA9PiB7XG4gICAgZGVmYXVsdHMuQ2hlY2tBbGJQcm9wcyhwcm9wcyk7XG4gIH07XG5cbiAgZXhwZWN0KGFwcCkudG9UaHJvd0Vycm9yKFwiQW55IGV4aXN0aW5nIFZQQyBtdXN0IGJlIGRlZmluZWQgaW4gdGhlIGNvbnN0cnVjdCBwcm9wcyAocHJvcHMuZXhpc3RpbmdWcGMpLiBBIFZQQyBzcGVjaWZpZWQgaW4gdGhlIGxvYWRCYWxhbmNlclByb3BzIG11c3QgYmUgdGhlIHNhbWUgVlBDXCIpO1xufSk7XG4iXX0=