"""
Phase 2: JS generator for Electron preload and runtime stub.
- Consumes Python-side API schema (from dars.desktop.api.get_schema()).
- Emits:
  * preload.js content (uses contextBridge to expose safe API)
  * stub.js content (runtime helper that calls window.DarsDesktopAPI)
Optionally writes files when paths are provided.
"""
from __future__ import annotations
from typing import Dict, Optional

PREAMBLE = """// Auto-generated by Dars (Phase 2)
// Do not edit manually; generation will overwrite this file.
"""


def _gen_preload_api_impl(schema: Dict[str, Dict[str, str]]) -> str:
    lines = [
        "const { contextBridge, ipcRenderer } = require('electron');",
        "",
        "// Build a safe API surface based on provided schema",
        "const api = {};",
    ]
    for ns, methods in schema.items():
        lines.append(f"api['{ns}'] = {{}};")
        for m in methods.keys():
            # Forward all calls to ipcRenderer.invoke with a channel name
            channel = f"dars::{ns}::{m}"
            lines.append(
                f"api['{ns}']['{m}'] = (...args) => ipcRenderer.invoke('{channel}', ...args);"
            )
    # No console override needed as we use DevTools in dev mode
    lines.append("")
    lines.append("contextBridge.exposeInMainWorld('DarsDesktopAPI', api);")
    return "\n".join(lines)


def generate_preload_js(schema: Dict[str, Dict[str, str]]) -> str:
    return PREAMBLE + _gen_preload_api_impl(schema) + "\n"


def _gen_stub_impl(schema: Dict[str, Dict[str, str]]) -> str:
    # Build a runtime stub that prefers a preload-exposed `DarsDesktopAPI`,
    # but falls back to using a lightweight `DarsIPC.invoke` bridge if present
    # (some dev preload scripts expose `DarsIPC.invoke`). If neither exists,
    # the methods will throw a helpful error when called.
    import json
    # Convert schema to a JS-friendly shape: namespace -> [method, ...]
    js_schema = {ns: list(methods.keys()) for ns, methods in schema.items()}
    js_schema_literal = json.dumps(js_schema)

    lines = []
    lines.append("// Runtime stub for Renderer (Dars) to call into preload-exposed API")
    lines.append(f"const _DARS_SCHEMA = {js_schema_literal};")
    lines.append("const _make_api_from_schema = (schema) => {")
    lines.append("  const api = {};")
    lines.append("  for (const ns of Object.keys(schema)) {")
    lines.append("    api[ns] = {};")
    lines.append("    for (const m of schema[ns]) {")
    lines.append("      api[ns][m] = (...args) => {")
    lines.append("        // Prefer preload-exposed DarsDesktopAPI if available")
    lines.append("        try {")
    lines.append("          if (typeof window !== 'undefined' && window.DarsDesktopAPI && window.DarsDesktopAPI[ns] && typeof window.DarsDesktopAPI[ns][m] === 'function') {")
    lines.append("            return window.DarsDesktopAPI[ns][m](...args);")
    lines.append("          }")
    lines.append("        } catch (_) {}")
    lines.append("        // Fallback: if a DarsIPC bridge with invoke is present, use it")
    lines.append("        try {")
    lines.append("          if (typeof globalThis !== 'undefined' && globalThis.DarsIPC && typeof globalThis.DarsIPC.invoke === 'function') {")
    lines.append("            return globalThis.DarsIPC.invoke(`dars::${ns}::${m}`, ...args);")
    lines.append("          }")
    lines.append("        } catch (_) {}")
    lines.append("        throw new Error('Dars desktop API not available: ensure preload exposes DarsDesktopAPI or DarsIPC.invoke');")
    lines.append("      };")
    lines.append("    }")
    lines.append("  }")
    lines.append("  return api;")
    lines.append("};")
    lines.append("")
    lines.append("export const DarsDesktopAPI = _make_api_from_schema(_DARS_SCHEMA);")
    return "\n".join(lines) + "\n"


def generate_stub_js(schema: Dict[str, Dict[str, str]]) -> str:
    return PREAMBLE + _gen_stub_impl(schema) + "\n"


def write_backend_templates(out_dir: str, *, package_json: Optional[str] = None, main_js: Optional[str] = None, preload_js: Optional[str] = None) -> None:
    """Write static backend templates if provided. Phase 3 exporter will call this.
    """
    import os
    os.makedirs(out_dir, exist_ok=True)
    if package_json is not None:
        with open(os.path.join(out_dir, 'package.json'), 'w', encoding='utf-8') as f:
            f.write(package_json)
    if main_js is not None:
        with open(os.path.join(out_dir, 'main.js'), 'w', encoding='utf-8') as f:
            f.write(main_js)
    if preload_js is not None:
        with open(os.path.join(out_dir, 'preload.js'), 'w', encoding='utf-8') as f:
            f.write(preload_js)
