"""Class that represents a realized subset of the `CellArrDataset`.

This class provides a slice data class usually generated by the access
methods from
:py:func:`cellarr.CellArrDataset.CellArrDataset`.

Example:

    .. code-block:: python

        from cellarr import CellArrDataset

        cd = CellArrDataset(dataset_path="/path/to/cellar/dir")
        gene_list = ["gene_1", "gene_95", "gene_50"]
        result1 = cd[0, gene_list]

        print(result1)
"""

from dataclasses import dataclass
from typing import Any

import anndata
import pandas as pd
import summarizedexperiment as se

__author__ = "Jayaram Kancherla"
__copyright__ = "Jayaram Kancherla"
__license__ = "MIT"


@dataclass
class CellArrDatasetSlice:
    """Class that represents a realized subset of the `CellArrDataset`."""

    cell_metadata: pd.DataFrame
    gene_annotation: pd.DataFrame
    matrix: Any

    def to_anndata(self):
        return anndata.AnnData(
            layers={"matrix": self.matrix.tocsr()},
            obs=self.cell_metadata,
            var=self.gene_annotation,
        )

    def to_summarizedexperiment(self):
        return se.SummarizedExperiment(
            assays={"matrix": self.matrix.tocsr().transpose()},
            row_data=self.gene_annotation,
            column_data=self.cell_metadata,
        )
