#
# reviewboard/admin/checks.py -- Dependency checks for items which are used in
#                                the admin UI. For the most part, when one of
#                                these fails, some piece of UI is disabled with
#                                the returned error message.
#
# Copyright (c) 2008-2009  Christian Hammond
# Copyright (c) 2009  David Trowbridge
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
# IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
# CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
# TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
# SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

from __future__ import annotations

import getpass
import os
import sys
from typing import Optional

from django.conf import settings
from django.core.exceptions import ImproperlyConfigured
from django.db import DatabaseError
from django.utils.translation import gettext as _
from djblets.util.filesystem import is_exe_in_path
from djblets.siteconfig.models import SiteConfiguration

import reviewboard
from reviewboard import get_version_string
from reviewboard.admin.import_utils import has_module
from reviewboard.admin.server import get_data_dir


_install_fine = False


def check_updates_required():
    """Check if there are manual updates required.

    Sometimes, especially in developer installs, some things need to be tweaked
    by hand before Review Board can be used on this server.
    """
    global _install_fine

    updates_required = []

    if not _install_fine:
        site_dir = os.path.dirname(settings.HTDOCS_ROOT)
        devel_install = (os.path.exists(os.path.join(settings.LOCAL_ROOT,
                                                     'manage.py')))
        siteconfig = None

        # Check if we can access a SiteConfiguration. There should always
        # be one, unless the user has erased stuff by hand.
        #
        # This also checks for any sort of errors in talking to the database.
        # This could be due to the database being down, or corrupt, or
        # tables locked, or an empty database, or other cases. We want to
        # catch this before getting the point where plain 500 Internal Server
        # Errors appear.
        try:
            siteconfig = SiteConfiguration.objects.get_current()
        except (DatabaseError, SiteConfiguration.DoesNotExist) as e:
            updates_required.append((
                'admin/manual-updates/database-error.html', {
                    'error': e,
                }
            ))

        # Check if the version running matches the last stored version.
        # Only do this for non-debug installs, as it's really annoying on
        # a developer install.:
        cur_version = get_version_string()

        if siteconfig and siteconfig.version != cur_version:
            updates_required.append((
                'admin/manual-updates/version-mismatch.html', {
                    'current_version': cur_version,
                    'stored_version': siteconfig.version,
                    'site_dir': site_dir,
                    'devel_install': devel_install,
                    'python_ver': '%s.%s.%s' % sys.version_info[:3],
                    'package_path': os.path.dirname(reviewboard.__file__),
                }
            ))

        # Check if the site has moved and the old media directory no longer
        # exists.
        if siteconfig and not os.path.exists(settings.STATIC_ROOT):
            new_static_root = os.path.join(settings.HTDOCS_ROOT, 'static')

            if os.path.exists(new_static_root):
                siteconfig.set('site_static_root', new_static_root)
                settings.STATIC_ROOT = new_static_root

        # Check if the site has moved and the old media directory no longer
        # exists.
        if siteconfig and not os.path.exists(settings.MEDIA_ROOT):
            new_media_root = os.path.join(settings.HTDOCS_ROOT, 'media')

            if os.path.exists(new_media_root):
                siteconfig.set('site_media_root', new_media_root)
                settings.MEDIA_ROOT = new_media_root

        # Check if the user has any pending static media configuration
        # changes they need to make.
        if siteconfig and 'manual-updates' in siteconfig.settings:
            stored_updates = siteconfig.settings['manual-updates']

            if not stored_updates.get('static-media', False):
                updates_required.append((
                    'admin/manual-updates/server-static-config.html', {
                        'STATIC_ROOT': settings.STATIC_ROOT,
                        'SITE_ROOT': settings.SITE_ROOT,
                        'SITE_DIR': settings.LOCAL_ROOT,
                    }
                ))

        # Check if there's a media/uploaded/images directory. If not, this is
        # either a new install or is using the old-style media setup and needs
        # to be manually upgraded.
        uploaded_dir = os.path.join(settings.MEDIA_ROOT, "uploaded")

        if not os.path.isdir(uploaded_dir) or \
           not os.path.isdir(os.path.join(uploaded_dir, "images")):
            updates_required.append((
                "admin/manual-updates/media-upload-dir.html", {
                    'MEDIA_ROOT': settings.MEDIA_ROOT
                }
            ))

        try:
            username = getpass.getuser()
        except ImportError:
            # This will happen if running on Windows (which doesn't have
            # the pwd module) and if %LOGNAME%, %USER%, %LNAME% and
            # %USERNAME% are all undefined.
            username = "<server username>"

        # Check if the data directory (should be $HOME) is writable by us.
        data_dir: Optional[str] = None
        data_dir_writable: bool = False

        try:
            data_dir = get_data_dir()
            data_dir_writable = os.access(data_dir, os.W_OK)
        except ImproperlyConfigured:
            pass

        if (not data_dir or
            not data_dir_writable or
            not os.path.isdir(data_dir)):
            try:
                username = getpass.getuser()
            except ImportError:
                # This will happen if running on Windows (which doesn't have
                # the pwd module) and if %LOGNAME%, %USER%, %LNAME% and
                # %USERNAME% are all undefined.
                username = "<server username>"

            updates_required.append((
                'admin/manual-updates/data-dir.html', {
                    'data_dir': data_dir,
                    'writable': data_dir_writable,
                    'server_user': username,
                    'expected_data_dir': os.path.join(site_dir, 'data'),
                }
            ))

        # Check if the the legacy htdocs and modern static extension
        # directories exist and are writable by us.
        ext_roots = [settings.MEDIA_ROOT]

        if not settings.DEBUG:
            ext_roots.append(settings.STATIC_ROOT)

        for root in ext_roots:
            ext_dir = os.path.join(root, 'ext')

            if not os.path.isdir(ext_dir) or not os.access(ext_dir, os.W_OK):
                updates_required.append((
                    'admin/manual-updates/ext-dir.html', {
                        'ext_dir': ext_dir,
                        'writable': os.access(ext_dir, os.W_OK),
                        'server_user': username,
                    }
                ))

        if not is_exe_in_path('patch'):
            if sys.platform == 'win32':
                binaryname = 'patch.exe'
            else:
                binaryname = 'patch'

            updates_required.append((
                "admin/manual-updates/install-patch.html", {
                    'platform': sys.platform,
                    'binaryname': binaryname,
                    'search_path': os.getenv('PATH'),
                }
            ))

        #
        # NOTE: Add new checks above this.
        #

        _install_fine = not updates_required

    return updates_required


def reset_check_cache():
    """Reset the cached data of all checks.

    This is mainly useful during unit tests.
    """
    global _install_fine

    _install_fine = False


def get_can_enable_ldap():
    """Check whether LDAP authentication can be enabled.

    Returns:
        tuple:
        A tuple containing:

        1. A boolean indicating whether the support can be enabled.
        2. A localized string explaining how to enable the support, or ``None``
           if support is available.
    """
    if has_module('ldap'):
        return True, None
    else:
        return (False, _(
            'To enable support for LDAP and Active Directory, you will need '
            'to install the ReviewBoard[ldap] module (e.g., `pip install '
            'ReviewBoard[ldap]`).'
        ))


def get_can_use_amazon_s3():
    """Check whether django-storages (Amazon S3 backend) is installed."""
    try:
        if has_module('storages.backends.s3', members=['S3Storage']):
            return (True, None)
        else:
            return (False, _(
                'Amazon S3 depends on ReviewBoard[s3], which is not installed'
            ))
    except ImproperlyConfigured as e:
        return (False, _('Amazon S3 backend failed to load: %s') % e)


def get_can_use_openstack_swift():
    """Check whether django-storage-swift is installed."""
    try:
        if has_module('swift.storage', members=['SwiftStorage']):
            return (True, None)
        else:
            return (False, _(
                'OpenStack Swift depends on django-storage-swift, '
                'which is not installed'
            ))
    except ImproperlyConfigured as e:
        return (False, _('OpenStack Swift backend failed to load: %s') % e)


def get_can_use_couchdb():
    """Check whether django-storages (CouchDB backend) is installed.

    Version Changed:
        7.0:
        This now always returns ``False``. The support for CouchDB in
        django-storages was deprecated in 1.5.0, and has since been removed.
    """
    return (False, _(
        'CouchDB support has been removed from django-storages, and can '
        'no longer be used in Review Board.'
    ))
