"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomResourceProvider = exports.CustomResourceProviderRuntime = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const cxapi = require("@aws-cdk/cx-api");
const asset_staging_1 = require("../asset-staging");
const assets_1 = require("../assets");
const cfn_resource_1 = require("../cfn-resource");
const duration_1 = require("../duration");
const size_1 = require("../size");
const stack_1 = require("../stack");
const token_1 = require("../token");
const ENTRYPOINT_FILENAME = '__entrypoint__';
const ENTRYPOINT_NODEJS_SOURCE = path.join(__dirname, 'nodejs-entrypoint.js');
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("../construct-compat");
/**
 * The lambda runtime to use for the resource provider. This also indicates
 * which language is used for the handler.
 */
var CustomResourceProviderRuntime;
(function (CustomResourceProviderRuntime) {
    /**
     * Node.js 12.x
     *
     * @deprecated Use {@link NODEJS_14_X}
     */
    CustomResourceProviderRuntime["NODEJS_12"] = "nodejs12.x";
    /**
     * Node.js 12.x
     */
    CustomResourceProviderRuntime["NODEJS_12_X"] = "nodejs12.x";
    /**
     * Node.js 14.x
     */
    CustomResourceProviderRuntime["NODEJS_14_X"] = "nodejs14.x";
})(CustomResourceProviderRuntime = exports.CustomResourceProviderRuntime || (exports.CustomResourceProviderRuntime = {}));
/**
 * An AWS-Lambda backed custom resource provider, for CDK Construct Library constructs
 *
 * This is a provider for `CustomResource` constructs, backed by an AWS Lambda
 * Function. It only supports NodeJS runtimes.
 *
 * **This is not a generic custom resource provider class**. It is specifically
 * intended to be used only by constructs in the AWS CDK Construct Library, and
 * only exists here because of reverse dependency issues (for example, it cannot
 * use `iam.PolicyStatement` objects, since the `iam` library already depends on
 * the CDK `core` library and we cannot have cyclic dependencies).
 *
 * If you are not writing constructs for the AWS Construct Library, you should
 * use the `Provider` class in the `custom-resources` module instead, which has
 * a better API and supports all Lambda runtimes, not just Node.
 *
 * N.B.: When you are writing Custom Resource Providers, there are a number of
 * lifecycle events you have to pay attention to. These are documented in the
 * README of the `custom-resources` module. Be sure to give the documentation
 * in that module a read, regardless of whether you end up using the Provider
 * class in there or this one.
 */
class CustomResourceProvider extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, CustomResourceProvider);
            }
            throw error;
        }
        const stack = stack_1.Stack.of(scope);
        // copy the entry point to the code directory
        fs.copyFileSync(ENTRYPOINT_NODEJS_SOURCE, path.join(props.codeDirectory, `${ENTRYPOINT_FILENAME}.js`));
        // verify we have an index file there
        if (!fs.existsSync(path.join(props.codeDirectory, 'index.js'))) {
            throw new Error(`cannot find ${props.codeDirectory}/index.js`);
        }
        const staging = new asset_staging_1.AssetStaging(this, 'Staging', {
            sourcePath: props.codeDirectory,
        });
        const assetFileName = staging.relativeStagedPath(stack);
        const asset = stack.synthesizer.addFileAsset({
            fileName: assetFileName,
            sourceHash: staging.assetHash,
            packaging: assets_1.FileAssetPackaging.ZIP_DIRECTORY,
        });
        const policies = !props.policyStatements ? undefined : [
            {
                PolicyName: 'Inline',
                PolicyDocument: {
                    Version: '2012-10-17',
                    Statement: props.policyStatements,
                },
            },
        ];
        const role = new cfn_resource_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'lambda.amazonaws.com' } }],
                },
                ManagedPolicyArns: [
                    { 'Fn::Sub': 'arn:${AWS::Partition}:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole' },
                ],
                Policies: policies,
            },
        });
        this.roleArn = token_1.Token.asString(role.getAtt('Arn'));
        const timeout = props.timeout ?? duration_1.Duration.minutes(15);
        const memory = props.memorySize ?? size_1.Size.mebibytes(128);
        const handler = new cfn_resource_1.CfnResource(this, 'Handler', {
            type: 'AWS::Lambda::Function',
            properties: {
                Code: {
                    S3Bucket: asset.bucketName,
                    S3Key: asset.objectKey,
                },
                Timeout: timeout.toSeconds(),
                MemorySize: memory.toMebibytes(),
                Handler: `${ENTRYPOINT_FILENAME}.handler`,
                Role: role.getAtt('Arn'),
                Runtime: props.runtime,
                Environment: this.renderEnvironmentVariables(props.environment),
                Description: props.description ?? undefined,
            },
        });
        handler.addDependsOn(role);
        if (this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PATH_KEY, assetFileName);
            handler.addMetadata(cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY, 'Code');
        }
        this.serviceToken = token_1.Token.asString(handler.getAtt('Arn'));
    }
    /**
     * Returns a stack-level singleton ARN (service token) for the custom resource
     * provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreate(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreate);
            }
            throw error;
        }
        return this.getOrCreateProvider(scope, uniqueid, props).serviceToken;
    }
    /**
     * Returns a stack-level singleton for the custom resource provider.
     *
     * @param scope Construct scope
     * @param uniqueid A globally unique id that will be used for the stack-level
     * construct.
     * @param props Provider properties which will only be applied when the
     * provider is first created.
     * @returns the service token of the custom resource provider, which should be
     * used when defining a `CustomResource`.
     */
    static getOrCreateProvider(scope, uniqueid, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_core_CustomResourceProviderProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getOrCreateProvider);
            }
            throw error;
        }
        const id = `${uniqueid}CustomResourceProvider`;
        const stack = stack_1.Stack.of(scope);
        const provider = stack.node.tryFindChild(id)
            ?? new CustomResourceProvider(stack, id, props);
        return provider;
    }
    renderEnvironmentVariables(env) {
        if (!env || Object.keys(env).length === 0) {
            return undefined;
        }
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it)
        const variables = {};
        const keys = Object.keys(env).sort();
        for (const key of keys) {
            variables[key] = env[key];
        }
        return { Variables: variables };
    }
}
exports.CustomResourceProvider = CustomResourceProvider;
_a = JSII_RTTI_SYMBOL_1;
CustomResourceProvider[_a] = { fqn: "@aws-cdk/core.CustomResourceProvider", version: "1.161.0" };
//# sourceMappingURL=data:application/json;base64,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