"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const principals_1 = require("./principals");
const util_1 = require("./util");
/**
 * Represents a statement in an IAM policy document.
 */
class PolicyStatement {
    constructor(props = {}) {
        this.action = new Array();
        this.notAction = new Array();
        this.principal = {};
        this.notPrincipal = {};
        this.resource = new Array();
        this.notResource = new Array();
        this.condition = {};
        // Validate actions
        for (const action of [...props.actions || [], ...props.notActions || []]) {
            if (!/^(\*|[a-zA-Z0-9-]+:[a-zA-Z0-9*]+)$/.test(action)) {
                throw new Error(`Action '${action}' is invalid. An action string consists of a service namespace, a colon, and the name of an action. Action names can include wildcards.`);
            }
        }
        this.effect = props.effect || Effect.ALLOW;
        this.addActions(...props.actions || []);
        this.addNotActions(...props.notActions || []);
        this.addPrincipals(...props.principals || []);
        this.addNotPrincipals(...props.notPrincipals || []);
        this.addResources(...props.resources || []);
        this.addNotResources(...props.notResources || []);
        if (props.conditions !== undefined) {
            this.addConditions(props.conditions);
        }
    }
    //
    // Actions
    //
    addActions(...actions) {
        if (actions.length > 0 && this.notAction.length > 0) {
            throw new Error(`Cannot add 'Actions' to policy statement if 'NotActions' have been added`);
        }
        this.action.push(...actions);
    }
    addNotActions(...notActions) {
        if (notActions.length > 0 && this.action.length > 0) {
            throw new Error(`Cannot add 'NotActions' to policy statement if 'Actions' have been added`);
        }
        this.notAction.push(...notActions);
    }
    //
    // Principal
    //
    /**
     * Indicates if this permission has a "Principal" section.
     */
    get hasPrincipal() {
        return Object.keys(this.principal).length > 0 || Object.keys(this.notPrincipal).length > 0;
    }
    addPrincipals(...principals) {
        if (Object.keys(principals).length > 0 && Object.keys(this.notPrincipal).length > 0) {
            throw new Error(`Cannot add 'Principals' to policy statement if 'NotPrincipals' have been added`);
        }
        for (const principal of principals) {
            const fragment = principal.policyFragment;
            util_1.mergePrincipal(this.principal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addNotPrincipals(...notPrincipals) {
        if (Object.keys(notPrincipals).length > 0 && Object.keys(this.principal).length > 0) {
            throw new Error(`Cannot add 'NotPrincipals' to policy statement if 'Principals' have been added`);
        }
        for (const notPrincipal of notPrincipals) {
            const fragment = notPrincipal.policyFragment;
            util_1.mergePrincipal(this.notPrincipal, fragment.principalJson);
            this.addConditions(fragment.conditions);
        }
    }
    addAwsAccountPrincipal(accountId) {
        this.addPrincipals(new principals_1.AccountPrincipal(accountId));
    }
    addArnPrincipal(arn) {
        this.addPrincipals(new principals_1.ArnPrincipal(arn));
    }
    /**
     * Adds a service principal to this policy statement.
     *
     * @param service the service name for which a service principal is requested (e.g: `s3.amazonaws.com`).
     * @param opts    options for adding the service principal (such as specifying a principal in a different region)
     */
    addServicePrincipal(service, opts) {
        this.addPrincipals(new principals_1.ServicePrincipal(service, opts));
    }
    addFederatedPrincipal(federated, conditions) {
        this.addPrincipals(new principals_1.FederatedPrincipal(federated, conditions));
    }
    addAccountRootPrincipal() {
        this.addPrincipals(new principals_1.AccountRootPrincipal());
    }
    addCanonicalUserPrincipal(canonicalUserId) {
        this.addPrincipals(new principals_1.CanonicalUserPrincipal(canonicalUserId));
    }
    addAnyPrincipal() {
        this.addPrincipals(new principals_1.Anyone());
    }
    //
    // Resources
    //
    addResources(...arns) {
        if (arns.length > 0 && this.notResource.length > 0) {
            throw new Error(`Cannot add 'Resources' to policy statement if 'NotResources' have been added`);
        }
        this.resource.push(...arns);
    }
    addNotResources(...arns) {
        if (arns.length > 0 && this.resource.length > 0) {
            throw new Error(`Cannot add 'NotResources' to policy statement if 'Resources' have been added`);
        }
        this.notResource.push(...arns);
    }
    /**
     * Adds a ``"*"`` resource to this statement.
     */
    addAllResources() {
        this.addResources('*');
    }
    /**
     * Indicates if this permission as at least one resource associated with it.
     */
    get hasResource() {
        return this.resource && this.resource.length > 0;
    }
    //
    // Condition
    //
    /**
     * Add a condition to the Policy
     */
    addCondition(key, value) {
        this.condition[key] = value;
    }
    /**
     * Add multiple conditions to the Policy
     */
    addConditions(conditions) {
        Object.keys(conditions).map(key => {
            this.addCondition(key, conditions[key]);
        });
    }
    /**
     * Add a condition that limits to a given account
     */
    addAccountCondition(accountId) {
        this.addCondition('StringEquals', { 'sts:ExternalId': accountId });
    }
    toStatementJson() {
        return noUndef({
            Action: _norm(this.action),
            NotAction: _norm(this.notAction),
            Condition: _norm(this.condition),
            Effect: _norm(this.effect),
            Principal: _normPrincipal(this.principal),
            NotPrincipal: _normPrincipal(this.notPrincipal),
            Resource: _norm(this.resource),
            NotResource: _norm(this.notResource),
            Sid: _norm(this.sid),
        });
        function _norm(values) {
            if (typeof (values) === 'undefined') {
                return undefined;
            }
            if (cdk.Token.isUnresolved(values)) {
                return values;
            }
            if (Array.isArray(values)) {
                if (!values || values.length === 0) {
                    return undefined;
                }
                if (values.length === 1) {
                    return values[0];
                }
                return values;
            }
            if (typeof (values) === 'object') {
                if (Object.keys(values).length === 0) {
                    return undefined;
                }
            }
            return values;
        }
        function _normPrincipal(principal) {
            const keys = Object.keys(principal);
            if (keys.length === 0) {
                return undefined;
            }
            const result = {};
            for (const key of keys) {
                const normVal = _norm(principal[key]);
                if (normVal) {
                    result[key] = normVal;
                }
            }
            if (Object.keys(result).length === 1 && result.AWS === '*') {
                return '*';
            }
            return result;
        }
    }
    toString() {
        return cdk.Token.asString(this, {
            displayHint: 'PolicyStatement'
        });
    }
    /**
     * JSON-ify the statement
     *
     * Used when JSON.stringify() is called
     */
    toJSON() {
        return this.toStatementJson();
    }
}
exports.PolicyStatement = PolicyStatement;
var Effect;
(function (Effect) {
    Effect["ALLOW"] = "Allow";
    Effect["DENY"] = "Deny";
})(Effect = exports.Effect || (exports.Effect = {}));
function noUndef(x) {
    const ret = {};
    for (const [key, value] of Object.entries(x)) {
        if (value !== undefined) {
            ret[key] = value;
        }
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoicG9saWN5LXN0YXRlbWVudC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInBvbGljeS1zdGF0ZW1lbnQudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxxQ0FBc0M7QUFDdEMsNkNBQytGO0FBQy9GLGlDQUF3QztBQUV4Qzs7R0FFRztBQUNILE1BQWEsZUFBZTtJQWUxQixZQUFZLFFBQThCLEVBQUU7UUFSM0IsV0FBTSxHQUFHLElBQUksS0FBSyxFQUFPLENBQUM7UUFDMUIsY0FBUyxHQUFHLElBQUksS0FBSyxFQUFPLENBQUM7UUFDN0IsY0FBUyxHQUE2QixFQUFFLENBQUM7UUFDekMsaUJBQVksR0FBNkIsRUFBRSxDQUFDO1FBQzVDLGFBQVEsR0FBRyxJQUFJLEtBQUssRUFBTyxDQUFDO1FBQzVCLGdCQUFXLEdBQUcsSUFBSSxLQUFLLEVBQU8sQ0FBQztRQUMvQixjQUFTLEdBQTJCLEVBQUcsQ0FBQztRQUd2RCxtQkFBbUI7UUFDbkIsS0FBSyxNQUFNLE1BQU0sSUFBSSxDQUFDLEdBQUcsS0FBSyxDQUFDLE9BQU8sSUFBSSxFQUFFLEVBQUUsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxFQUFFO1lBQ3hFLElBQUksQ0FBQyxvQ0FBb0MsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3RELE1BQU0sSUFBSSxLQUFLLENBQUMsV0FBVyxNQUFNLHlJQUF5SSxDQUFDLENBQUM7YUFDN0s7U0FDRjtRQUVELElBQUksQ0FBQyxNQUFNLEdBQUcsS0FBSyxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsS0FBSyxDQUFDO1FBRTNDLElBQUksQ0FBQyxVQUFVLENBQUMsR0FBRyxLQUFLLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQ3hDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxhQUFhLENBQUMsR0FBRyxLQUFLLENBQUMsVUFBVSxJQUFJLEVBQUUsQ0FBQyxDQUFDO1FBQzlDLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxHQUFHLEtBQUssQ0FBQyxhQUFhLElBQUksRUFBRSxDQUFDLENBQUM7UUFDcEQsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLEtBQUssQ0FBQyxTQUFTLElBQUksRUFBRSxDQUFDLENBQUM7UUFDNUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxHQUFHLEtBQUssQ0FBQyxZQUFZLElBQUksRUFBRSxDQUFDLENBQUM7UUFDbEQsSUFBSSxLQUFLLENBQUMsVUFBVSxLQUFLLFNBQVMsRUFBRTtZQUNsQyxJQUFJLENBQUMsYUFBYSxDQUFDLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUN0QztJQUNILENBQUM7SUFFRCxFQUFFO0lBQ0YsVUFBVTtJQUNWLEVBQUU7SUFFSyxVQUFVLENBQUMsR0FBRyxPQUFpQjtRQUNwQyxJQUFJLE9BQU8sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxTQUFTLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDN0Y7UUFDRCxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLE9BQU8sQ0FBQyxDQUFDO0lBQy9CLENBQUM7SUFFTSxhQUFhLENBQUMsR0FBRyxVQUFvQjtRQUMxQyxJQUFJLFVBQVUsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxJQUFJLElBQUksQ0FBQyxNQUFNLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuRCxNQUFNLElBQUksS0FBSyxDQUFDLDBFQUEwRSxDQUFDLENBQUM7U0FDN0Y7UUFDRCxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxHQUFHLFVBQVUsQ0FBQyxDQUFDO0lBQ3JDLENBQUM7SUFFRCxFQUFFO0lBQ0YsWUFBWTtJQUNaLEVBQUU7SUFFRjs7T0FFRztJQUNILElBQVcsWUFBWTtRQUNyQixPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsQ0FBQztJQUM3RixDQUFDO0lBRU0sYUFBYSxDQUFDLEdBQUcsVUFBd0I7UUFDOUMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUNuRixNQUFNLElBQUksS0FBSyxDQUFDLGdGQUFnRixDQUFDLENBQUM7U0FDbkc7UUFDRCxLQUFLLE1BQU0sU0FBUyxJQUFJLFVBQVUsRUFBRTtZQUNsQyxNQUFNLFFBQVEsR0FBRyxTQUFTLENBQUMsY0FBYyxDQUFDO1lBQzFDLHFCQUFjLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxRQUFRLENBQUMsYUFBYSxDQUFDLENBQUM7WUFDdkQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxRQUFRLENBQUMsVUFBVSxDQUFDLENBQUM7U0FDekM7SUFDSCxDQUFDO0lBRU0sZ0JBQWdCLENBQUMsR0FBRyxhQUEyQjtRQUNwRCxJQUFJLE1BQU0sQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ25GLE1BQU0sSUFBSSxLQUFLLENBQUMsZ0ZBQWdGLENBQUMsQ0FBQztTQUNuRztRQUNELEtBQUssTUFBTSxZQUFZLElBQUksYUFBYSxFQUFFO1lBQ3hDLE1BQU0sUUFBUSxHQUFHLFlBQVksQ0FBQyxjQUFjLENBQUM7WUFDN0MscUJBQWMsQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFLFFBQVEsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUMxRCxJQUFJLENBQUMsYUFBYSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUMsQ0FBQztTQUN6QztJQUNILENBQUM7SUFFTSxzQkFBc0IsQ0FBQyxTQUFpQjtRQUM3QyxJQUFJLENBQUMsYUFBYSxDQUFDLElBQUksNkJBQWdCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQztJQUN0RCxDQUFDO0lBRU0sZUFBZSxDQUFDLEdBQVc7UUFDaEMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLHlCQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztJQUM1QyxDQUFDO0lBRUQ7Ozs7O09BS0c7SUFDSSxtQkFBbUIsQ0FBQyxPQUFlLEVBQUUsSUFBMkI7UUFDckUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLDZCQUFnQixDQUFDLE9BQU8sRUFBRSxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQzFELENBQUM7SUFFTSxxQkFBcUIsQ0FBQyxTQUFjLEVBQUUsVUFBZ0M7UUFDM0UsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLCtCQUFrQixDQUFDLFNBQVMsRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDO0lBQ3BFLENBQUM7SUFFTSx1QkFBdUI7UUFDNUIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLGlDQUFvQixFQUFFLENBQUMsQ0FBQztJQUNqRCxDQUFDO0lBRU0seUJBQXlCLENBQUMsZUFBdUI7UUFDdEQsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLG1DQUFzQixDQUFDLGVBQWUsQ0FBQyxDQUFDLENBQUM7SUFDbEUsQ0FBQztJQUVNLGVBQWU7UUFDcEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLG1CQUFNLEVBQUUsQ0FBQyxDQUFDO0lBQ25DLENBQUM7SUFFRCxFQUFFO0lBQ0YsWUFBWTtJQUNaLEVBQUU7SUFFSyxZQUFZLENBQUMsR0FBRyxJQUFjO1FBQ25DLElBQUksSUFBSSxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksSUFBSSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2xELE1BQU0sSUFBSSxLQUFLLENBQUMsOEVBQThFLENBQUMsQ0FBQztTQUNqRztRQUNELElBQUksQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLEdBQUcsSUFBSSxDQUFDLENBQUM7SUFDOUIsQ0FBQztJQUVNLGVBQWUsQ0FBQyxHQUFHLElBQWM7UUFDdEMsSUFBSSxJQUFJLENBQUMsTUFBTSxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDL0MsTUFBTSxJQUFJLEtBQUssQ0FBQyw4RUFBOEUsQ0FBQyxDQUFDO1NBQ2pHO1FBQ0QsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsR0FBRyxJQUFJLENBQUMsQ0FBQztJQUNqQyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxlQUFlO1FBQ3BCLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUM7SUFDekIsQ0FBQztJQUVEOztPQUVHO0lBQ0gsSUFBVyxXQUFXO1FBQ3BCLE9BQU8sSUFBSSxDQUFDLFFBQVEsSUFBSSxJQUFJLENBQUMsUUFBUSxDQUFDLE1BQU0sR0FBRyxDQUFDLENBQUM7SUFDbkQsQ0FBQztJQUVELEVBQUU7SUFDRixZQUFZO0lBQ1osRUFBRTtJQUVGOztPQUVHO0lBQ0ksWUFBWSxDQUFDLEdBQVcsRUFBRSxLQUFVO1FBQ3pDLElBQUksQ0FBQyxTQUFTLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO0lBQzlCLENBQUM7SUFFRDs7T0FFRztJQUNJLGFBQWEsQ0FBQyxVQUFnQztRQUNuRCxNQUFNLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRTtZQUNoQyxJQUFJLENBQUMsWUFBWSxDQUFDLEdBQUcsRUFBRSxVQUFVLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUMxQyxDQUFDLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7T0FFRztJQUNJLG1CQUFtQixDQUFDLFNBQWlCO1FBQzFDLElBQUksQ0FBQyxZQUFZLENBQUMsY0FBYyxFQUFFLEVBQUUsZ0JBQWdCLEVBQUUsU0FBUyxFQUFFLENBQUMsQ0FBQztJQUNyRSxDQUFDO0lBRU0sZUFBZTtRQUNwQixPQUFPLE9BQU8sQ0FBQztZQUNiLE1BQU0sRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztZQUMxQixTQUFTLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDaEMsU0FBUyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDO1lBQ2hDLE1BQU0sRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQztZQUMxQixTQUFTLEVBQUUsY0FBYyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDekMsWUFBWSxFQUFFLGNBQWMsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDO1lBQy9DLFFBQVEsRUFBRSxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQztZQUM5QixXQUFXLEVBQUUsS0FBSyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUM7WUFDcEMsR0FBRyxFQUFFLEtBQUssQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDO1NBQ3JCLENBQUMsQ0FBQztRQUVILFNBQVMsS0FBSyxDQUFDLE1BQVc7WUFFeEIsSUFBSSxPQUFNLENBQUMsTUFBTSxDQUFDLEtBQUssV0FBVyxFQUFFO2dCQUNsQyxPQUFPLFNBQVMsQ0FBQzthQUNsQjtZQUVELElBQUksR0FBRyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ2xDLE9BQU8sTUFBTSxDQUFDO2FBQ2Y7WUFFRCxJQUFJLEtBQUssQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUU7Z0JBQ3pCLElBQUksQ0FBQyxNQUFNLElBQUksTUFBTSxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7b0JBQ2xDLE9BQU8sU0FBUyxDQUFDO2lCQUNsQjtnQkFFRCxJQUFJLE1BQU0sQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO29CQUN2QixPQUFPLE1BQU0sQ0FBQyxDQUFDLENBQUMsQ0FBQztpQkFDbEI7Z0JBRUQsT0FBTyxNQUFNLENBQUM7YUFDZjtZQUVELElBQUksT0FBTSxDQUFDLE1BQU0sQ0FBQyxLQUFLLFFBQVEsRUFBRTtnQkFDL0IsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7b0JBQ3BDLE9BQU8sU0FBUyxDQUFDO2lCQUNsQjthQUNGO1lBRUQsT0FBTyxNQUFNLENBQUM7UUFDaEIsQ0FBQztRQUVELFNBQVMsY0FBYyxDQUFDLFNBQW1DO1lBQ3pELE1BQU0sSUFBSSxHQUFHLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUM7WUFDcEMsSUFBSSxJQUFJLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtnQkFBRSxPQUFPLFNBQVMsQ0FBQzthQUFFO1lBQzVDLE1BQU0sTUFBTSxHQUFRLEVBQUUsQ0FBQztZQUN2QixLQUFLLE1BQU0sR0FBRyxJQUFJLElBQUksRUFBRTtnQkFDdEIsTUFBTSxPQUFPLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDO2dCQUN0QyxJQUFJLE9BQU8sRUFBRTtvQkFDWCxNQUFNLENBQUMsR0FBRyxDQUFDLEdBQUcsT0FBTyxDQUFDO2lCQUN2QjthQUNGO1lBQ0QsSUFBSSxNQUFNLENBQUMsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksTUFBTSxDQUFDLEdBQUcsS0FBSyxHQUFHLEVBQUU7Z0JBQzFELE9BQU8sR0FBRyxDQUFDO2FBQ1o7WUFDRCxPQUFPLE1BQU0sQ0FBQztRQUNoQixDQUFDO0lBQ0gsQ0FBQztJQUVNLFFBQVE7UUFDYixPQUFPLEdBQUcsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRTtZQUM5QixXQUFXLEVBQUUsaUJBQWlCO1NBQy9CLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTTtRQUNYLE9BQU8sSUFBSSxDQUFDLGVBQWUsRUFBRSxDQUFDO0lBQ2hDLENBQUM7Q0FDRjtBQS9QRCwwQ0ErUEM7QUFFRCxJQUFZLE1BR1g7QUFIRCxXQUFZLE1BQU07SUFDaEIseUJBQWUsQ0FBQTtJQUNmLHVCQUFhLENBQUE7QUFDZixDQUFDLEVBSFcsTUFBTSxHQUFOLGNBQU0sS0FBTixjQUFNLFFBR2pCO0FBK0RELFNBQVMsT0FBTyxDQUFDLENBQU07SUFDckIsTUFBTSxHQUFHLEdBQVEsRUFBRSxDQUFDO0lBQ3BCLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFO1FBQzVDLElBQUksS0FBSyxLQUFLLFNBQVMsRUFBRTtZQUN2QixHQUFHLENBQUMsR0FBRyxDQUFDLEdBQUcsS0FBSyxDQUFDO1NBQ2xCO0tBQ0Y7SUFDRCxPQUFPLEdBQUcsQ0FBQztBQUNiLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgY2RrID0gcmVxdWlyZSgnQGF3cy1jZGsvY29yZScpO1xuaW1wb3J0IHsgQWNjb3VudFByaW5jaXBhbCwgQWNjb3VudFJvb3RQcmluY2lwYWwsIEFueW9uZSwgQXJuUHJpbmNpcGFsLCBDYW5vbmljYWxVc2VyUHJpbmNpcGFsLFxuICBGZWRlcmF0ZWRQcmluY2lwYWwsIElQcmluY2lwYWwsIFNlcnZpY2VQcmluY2lwYWwsIFNlcnZpY2VQcmluY2lwYWxPcHRzIH0gZnJvbSAnLi9wcmluY2lwYWxzJztcbmltcG9ydCB7IG1lcmdlUHJpbmNpcGFsIH0gZnJvbSAnLi91dGlsJztcblxuLyoqXG4gKiBSZXByZXNlbnRzIGEgc3RhdGVtZW50IGluIGFuIElBTSBwb2xpY3kgZG9jdW1lbnQuXG4gKi9cbmV4cG9ydCBjbGFzcyBQb2xpY3lTdGF0ZW1lbnQge1xuICAvKipcbiAgICogU3RhdGVtZW50IElEIGZvciB0aGlzIHN0YXRlbWVudFxuICAgKi9cbiAgcHVibGljIHNpZD86IHN0cmluZztcbiAgcHVibGljIGVmZmVjdDogRWZmZWN0O1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgYWN0aW9uID0gbmV3IEFycmF5PGFueT4oKTtcbiAgcHJpdmF0ZSByZWFkb25seSBub3RBY3Rpb24gPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIHJlYWRvbmx5IHByaW5jaXBhbDogeyBba2V5OiBzdHJpbmddOiBhbnlbXSB9ID0ge307XG4gIHByaXZhdGUgcmVhZG9ubHkgbm90UHJpbmNpcGFsOiB7IFtrZXk6IHN0cmluZ106IGFueVtdIH0gPSB7fTtcbiAgcHJpdmF0ZSByZWFkb25seSByZXNvdXJjZSA9IG5ldyBBcnJheTxhbnk+KCk7XG4gIHByaXZhdGUgcmVhZG9ubHkgbm90UmVzb3VyY2UgPSBuZXcgQXJyYXk8YW55PigpO1xuICBwcml2YXRlIHJlYWRvbmx5IGNvbmRpdGlvbjogeyBba2V5OiBzdHJpbmddOiBhbnkgfSA9IHsgfTtcblxuICBjb25zdHJ1Y3Rvcihwcm9wczogUG9saWN5U3RhdGVtZW50UHJvcHMgPSB7fSkge1xuICAgIC8vIFZhbGlkYXRlIGFjdGlvbnNcbiAgICBmb3IgKGNvbnN0IGFjdGlvbiBvZiBbLi4ucHJvcHMuYWN0aW9ucyB8fCBbXSwgLi4ucHJvcHMubm90QWN0aW9ucyB8fCBbXV0pIHtcbiAgICAgIGlmICghL14oXFwqfFthLXpBLVowLTktXSs6W2EtekEtWjAtOSpdKykkLy50ZXN0KGFjdGlvbikpIHtcbiAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBBY3Rpb24gJyR7YWN0aW9ufScgaXMgaW52YWxpZC4gQW4gYWN0aW9uIHN0cmluZyBjb25zaXN0cyBvZiBhIHNlcnZpY2UgbmFtZXNwYWNlLCBhIGNvbG9uLCBhbmQgdGhlIG5hbWUgb2YgYW4gYWN0aW9uLiBBY3Rpb24gbmFtZXMgY2FuIGluY2x1ZGUgd2lsZGNhcmRzLmApO1xuICAgICAgfVxuICAgIH1cblxuICAgIHRoaXMuZWZmZWN0ID0gcHJvcHMuZWZmZWN0IHx8IEVmZmVjdC5BTExPVztcblxuICAgIHRoaXMuYWRkQWN0aW9ucyguLi5wcm9wcy5hY3Rpb25zIHx8IFtdKTtcbiAgICB0aGlzLmFkZE5vdEFjdGlvbnMoLi4ucHJvcHMubm90QWN0aW9ucyB8fCBbXSk7XG4gICAgdGhpcy5hZGRQcmluY2lwYWxzKC4uLnByb3BzLnByaW5jaXBhbHMgfHwgW10pO1xuICAgIHRoaXMuYWRkTm90UHJpbmNpcGFscyguLi5wcm9wcy5ub3RQcmluY2lwYWxzIHx8IFtdKTtcbiAgICB0aGlzLmFkZFJlc291cmNlcyguLi5wcm9wcy5yZXNvdXJjZXMgfHwgW10pO1xuICAgIHRoaXMuYWRkTm90UmVzb3VyY2VzKC4uLnByb3BzLm5vdFJlc291cmNlcyB8fCBbXSk7XG4gICAgaWYgKHByb3BzLmNvbmRpdGlvbnMgIT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhpcy5hZGRDb25kaXRpb25zKHByb3BzLmNvbmRpdGlvbnMpO1xuICAgIH1cbiAgfVxuXG4gIC8vXG4gIC8vIEFjdGlvbnNcbiAgLy9cblxuICBwdWJsaWMgYWRkQWN0aW9ucyguLi5hY3Rpb25zOiBzdHJpbmdbXSkge1xuICAgIGlmIChhY3Rpb25zLmxlbmd0aCA+IDAgJiYgdGhpcy5ub3RBY3Rpb24ubGVuZ3RoID4gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBDYW5ub3QgYWRkICdBY3Rpb25zJyB0byBwb2xpY3kgc3RhdGVtZW50IGlmICdOb3RBY3Rpb25zJyBoYXZlIGJlZW4gYWRkZWRgKTtcbiAgICB9XG4gICAgdGhpcy5hY3Rpb24ucHVzaCguLi5hY3Rpb25zKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGROb3RBY3Rpb25zKC4uLm5vdEFjdGlvbnM6IHN0cmluZ1tdKSB7XG4gICAgaWYgKG5vdEFjdGlvbnMubGVuZ3RoID4gMCAmJiB0aGlzLmFjdGlvbi5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhZGQgJ05vdEFjdGlvbnMnIHRvIHBvbGljeSBzdGF0ZW1lbnQgaWYgJ0FjdGlvbnMnIGhhdmUgYmVlbiBhZGRlZGApO1xuICAgIH1cbiAgICB0aGlzLm5vdEFjdGlvbi5wdXNoKC4uLm5vdEFjdGlvbnMpO1xuICB9XG5cbiAgLy9cbiAgLy8gUHJpbmNpcGFsXG4gIC8vXG5cbiAgLyoqXG4gICAqIEluZGljYXRlcyBpZiB0aGlzIHBlcm1pc3Npb24gaGFzIGEgXCJQcmluY2lwYWxcIiBzZWN0aW9uLlxuICAgKi9cbiAgcHVibGljIGdldCBoYXNQcmluY2lwYWwoKSB7XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKHRoaXMucHJpbmNpcGFsKS5sZW5ndGggPiAwIHx8IE9iamVjdC5rZXlzKHRoaXMubm90UHJpbmNpcGFsKS5sZW5ndGggPiAwO1xuICB9XG5cbiAgcHVibGljIGFkZFByaW5jaXBhbHMoLi4ucHJpbmNpcGFsczogSVByaW5jaXBhbFtdKSB7XG4gICAgaWYgKE9iamVjdC5rZXlzKHByaW5jaXBhbHMpLmxlbmd0aCA+IDAgJiYgT2JqZWN0LmtleXModGhpcy5ub3RQcmluY2lwYWwpLmxlbmd0aCA+IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQ2Fubm90IGFkZCAnUHJpbmNpcGFscycgdG8gcG9saWN5IHN0YXRlbWVudCBpZiAnTm90UHJpbmNpcGFscycgaGF2ZSBiZWVuIGFkZGVkYCk7XG4gICAgfVxuICAgIGZvciAoY29uc3QgcHJpbmNpcGFsIG9mIHByaW5jaXBhbHMpIHtcbiAgICAgIGNvbnN0IGZyYWdtZW50ID0gcHJpbmNpcGFsLnBvbGljeUZyYWdtZW50O1xuICAgICAgbWVyZ2VQcmluY2lwYWwodGhpcy5wcmluY2lwYWwsIGZyYWdtZW50LnByaW5jaXBhbEpzb24pO1xuICAgICAgdGhpcy5hZGRDb25kaXRpb25zKGZyYWdtZW50LmNvbmRpdGlvbnMpO1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyBhZGROb3RQcmluY2lwYWxzKC4uLm5vdFByaW5jaXBhbHM6IElQcmluY2lwYWxbXSkge1xuICAgIGlmIChPYmplY3Qua2V5cyhub3RQcmluY2lwYWxzKS5sZW5ndGggPiAwICYmIE9iamVjdC5rZXlzKHRoaXMucHJpbmNpcGFsKS5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhZGQgJ05vdFByaW5jaXBhbHMnIHRvIHBvbGljeSBzdGF0ZW1lbnQgaWYgJ1ByaW5jaXBhbHMnIGhhdmUgYmVlbiBhZGRlZGApO1xuICAgIH1cbiAgICBmb3IgKGNvbnN0IG5vdFByaW5jaXBhbCBvZiBub3RQcmluY2lwYWxzKSB7XG4gICAgICBjb25zdCBmcmFnbWVudCA9IG5vdFByaW5jaXBhbC5wb2xpY3lGcmFnbWVudDtcbiAgICAgIG1lcmdlUHJpbmNpcGFsKHRoaXMubm90UHJpbmNpcGFsLCBmcmFnbWVudC5wcmluY2lwYWxKc29uKTtcbiAgICAgIHRoaXMuYWRkQ29uZGl0aW9ucyhmcmFnbWVudC5jb25kaXRpb25zKTtcbiAgICB9XG4gIH1cblxuICBwdWJsaWMgYWRkQXdzQWNjb3VudFByaW5jaXBhbChhY2NvdW50SWQ6IHN0cmluZykge1xuICAgIHRoaXMuYWRkUHJpbmNpcGFscyhuZXcgQWNjb3VudFByaW5jaXBhbChhY2NvdW50SWQpKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRBcm5QcmluY2lwYWwoYXJuOiBzdHJpbmcpIHtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMobmV3IEFyblByaW5jaXBhbChhcm4pKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGRzIGEgc2VydmljZSBwcmluY2lwYWwgdG8gdGhpcyBwb2xpY3kgc3RhdGVtZW50LlxuICAgKlxuICAgKiBAcGFyYW0gc2VydmljZSB0aGUgc2VydmljZSBuYW1lIGZvciB3aGljaCBhIHNlcnZpY2UgcHJpbmNpcGFsIGlzIHJlcXVlc3RlZCAoZS5nOiBgczMuYW1hem9uYXdzLmNvbWApLlxuICAgKiBAcGFyYW0gb3B0cyAgICBvcHRpb25zIGZvciBhZGRpbmcgdGhlIHNlcnZpY2UgcHJpbmNpcGFsIChzdWNoIGFzIHNwZWNpZnlpbmcgYSBwcmluY2lwYWwgaW4gYSBkaWZmZXJlbnQgcmVnaW9uKVxuICAgKi9cbiAgcHVibGljIGFkZFNlcnZpY2VQcmluY2lwYWwoc2VydmljZTogc3RyaW5nLCBvcHRzPzogU2VydmljZVByaW5jaXBhbE9wdHMpIHtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMobmV3IFNlcnZpY2VQcmluY2lwYWwoc2VydmljZSwgb3B0cykpO1xuICB9XG5cbiAgcHVibGljIGFkZEZlZGVyYXRlZFByaW5jaXBhbChmZWRlcmF0ZWQ6IGFueSwgY29uZGl0aW9uczoge1trZXk6IHN0cmluZ106IGFueX0pIHtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMobmV3IEZlZGVyYXRlZFByaW5jaXBhbChmZWRlcmF0ZWQsIGNvbmRpdGlvbnMpKTtcbiAgfVxuXG4gIHB1YmxpYyBhZGRBY2NvdW50Um9vdFByaW5jaXBhbCgpIHtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMobmV3IEFjY291bnRSb290UHJpbmNpcGFsKCkpO1xuICB9XG5cbiAgcHVibGljIGFkZENhbm9uaWNhbFVzZXJQcmluY2lwYWwoY2Fub25pY2FsVXNlcklkOiBzdHJpbmcpIHtcbiAgICB0aGlzLmFkZFByaW5jaXBhbHMobmV3IENhbm9uaWNhbFVzZXJQcmluY2lwYWwoY2Fub25pY2FsVXNlcklkKSk7XG4gIH1cblxuICBwdWJsaWMgYWRkQW55UHJpbmNpcGFsKCkge1xuICAgIHRoaXMuYWRkUHJpbmNpcGFscyhuZXcgQW55b25lKCkpO1xuICB9XG5cbiAgLy9cbiAgLy8gUmVzb3VyY2VzXG4gIC8vXG5cbiAgcHVibGljIGFkZFJlc291cmNlcyguLi5hcm5zOiBzdHJpbmdbXSkge1xuICAgIGlmIChhcm5zLmxlbmd0aCA+IDAgJiYgdGhpcy5ub3RSZXNvdXJjZS5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhZGQgJ1Jlc291cmNlcycgdG8gcG9saWN5IHN0YXRlbWVudCBpZiAnTm90UmVzb3VyY2VzJyBoYXZlIGJlZW4gYWRkZWRgKTtcbiAgICB9XG4gICAgdGhpcy5yZXNvdXJjZS5wdXNoKC4uLmFybnMpO1xuICB9XG5cbiAgcHVibGljIGFkZE5vdFJlc291cmNlcyguLi5hcm5zOiBzdHJpbmdbXSkge1xuICAgIGlmIChhcm5zLmxlbmd0aCA+IDAgJiYgdGhpcy5yZXNvdXJjZS5sZW5ndGggPiAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYENhbm5vdCBhZGQgJ05vdFJlc291cmNlcycgdG8gcG9saWN5IHN0YXRlbWVudCBpZiAnUmVzb3VyY2VzJyBoYXZlIGJlZW4gYWRkZWRgKTtcbiAgICB9XG4gICAgdGhpcy5ub3RSZXNvdXJjZS5wdXNoKC4uLmFybnMpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBgYFwiKlwiYGAgcmVzb3VyY2UgdG8gdGhpcyBzdGF0ZW1lbnQuXG4gICAqL1xuICBwdWJsaWMgYWRkQWxsUmVzb3VyY2VzKCkge1xuICAgIHRoaXMuYWRkUmVzb3VyY2VzKCcqJyk7XG4gIH1cblxuICAvKipcbiAgICogSW5kaWNhdGVzIGlmIHRoaXMgcGVybWlzc2lvbiBhcyBhdCBsZWFzdCBvbmUgcmVzb3VyY2UgYXNzb2NpYXRlZCB3aXRoIGl0LlxuICAgKi9cbiAgcHVibGljIGdldCBoYXNSZXNvdXJjZSgpIHtcbiAgICByZXR1cm4gdGhpcy5yZXNvdXJjZSAmJiB0aGlzLnJlc291cmNlLmxlbmd0aCA+IDA7XG4gIH1cblxuICAvL1xuICAvLyBDb25kaXRpb25cbiAgLy9cblxuICAvKipcbiAgICogQWRkIGEgY29uZGl0aW9uIHRvIHRoZSBQb2xpY3lcbiAgICovXG4gIHB1YmxpYyBhZGRDb25kaXRpb24oa2V5OiBzdHJpbmcsIHZhbHVlOiBhbnkpIHtcbiAgICB0aGlzLmNvbmRpdGlvbltrZXldID0gdmFsdWU7XG4gIH1cblxuICAvKipcbiAgICogQWRkIG11bHRpcGxlIGNvbmRpdGlvbnMgdG8gdGhlIFBvbGljeVxuICAgKi9cbiAgcHVibGljIGFkZENvbmRpdGlvbnMoY29uZGl0aW9uczoge1trZXk6IHN0cmluZ106IGFueX0pIHtcbiAgICBPYmplY3Qua2V5cyhjb25kaXRpb25zKS5tYXAoa2V5ID0+IHtcbiAgICAgIHRoaXMuYWRkQ29uZGl0aW9uKGtleSwgY29uZGl0aW9uc1trZXldKTtcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBjb25kaXRpb24gdGhhdCBsaW1pdHMgdG8gYSBnaXZlbiBhY2NvdW50XG4gICAqL1xuICBwdWJsaWMgYWRkQWNjb3VudENvbmRpdGlvbihhY2NvdW50SWQ6IHN0cmluZykge1xuICAgIHRoaXMuYWRkQ29uZGl0aW9uKCdTdHJpbmdFcXVhbHMnLCB7ICdzdHM6RXh0ZXJuYWxJZCc6IGFjY291bnRJZCB9KTtcbiAgfVxuXG4gIHB1YmxpYyB0b1N0YXRlbWVudEpzb24oKTogYW55IHtcbiAgICByZXR1cm4gbm9VbmRlZih7XG4gICAgICBBY3Rpb246IF9ub3JtKHRoaXMuYWN0aW9uKSxcbiAgICAgIE5vdEFjdGlvbjogX25vcm0odGhpcy5ub3RBY3Rpb24pLFxuICAgICAgQ29uZGl0aW9uOiBfbm9ybSh0aGlzLmNvbmRpdGlvbiksXG4gICAgICBFZmZlY3Q6IF9ub3JtKHRoaXMuZWZmZWN0KSxcbiAgICAgIFByaW5jaXBhbDogX25vcm1QcmluY2lwYWwodGhpcy5wcmluY2lwYWwpLFxuICAgICAgTm90UHJpbmNpcGFsOiBfbm9ybVByaW5jaXBhbCh0aGlzLm5vdFByaW5jaXBhbCksXG4gICAgICBSZXNvdXJjZTogX25vcm0odGhpcy5yZXNvdXJjZSksXG4gICAgICBOb3RSZXNvdXJjZTogX25vcm0odGhpcy5ub3RSZXNvdXJjZSksXG4gICAgICBTaWQ6IF9ub3JtKHRoaXMuc2lkKSxcbiAgICB9KTtcblxuICAgIGZ1bmN0aW9uIF9ub3JtKHZhbHVlczogYW55KSB7XG5cbiAgICAgIGlmICh0eXBlb2YodmFsdWVzKSA9PT0gJ3VuZGVmaW5lZCcpIHtcbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgIH1cblxuICAgICAgaWYgKGNkay5Ub2tlbi5pc1VucmVzb2x2ZWQodmFsdWVzKSkge1xuICAgICAgICByZXR1cm4gdmFsdWVzO1xuICAgICAgfVxuXG4gICAgICBpZiAoQXJyYXkuaXNBcnJheSh2YWx1ZXMpKSB7XG4gICAgICAgIGlmICghdmFsdWVzIHx8IHZhbHVlcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG5cbiAgICAgICAgaWYgKHZhbHVlcy5sZW5ndGggPT09IDEpIHtcbiAgICAgICAgICByZXR1cm4gdmFsdWVzWzBdO1xuICAgICAgICB9XG5cbiAgICAgICAgcmV0dXJuIHZhbHVlcztcbiAgICAgIH1cblxuICAgICAgaWYgKHR5cGVvZih2YWx1ZXMpID09PSAnb2JqZWN0Jykge1xuICAgICAgICBpZiAoT2JqZWN0LmtleXModmFsdWVzKS5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICB9XG5cbiAgICAgIHJldHVybiB2YWx1ZXM7XG4gICAgfVxuXG4gICAgZnVuY3Rpb24gX25vcm1QcmluY2lwYWwocHJpbmNpcGFsOiB7IFtrZXk6IHN0cmluZ106IGFueVtdIH0pIHtcbiAgICAgIGNvbnN0IGtleXMgPSBPYmplY3Qua2V5cyhwcmluY2lwYWwpO1xuICAgICAgaWYgKGtleXMubGVuZ3RoID09PSAwKSB7IHJldHVybiB1bmRlZmluZWQ7IH1cbiAgICAgIGNvbnN0IHJlc3VsdDogYW55ID0ge307XG4gICAgICBmb3IgKGNvbnN0IGtleSBvZiBrZXlzKSB7XG4gICAgICAgIGNvbnN0IG5vcm1WYWwgPSBfbm9ybShwcmluY2lwYWxba2V5XSk7XG4gICAgICAgIGlmIChub3JtVmFsKSB7XG4gICAgICAgICAgcmVzdWx0W2tleV0gPSBub3JtVmFsO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgICBpZiAoT2JqZWN0LmtleXMocmVzdWx0KS5sZW5ndGggPT09IDEgJiYgcmVzdWx0LkFXUyA9PT0gJyonKSB7XG4gICAgICAgIHJldHVybiAnKic7XG4gICAgICB9XG4gICAgICByZXR1cm4gcmVzdWx0O1xuICAgIH1cbiAgfVxuXG4gIHB1YmxpYyB0b1N0cmluZygpIHtcbiAgICByZXR1cm4gY2RrLlRva2VuLmFzU3RyaW5nKHRoaXMsIHtcbiAgICAgIGRpc3BsYXlIaW50OiAnUG9saWN5U3RhdGVtZW50J1xuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEpTT04taWZ5IHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogVXNlZCB3aGVuIEpTT04uc3RyaW5naWZ5KCkgaXMgY2FsbGVkXG4gICAqL1xuICBwdWJsaWMgdG9KU09OKCkge1xuICAgIHJldHVybiB0aGlzLnRvU3RhdGVtZW50SnNvbigpO1xuICB9XG59XG5cbmV4cG9ydCBlbnVtIEVmZmVjdCB7XG4gIEFMTE9XID0gJ0FsbG93JyxcbiAgREVOWSA9ICdEZW55Jyxcbn1cblxuLyoqXG4gKiBJbnRlcmZhY2UgZm9yIGNyZWF0aW5nIGEgcG9saWN5IHN0YXRlbWVudFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFBvbGljeVN0YXRlbWVudFByb3BzIHtcbiAgLyoqXG4gICAqIExpc3Qgb2YgYWN0aW9ucyB0byBhZGQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGFjdGlvbnNcbiAgICovXG4gIHJlYWRvbmx5IGFjdGlvbnM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogTGlzdCBvZiBub3QgYWN0aW9ucyB0byBhZGQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIG5vdC1hY3Rpb25zXG4gICAqL1xuICByZWFkb25seSBub3RBY3Rpb25zPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2YgcHJpbmNpcGFscyB0byBhZGQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHByaW5jaXBhbHNcbiAgICovXG4gIHJlYWRvbmx5IHByaW5jaXBhbHM/OiBJUHJpbmNpcGFsW107XG5cbiAgLyoqXG4gICAqIExpc3Qgb2Ygbm90IHByaW5jaXBhbHMgdG8gYWRkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBub3QgcHJpbmNpcGFsc1xuICAgKi9cbiAgcmVhZG9ubHkgbm90UHJpbmNpcGFscz86IElQcmluY2lwYWxbXTtcblxuICAvKipcbiAgICogUmVzb3VyY2UgQVJOcyB0byBhZGQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIHJlc291cmNlc1xuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VzPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIE5vdFJlc291cmNlIEFSTnMgdG8gYWRkIHRvIHRoZSBzdGF0ZW1lbnRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBubyBub3QtcmVzb3VyY2VzXG4gICAqL1xuICByZWFkb25seSBub3RSZXNvdXJjZXM/OiBzdHJpbmdbXTtcblxuICAvKipcbiAgICogQ29uZGl0aW9ucyB0byBhZGQgdG8gdGhlIHN0YXRlbWVudFxuICAgKlxuICAgKiBAZGVmYXVsdCAtIG5vIGNvbmRpdGlvblxuICAgKi9cbiAgcmVhZG9ubHkgY29uZGl0aW9ucz86IHtba2V5OiBzdHJpbmddOiBhbnl9O1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRvIGFsbG93IG9yIGRlbnkgdGhlIGFjdGlvbnMgaW4gdGhpcyBzdGF0ZW1lbnRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBhbGxvd1xuICAgKi9cbiAgcmVhZG9ubHkgZWZmZWN0PzogRWZmZWN0O1xufVxuXG5mdW5jdGlvbiBub1VuZGVmKHg6IGFueSk6IGFueSB7XG4gIGNvbnN0IHJldDogYW55ID0ge307XG4gIGZvciAoY29uc3QgW2tleSwgdmFsdWVdIG9mIE9iamVjdC5lbnRyaWVzKHgpKSB7XG4gICAgaWYgKHZhbHVlICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldFtrZXldID0gdmFsdWU7XG4gICAgfVxuICB9XG4gIHJldHVybiByZXQ7XG59Il19