"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const iam = require("../lib");
describe('automatic cross-stack references', () => {
    test('automatic exports are created when attributes are referneced across stacks', () => {
        // GIVEN
        const app = new cdk.App();
        const stackWithUser = new cdk.Stack(app, 'stack1');
        const stackWithGroup = new cdk.Stack(app, 'stack2');
        const user = new iam.User(stackWithUser, 'User');
        const group = new iam.Group(stackWithGroup, 'Group');
        // WHEN
        group.addUser(user);
        //
        // `group.addUser` adds the group to the user resource definition, so we expect
        // that an automatic export will be created for the group and the user's stack
        // to use ImportValue to import it.
        // note that order of "expect"s matters. we first need to synthesize the user's
        // stack so that the cross stack reference will be reported and only then the
        // group's stack. in the real world, App will take care of this.
        //
        // THEN
        expect(stackWithUser).toMatchTemplate({
            Resources: {
                User00B015A1: {
                    Type: "AWS::IAM::User",
                    Properties: {
                        Groups: [{ "Fn::ImportValue": "stack2:ExportsOutputRefGroupC77FDACD8CF7DD5B" }]
                    }
                }
            }
        });
        expect(stackWithGroup).toMatchTemplate({
            Outputs: {
                ExportsOutputRefGroupC77FDACD8CF7DD5B: {
                    Value: { Ref: "GroupC77FDACD" },
                    Export: { Name: "stack2:ExportsOutputRefGroupC77FDACD8CF7DD5B" }
                }
            },
            Resources: {
                GroupC77FDACD: {
                    Type: "AWS::IAM::Group"
                }
            }
        });
    });
    test('cannot reference tokens across apps', () => {
        // GIVEN
        const stack1 = new cdk.Stack();
        const stack2 = new cdk.Stack();
        const user = new iam.User(stack1, 'User');
        const group = new iam.Group(stack2, 'Group');
        group.addUser(user);
        // THEN
        expect(() => assert_1.SynthUtils.synthesize(stack1)).toThrow(/Cannot reference across apps/);
    });
});
//# sourceMappingURL=data:application/json;base64,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