"""The unpolarized, space-like |N3LO| quark-quark non-singlet |OME|."""
import numba as nb
import numpy as np

from .....harmonics import cache as c
from .....harmonics.log_functions import lm11, lm12, lm13


@nb.njit(cache=True)
def A_qqNS(n, cache, nf, L, eta):
    r"""Compute the |N3LO| non singlet |OME| :math:`A_{qq}^{NS,(3)}(N)`.

    The expression is presented in :cite:`Bierenbaum:2009mv` and
    :cite:`Ablinger:2014vwa`. It contains some weight 5 harmonics sums.

    When using the code, please cite the complete list of references
    available in :mod:`ekore.matching_conditions.as3`.

    Note the part proportional to nf^0 includes weight = 5
    harmonics and has been parametrized in Mellin space.
    For this piece the accuracy wrt the known moments is below the 0.01% (N<1000)
    and the absolute diff is within 5e-3.
    All the other contributions are provided exact.

    Parameters
    ----------
    n : complex
        Mellin moment
    cache: numpy.ndarray
        Harmonic sum cache
    nf : int
        number of active flavor below the threshold
    L : float
        :math:`\ln(\mu_F^2 / m_h^2)`
    eta: int
        :math:`(-1)^n` factor to be continued with 1 for singlet like and -1 for non-singlet like

    Returns
    -------
    complex
        :math:`A_{qq}^{NS,(3)}(N)`

    """
    is_singlet = eta == 1
    S1 = c.get(c.S1, cache, n)
    S2 = c.get(c.S2, cache, n)
    Sm2 = c.get(c.Sm2, cache, n, is_singlet)
    S3 = c.get(c.S3, cache, n)
    S21 = c.get(c.S21, cache, n)
    Sm21 = c.get(c.Sm21, cache, n, is_singlet)
    Sm3 = c.get(c.Sm3, cache, n, is_singlet)
    S4 = c.get(c.S4, cache, n)
    S31 = c.get(c.S31, cache, n)
    S211 = c.get(c.S211, cache, n)
    Sm22 = c.get(c.Sm22, cache, n, is_singlet)
    Sm211 = c.get(c.Sm211, cache, n, is_singlet)
    Sm4 = c.get(c.Sm4, cache, n, is_singlet)
    S5 = c.get(c.S5, cache, n)
    Sm5 = c.get(c.Sm5, cache, n, is_singlet)

    a_qqNS_l0_nf1 = (
        9.972841687007257 / np.power(1.0 + n, 5)
        + 0.7901234567901235 / (np.power(n, 4) * np.power(1.0 + n, 5))
        + 1.0534979423868314 / (np.power(n, 3) * np.power(1.0 + n, 5))
        - 3.4238683127572016 / (np.power(n, 2) * np.power(1.0 + n, 5))
        - 4.3551062252579404 / (n * np.power(1.0 + n, 5))
        + (22.86861739934704 * n) / np.power(1.0 + n, 5)
        + (17.894957028741118 * np.power(n, 2)) / np.power(1.0 + n, 5)
        + (8.836818914391468 * np.power(n, 3)) / np.power(1.0 + n, 5)
        + (3.9622177138576284 * np.power(n, 4)) / np.power(1.0 + n, 5)
        + (0.7924435427715257 * np.power(n, 5)) / np.power(1.0 + n, 5)
        - (3.405013886932678 * S1) / np.power(1.0 + n, 5)
        - (17.025069434663394 * n * S1) / np.power(1.0 + n, 5)
        - (34.05013886932679 * np.power(n, 2) * S1) / np.power(1.0 + n, 5)
        - (34.05013886932679 * np.power(n, 3) * S1) / np.power(1.0 + n, 5)
        - (17.025069434663394 * np.power(n, 4) * S1) / np.power(1.0 + n, 5)
        - (3.405013886932678 * np.power(n, 5) * S1) / np.power(1.0 + n, 5)
        + (0.5267489711934157 * S2) / np.power(1.0 + n, 5)
        + (2.633744855967078 * n * S2) / np.power(1.0 + n, 5)
        + (5.267489711934156 * np.power(n, 2) * S2) / np.power(1.0 + n, 5)
        + (5.267489711934156 * np.power(n, 3) * S2) / np.power(1.0 + n, 5)
        + (2.633744855967078 * np.power(n, 4) * S2) / np.power(1.0 + n, 5)
        + (0.5267489711934157 * np.power(n, 5) * S2) / np.power(1.0 + n, 5)
        + (2.633744855967078 * S3) / np.power(1.0 + n, 5)
        + (13.168724279835391 * n * S3) / np.power(1.0 + n, 5)
        + (26.337448559670783 * np.power(n, 2) * S3) / np.power(1.0 + n, 5)
        + (26.337448559670783 * np.power(n, 3) * S3) / np.power(1.0 + n, 5)
        + (13.168724279835391 * np.power(n, 4) * S3) / np.power(1.0 + n, 5)
        + (2.633744855967078 * np.power(n, 5) * S3) / np.power(1.0 + n, 5)
        - (1.5802469135802468 * S4) / np.power(1.0 + n, 5)
        - (7.901234567901234 * n * S4) / np.power(1.0 + n, 5)
        - (15.802469135802468 * np.power(n, 2) * S4) / np.power(1.0 + n, 5)
        - (15.802469135802468 * np.power(n, 3) * S4) / np.power(1.0 + n, 5)
        - (7.901234567901234 * np.power(n, 4) * S4) / np.power(1.0 + n, 5)
        - (1.5802469135802468 * np.power(n, 5) * S4) / np.power(1.0 + n, 5)
    )
    a_qqNS_l0_nf0 = (
        12.641975308641976
        + n * (17.11934156378601 - 27.259259259259256 * S1)
        + eta
        * (
            -1.580246913580247
            + n * (-2.8971193415637853 - 1.5802469135802468 * S1)
            + np.power(n, 8) * (-0.8559670781893005 + 0.3950617283950617 * S1)
            + np.power(n, 7) * (-3.0288065843621395 + 1.5802469135802468 * S1)
            + np.power(n, 4)
            * (
                -28.159122085048004
                + 15.275720164609055 * S1
                - 5.530864197530864 * np.power(S1, 2)
                - 5.530864197530864 * S2
            )
            + np.power(n, 5)
            * (
                -30.09053497942388
                + 19.358024691358025 * S1
                - 4.7407407407407405 * np.power(S1, 2)
                - 4.7407407407407405 * S2
            )
            + np.power(n, 3)
            * (
                -10.644718792866943
                + 3.818930041152263 * S1
                - 3.1604938271604937 * np.power(S1, 2)
                - 3.1604938271604937 * S2
            )
            + np.power(n, 6)
            * (
                -18.589849108367623
                + 10.403292181069954 * S1
                - 1.5802469135802468 * np.power(S1, 2)
                - 1.5802469135802468 * S2
            )
            + np.power(n, 2)
            * (
                -2.2167352537722906
                - 2.897119341563786 * S1
                - 0.7901234567901234 * np.power(S1, 2)
                - 0.7901234567901234 * S2
            )
        )
        + np.power(n, 11)
        * (57.69873135166052 - 21.333333333333332 * S21 - 5.333333333333333 * S3)
        + np.power(n, 2)
        * (
            33.536351165980804
            - 77.30041152263375 * S1
            + 5.530864197530864 * np.power(S1, 2)
            - 7.111111111111111 * S2
            + 1.5802469135802468 * Sm2
        )
        + np.power(n, 3)
        * (
            149.33749503118256
            + 22.814814814814817 * np.power(S1, 2)
            - 0.5267489711934156 * np.power(S1, 3)
            - 16.16460905349794 * S2
            + 7.901234567901234 * S21
            - 0.2633744855967078 * S3
            + S1
            * (-94.90260631001371 + 0.7901234567901234 * S2 - 1.5802469135802468 * Sm2)
            + 2.8971193415637857 * Sm2
            + 3.1604938271604937 * Sm21
        )
        + np.power(n, 4)
        * (
            313.3204581303606
            - 2.633744855967078 * np.power(S1, 3)
            - 11.050754458161864 * S2
            - 9.28395061728395 * np.power(S2, 2)
            + 26.2716049382716 * S21
            + 22.51851851851852 * S211
            + 13.958847736625515 * S3
            - 5.530864197530864 * S31
            - 11.65432098765432 * S4
            + 8.252400548696844 * Sm2
            + np.power(S1, 2)
            * (38.81481481481481 + 0.7901234567901234 * S2 + 1.5802469135802468 * Sm2)
            + 5.794238683127571 * Sm21
            + 6.320987654320987 * Sm211
            - 1.5802469135802468 * Sm22
            - 3.423868312757201 * Sm3
            + S1
            * (
                -141.92375385883807
                + 15.407407407407407 * S2
                - 15.802469135802468 * S21
                - 1.1851851851851851 * S3
                - 6.320987654320987 * Sm2
                - 3.1604938271604937 * Sm21
                + 1.5802469135802468 * Sm3
            )
            + 1.5802469135802468 * Sm4
        )
        + np.power(n, 8)
        * (
            -70.47890593844664
            - 305.90946502057614 * np.power(S2, 2)
            - 71.70370370370368 * S21
            + 360.2962962962963 * S211
            - 728.4499314128943 * S3
            - 538.2057613168724 * S31
            + 308.0164609053498 * S4
            + 7.901234567901236 * S5
            + np.power(S1, 3) * (-5.267489711934158 * S2 - 10.534979423868316 * Sm2)
            + 15.890260631001372 * Sm2
            + 63.20987654320989 * S21 * Sm2
            - 21.069958847736633 * S3 * Sm2
            + 189.80521262002742 * Sm21
            - 204.37860082304525 * Sm211
            + 51.09465020576131 * Sm22
            + S2
            * (
                120.24621135293253
                - 426.66666666666674 * S21
                + 451.68724279835396 * S3
                + 52.674897119341566 * Sm2
                - 94.81481481481484 * Sm21
                - 15.802469135802472 * Sm3
            )
            + np.power(S1, 2)
            * (
                -378.23845592465034
                + 0.7901234567901234 * S2
                + 158.0246913580247 * S21
                + 11.851851851851855 * S3
                + 1.5802469135802468 * Sm2
                + 31.604938271604944 * Sm21
                - 15.802469135802472 * Sm3
            )
            - 103.33058984910835 * Sm3
            + S1
            * (
                -142.9973249821577
                + 803.5994513031549 * S2
                + 185.67901234567904 * np.power(S2, 2)
                - 175.80246913580245 * S21
                - 450.37037037037044 * S211
                - 132.87242798353907 * S3
                + 110.61728395061729 * S31
                + 233.0864197530865 * S4
                - 198.23319615912206 * Sm2
                + 102.18930041152262 * Sm21
                - 126.41975308641977 * Sm211
                + 31.604938271604944 * Sm22
                + 54.25514403292181 * Sm3
                - 31.604938271604944 * Sm4
            )
            + 54.25514403292181 * Sm4
            - 31.604938271604944 * Sm5
        )
        + np.power(n, 7)
        * (
            265.13117338652285
            - 333.761316872428 * np.power(S2, 2)
            + 49.77777777777778 * S21
            + 427.85185185185185 * S211
            - 668.3347050754458 * S3
            - 554.798353909465 * S31
            + 273.0534979423868 * S4
            + 7.901234567901236 * S5
            + np.power(S1, 3)
            * (-1.0534979423868311 - 5.267489711934158 * S2 - 10.534979423868316 * Sm2)
            + 55.13305898491083 * Sm2
            + 63.20987654320989 * S21 * Sm2
            - 21.069958847736633 * S3 * Sm2
            + 172.42249657064468 * Sm21
            - 185.41563786008228 * Sm211
            + 46.35390946502057 * Sm22
            + S2
            * (
                5.875840982562162
                - 426.66666666666674 * S21
                + 451.68724279835396 * S3
                + 52.674897119341566 * Sm2
                - 94.81481481481484 * Sm21
                - 15.802469135802472 * Sm3
            )
            + np.power(S1, 2)
            * (
                -368.55944357897135
                + 3.1604938271604937 * S2
                + 158.0246913580247 * S21
                + 11.851851851851855 * S3
                + 6.320987654320987 * Sm2
                + 31.604938271604944 * Sm21
                - 15.802469135802472 * Sm3
            )
            - 116.76268861454045 * Sm3
            + S1
            * (
                -100.79602140092175
                + 843.5006858710562 * S2
                + 185.67901234567904 * np.power(S2, 2)
                - 223.20987654320987 * S21
                - 450.37037037037044 * S211
                - 136.42798353909464 * S3
                + 110.61728395061729 * S31
                + 233.0864197530865 * S4
                - 202.97393689986282 * Sm2
                + 92.70781893004114 * Sm21
                - 126.41975308641977 * Sm211
                + 31.604938271604944 * Sm22
                + 58.995884773662546 * Sm3
                - 31.604938271604944 * Sm4
            )
            + 58.995884773662546 * Sm4
            - 31.604938271604944 * Sm5
        )
        + np.power(n, 9)
        * (
            30.846395931033122
            - 148.3127572016461 * np.power(S2, 2)
            - 126.02469135802468 * S21
            + 168.88888888888889 * S211
            - 402.67764060356654 * S3
            - 266.33744855967075 * S31
            + 159.83539094650206 * S4
            + 3.950617283950618 * S5
            + np.power(S1, 3) * (-2.633744855967079 * S2 - 5.267489711934158 * Sm2)
            + 31.604938271604944 * S21 * Sm2
            - 10.534979423868316 * S3 * Sm2
            + 98.32647462277092 * Sm21
            - 105.34979423868312 * Sm211
            + 26.33744855967078 * Sm22
            + S2
            * (
                97.28634298785173
                - 213.33333333333337 * S21
                + 225.84362139917698 * S3
                + 26.337448559670783 * Sm2
                - 47.40740740740742 * Sm21
                - 7.901234567901236 * Sm3
            )
            + np.power(S1, 2)
            * (
                -191.24268475244867
                + 79.01234567901236 * S21
                + 5.925925925925927 * S3
                + 15.802469135802472 * Sm21
                - 7.901234567901236 * Sm3
            )
            - 49.16323731138546 * Sm3
            + S1
            * (
                -106.13898034903957
                + 382.63923182441704 * S2
                + 92.83950617283952 * np.power(S2, 2)
                - 80.0 * S21
                - 225.18518518518522 * S211
                - 65.84362139917695 * S3
                + 55.308641975308646 * S31
                + 116.54320987654324 * S4
                - 98.32647462277092 * Sm2
                + 52.67489711934156 * Sm21
                - 63.20987654320989 * Sm211
                + 15.802469135802472 * Sm22
                + 26.33744855967078 * Sm3
                - 15.802469135802472 * Sm4
            )
            + 26.33744855967078 * Sm4
            - 15.802469135802472 * Sm5
        )
        + np.power(n, 6)
        * (
            574.765958897151
            - 204.0164609053498 * np.power(S2, 2)
            + 99.95061728395063 * S21
            + 304.0 * S211
            - 282.6447187928669 * S3
            - 299.52263374485597 * S31
            + 89.90946502057614 * S4
            + 3.950617283950618 * S5
            + np.power(S1, 3)
            * (-3.687242798353909 - 2.633744855967079 * S2 - 5.267489711934158 * Sm2)
            + 69.79423868312756 * Sm2
            + 31.604938271604944 * S21 * Sm2
            - 10.534979423868316 * S3 * Sm2
            + 69.88203017832647 * Sm21
            - 67.4238683127572 * Sm211
            + 16.8559670781893 * Sm22
            + S2
            * (
                -70.51612614795073
                - 213.33333333333337 * S21
                + 225.84362139917698 * S3
                + 26.337448559670783 * Sm2
                - 47.40740740740742 * Sm21
                - 7.901234567901236 * Sm3
            )
            + np.power(S1, 2)
            * (
                -165.26737611047332
                + 4.7407407407407405 * S2
                + 79.01234567901236 * S21
                + 5.925925925925927 * S3
                + 9.481481481481481 * Sm2
                + 15.802469135802472 * Sm21
                - 7.901234567901236 * Sm3
            )
            - 74.44718792866941 * Sm3
            + S1
            * (
                -140.3207353264853
                + 462.8367626886146 * S2
                + 92.83950617283952 * np.power(S2, 2)
                - 174.8148148148148 * S21
                - 225.18518518518522 * S211
                - 72.95473251028807 * S3
                + 55.308641975308646 * S31
                + 116.54320987654324 * S4
                - 109.38820301783264 * Sm2
                + 33.7119341563786 * Sm21
                - 63.20987654320989 * Sm211
                + 15.802469135802472 * Sm22
                + 35.81893004115226 * Sm3
                - 15.802469135802472 * Sm4
            )
            + 35.81893004115226 * Sm4
            - 15.802469135802472 * Sm5
        )
        + np.power(n, 5)
        * (
            495.9021164140525
            - 66.79835390946502 * np.power(S2, 2)
            + 62.81481481481482 * S21
            + 123.85185185185182 * S211
            - 19.906721536351156 * S3
            - 75.3909465020576 * S31
            - 14.650205761316872 * S4
            + 0.7901234567901235 * S5
            + np.power(S1, 3)
            * (-4.7407407407407405 - 0.5267489711934156 * S2 - 1.0534979423868311 * Sm2)
            + 37.48696844993141 * Sm2
            + 6.320987654320988 * S21 * Sm2
            - 2.1069958847736623 * S3 * Sm2
            + 11.237311385459535 * Sm21
            + 4.213991769547325 * Sm211
            - 1.0534979423868311 * Sm22
            + S2
            * (
                -33.30542001697012
                - 42.66666666666667 * S21
                + 45.1687242798354 * S3
                + 5.267489711934158 * Sm2
                - 9.481481481481481 * Sm21
                - 1.580246913580247 * Sm3
            )
            + np.power(S1, 2)
            * (
                -0.6386604072798434
                + 3.1604938271604937 * S2
                + 15.802469135802472 * S21
                + 1.1851851851851851 * S3
                + 6.320987654320987 * Sm2
                + 3.160493827160494 * Sm21
                - 1.580246913580247 * Sm3
            )
            - 25.108367626886142 * Sm3
            + S1
            * (
                -195.1472285147626
                + 126.4636488340192 * S2
                + 18.567901234567906 * np.power(S2, 2)
                - 79.20987654320989 * S21
                - 45.03703703703704 * S211
                - 17.90946502057613 * S3
                + 11.06172839506173 * S31
                + 23.308641975308646 * S4
                - 30.727023319615913 * Sm2
                - 2.1069958847736623 * Sm21
                - 12.641975308641976 * Sm211
                + 3.160493827160494 * Sm22
                + 11.588477366255143 * Sm3
                - 3.160493827160494 * Sm4
            )
            + 11.588477366255143 * Sm4
            - 3.160493827160494 * Sm5
        )
        + np.power(n, 10)
        * (
            145.9284284602288
            - 29.662551440329217 * np.power(S2, 2)
            - 81.38271604938268 * S21
            + 33.77777777777777 * S211
            - 99.37997256515774 * S3
            - 53.26748971193416 * S31
            + 31.967078189300413 * S4
            + 0.7901234567901234 * S5
            + np.power(S1, 3) * (-0.5267489711934156 * S2 - 1.0534979423868311 * Sm2)
            + 6.320987654320987 * S21 * Sm2
            - 2.1069958847736623 * S3 * Sm2
            + 19.665294924554182 * Sm21
            - 21.069958847736622 * Sm211
            + 5.2674897119341555 * Sm22
            + S2
            * (
                22.12393526423702
                - 42.666666666666664 * S21
                + 45.16872427983539 * S3
                + 5.267489711934156 * Sm2
                - 9.481481481481481 * Sm21
                - 1.5802469135802468 * Sm3
            )
            + np.power(S1, 2)
            * (
                -38.465820901107016
                + 15.80246913580247 * S21
                + 1.1851851851851851 * S3
                + 3.1604938271604937 * Sm21
                - 1.5802469135802468 * Sm3
            )
            - 9.832647462277091 * Sm3
            + S1
            * (
                -21.70187014388199
                + 71.55006858710561 * S2
                + 18.567901234567906 * np.power(S2, 2)
                - 16.0 * S21
                - 45.03703703703704 * S211
                - 13.16872427983539 * S3
                + 11.061728395061728 * S31
                + 23.308641975308646 * S4
                - 19.665294924554182 * Sm2
                + 10.534979423868311 * Sm21
                - 12.641975308641975 * Sm211
                + 3.1604938271604937 * Sm22
                + 5.2674897119341555 * Sm3
                - 3.1604938271604937 * Sm4
            )
            + 5.2674897119341555 * Sm4
            - 3.1604938271604937 * Sm5
        )
    ) / (np.power(n, 5) * np.power(1.0 + n, 5))
    # Parametrized part
    if eta == 1:
        a_qqNS_l0_nf0_param = (
            147.3377644356437
            - 246.74348741339384 / n
            - 19.88581621175457 / (1.0 + n)
            + 2.216741536244232 / (2.0 + n)
            + 2.710088622384316 / (3.0 + n)
            + (81.82927934620507 * S1) / np.power(n, 2)
            - (143.78336783179037 * S1) / n
            - (1.2446110147965006 * S1) / np.power(1.0 + n, 4)
            + (3.71973600774796 * S1) / np.power(1.0 + n, 3)
            + (8.517401423204326 * S1) / np.power(1.0 + n, 2)
            - (29.02336180730714 * S1) / (1.0 + n)
            + (31.039302071700547 * np.power(S1, 2)) / np.power(n, 2)
            - (28.510035947808422 * np.power(S1, 2)) / n
            + (5.777341175835097 * np.power(S1, 3)) / np.power(n, 2)
            + (31.039302071700547 * S2) / np.power(n, 2)
            + (16.00539778660784 * S2) / n
            + (17.332023527505292 * S1 * S2) / np.power(n, 2)
            + (62.078604143401094 * S1 * S2) / n
            + (17.332023527505292 * np.power(S1, 2) * S2) / n
            + (17.332023527505292 * np.power(S2, 2)) / n
            + (11.554682351670195 * S3) / np.power(n, 2)
            + (62.078604143401094 * S3) / n
            + (34.664047055010585 * S1 * S3) / n
            + (34.664047055010585 * S4) / n
            + 27.26800934312273 * lm11(n, S1)
            - 24.747353827403803 * lm12(n, S1, S2)
            + 8.301615182417493 * lm13(n, S1, S2, S3)
        )
    else:
        a_qqNS_l0_nf0_param = (
            147.3378361891409
            - 46.47711367482187 / n
            - 80.35959476650793 / (1.0 + n)
            + 27.743026359702743 / (2.0 + n)
            - 8.598732660114614 / (3.0 + n)
            - (3.5925418806363756 * S1) / np.power(n, 2)
            - (70.69741634890381 * S1) / n
            - (4.737695897307676 * S1) / np.power(1.0 + n, 4)
            + (11.854495310365479 * S1) / np.power(1.0 + n, 3)
            - (1.0738877081747489 * S1) / np.power(1.0 + n, 2)
            - (70.56128531216365 * S1) / (1.0 + n)
            + (20.192883374025968 * np.power(S1, 2)) / np.power(n, 2)
            - (2.91852008643835 * np.power(S1, 2)) / n
            + (0.5914158192837431 * np.power(S1, 3)) / np.power(n, 2)
            + (20.192883374025968 * S2) / np.power(n, 2)
            - (9.274399189417498 * S2) / n
            + (1.7742474578512293 * S1 * S2) / np.power(n, 2)
            + (40.385766748051935 * S1 * S2) / n
            + (1.7742474578512293 * np.power(S1, 2) * S2) / n
            + (1.7742474578512293 * np.power(S2, 2)) / n
            + (1.1828316385674862 * S3) / np.power(n, 2)
            + (40.385766748051935 * S3) / n
            + (3.5484949157024586 * S1 * S3) / n
            + (3.5484949157024586 * S4) / n
            - 10.773506968172986 * lm11(n, S1)
            - 24.339074078635456 * lm12(n, S1, S2)
            + 8.318355694393711 * lm13(n, S1, S2, S3)
        )
    a_qqNS_l0 = a_qqNS_l0_nf0 + a_qqNS_l0_nf0_param + nf * a_qqNS_l0_nf1
    a_qqNS_l3 = (
        8.592592592592593 / (1.0 + n)
        + 5.728395061728395 / (n * (1.0 + n))
        + (8.592592592592593 * n) / (1.0 + n)
        - (0.5925925925925926 * nf) / (1.0 + n)
        - (0.3950617283950617 * nf) / (n * (1.0 + n))
        - (0.5925925925925926 * n * nf) / (1.0 + n)
        - 11.45679012345679 * S1
        + 0.7901234567901234 * nf * S1
    )
    a_qqNS_l2 = (
        113.33333333333333 / np.power(1.0 + n, 3)
        + 5.925925925925926 / (np.power(n, 3) * np.power(1.0 + n, 3))
        - (1.1851851851851851 * eta) / (np.power(n, 3) * np.power(1.0 + n, 3))
        + 21.333333333333332 / (np.power(n, 2) * np.power(1.0 + n, 3))
        - (2.3703703703703702 * eta) / (np.power(n, 2) * np.power(1.0 + n, 3))
        + 60.74074074074074 / (n * np.power(1.0 + n, 3))
        - (2.3703703703703702 * eta) / (n * np.power(1.0 + n, 3))
        + (124.0 * n) / np.power(1.0 + n, 3)
        + (89.33333333333333 * np.power(n, 2)) / np.power(1.0 + n, 3)
        + (29.77777777777778 * np.power(n, 3)) / np.power(1.0 + n, 3)
        + 4.6419753086419755 / np.power(1.0 + n, 2)
        - 1.1851851851851851 / (np.power(n, 2) * np.power(1.0 + n, 2))
        + 1.9753086419753085 / (n * np.power(1.0 + n, 2))
        + (0.5925925925925926 * n) / np.power(1.0 + n, 2)
        + (0.2962962962962963 * np.power(n, 2)) / np.power(1.0 + n, 2)
        - 44.839506172839506 * S1
        - (7.111111111111111 * S1) / np.power(1.0 + n, 2)
        - (9.481481481481481 * S1) / (np.power(n, 2) * np.power(1.0 + n, 2))
        - (18.962962962962962 * S1) / (n * np.power(1.0 + n, 2))
        - (14.222222222222221 * n * S1) / np.power(1.0 + n, 2)
        - (7.111111111111111 * np.power(n, 2) * S1) / np.power(1.0 + n, 2)
        + 2.3703703703703702 * S2
        - (14.222222222222221 * S2) / (1.0 + n)
        - (9.481481481481481 * S2) / (n * (1.0 + n))
        - (14.222222222222221 * n * S2) / (1.0 + n)
        + 18.962962962962962 * S1 * S2
        - 2.3703703703703702 * S3
        + (2.3703703703703702 * Sm2) / (n * (1.0 + n))
        - 4.7407407407407405 * S1 * Sm2
        + 4.7407407407407405 * Sm21
        - 2.3703703703703702 * Sm3
    )
    a_qqNS_l1 = (
        -198.09876543209876 / np.power(1.0 + n, 4)
        + (13.82716049382716 * eta) / np.power(1.0 + n, 4)
        + 16.59259259259259 / (np.power(n, 4) * np.power(1.0 + n, 4))
        - (2.3703703703703702 * eta) / (np.power(n, 4) * np.power(1.0 + n, 4))
        + 14.024691358024691 / (np.power(n, 3) * np.power(1.0 + n, 4))
        - (1.9753086419753085 * eta) / (np.power(n, 3) * np.power(1.0 + n, 4))
        + 31.80246913580247 / (np.power(n, 2) * np.power(1.0 + n, 4))
        + (7.703703703703703 * eta) / (np.power(n, 2) * np.power(1.0 + n, 4))
        + 34.074074074074076 / (n * np.power(1.0 + n, 4))
        + (15.209876543209877 * eta) / (n * np.power(1.0 + n, 4))
        - (113.77777777777777 * n) / np.power(1.0 + n, 4)
        + (1.7777777777777777 * eta * n) / np.power(1.0 + n, 4)
        + (445.28395061728395 * np.power(n, 2)) / np.power(1.0 + n, 4)
        + (0.5925925925925926 * eta * np.power(n, 2)) / np.power(1.0 + n, 4)
        + (469.037037037037 * np.power(n, 3)) / np.power(1.0 + n, 4)
        + (117.25925925925925 * np.power(n, 4)) / np.power(1.0 + n, 4)
        - 70.23315829196848 / (1.0 + n)
        - 46.82210552797899 / (n * (1.0 + n))
        - (70.23315829196848 * n) / (1.0 + n)
        - (16.016460905349795 * nf) / np.power(1.0 + n, 3)
        + (1.1851851851851851 * nf) / (np.power(n, 3) * np.power(1.0 + n, 3))
        - (0.7901234567901234 * nf) / (np.power(n, 2) * np.power(1.0 + n, 3))
        - (8.823045267489713 * nf) / (n * np.power(1.0 + n, 3))
        - (25.267489711934157 * n * nf) / np.power(1.0 + n, 3)
        - (25.925925925925927 * np.power(n, 2) * nf) / np.power(1.0 + n, 3)
        - (8.641975308641975 * np.power(n, 3) * nf) / np.power(1.0 + n, 3)
        + 93.64421105595798 * S1
        + (143.60493827160494 * S1) / np.power(1.0 + n, 3)
        - (29.62962962962963 * S1) / (np.power(n, 3) * np.power(1.0 + n, 3))
        - (2.3703703703703702 * eta * S1) / (np.power(n, 3) * np.power(1.0 + n, 3))
        - (27.65432098765432 * S1) / (np.power(n, 2) * np.power(1.0 + n, 3))
        - (4.7407407407407405 * eta * S1) / (np.power(n, 2) * np.power(1.0 + n, 3))
        + (73.48148148148148 * S1) / (n * np.power(1.0 + n, 3))
        - (4.7407407407407405 * eta * S1) / (n * np.power(1.0 + n, 3))
        + (127.4074074074074 * n * S1) / np.power(1.0 + n, 3)
        + (127.4074074074074 * np.power(n, 2) * S1) / np.power(1.0 + n, 3)
        + (42.46913580246913 * np.power(n, 3) * S1) / np.power(1.0 + n, 3)
        + 8.954732510288066 * nf * S1
        - 132.74074074074073 * S2
        + (62.41975308641975 * S2) / np.power(1.0 + n, 2)
        - (18.962962962962962 * S2) / (np.power(n, 2) * np.power(1.0 + n, 2))
        + (12.641975308641975 * S2) / (n * np.power(1.0 + n, 2))
        + (23.703703703703702 * n * S2) / np.power(1.0 + n, 2)
        + (11.851851851851851 * np.power(n, 2) * S2) / np.power(1.0 + n, 2)
        + 3.950617283950617 * nf * S2
        - 63.20987654320987 * S1 * S2
        + 18.962962962962962 * np.power(S2, 2)
        + (18.567901234567902 * S3) / (1.0 + n)
        - (5.925925925925926 * S3) / (n * (1.0 + n))
        + (18.567901234567902 * n * S3) / (1.0 + n)
        - 2.3703703703703702 * nf * S3
        + 11.851851851851851 * S1 * S3
        + 47.407407407407405 * S31
        - 30.814814814814813 * S4
        - (12.641975308641975 * Sm2) / np.power(1.0 + n, 2)
        + (2.3703703703703702 * Sm2) / (np.power(n, 2) * np.power(1.0 + n, 2))
        - (7.901234567901234 * Sm2) / (n * np.power(1.0 + n, 2))
        + 15.802469135802468 * S1 * Sm2
        - 4.7407407407407405 * S2 * Sm2
        - (15.802469135802468 * Sm21) / (1.0 + n)
        + (4.7407407407407405 * Sm21) / (n * (1.0 + n))
        - (15.802469135802468 * n * Sm21) / (1.0 + n)
        - 9.481481481481481 * S1 * Sm21
        + 18.962962962962962 * Sm211
        - 4.7407407407407405 * Sm22
        + (7.901234567901234 * Sm3) / (1.0 + n)
        + (2.3703703703703702 * Sm3) / (n * (1.0 + n))
        + (7.901234567901234 * n * Sm3) / (1.0 + n)
        - 4.7407407407407405 * S1 * Sm3
        - 4.7407407407407405 * Sm4
    )
    return a_qqNS_l0 + a_qqNS_l1 * L + a_qqNS_l2 * L**2 + a_qqNS_l3 * L**3
