"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CrossRegionParameter = exports.OnEvent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const ssm = require("aws-cdk-lib/aws-ssm");
const cr = require("aws-cdk-lib/custom-resources");
const change_case_1 = require("change-case");
const constructs_1 = require("constructs");
const ts_deepmerge_1 = require("ts-deepmerge");
const add_1 = require("./errors/add");
var OnEvent;
(function (OnEvent) {
    OnEvent["ON_CREATE"] = "onCreate";
    OnEvent["ON_UPDATE"] = "onUpdate";
    OnEvent["ON_DELETE"] = "onDelete";
})(OnEvent || (exports.OnEvent = OnEvent = {}));
/** Cross-Region SSM Parameter. */
class CrossRegionParameter extends constructs_1.Construct {
    /**
     * Define a new Cross-Region SSM Parameter.
     *
     * @example
     * new CrossRegionParameter(this, 'SayHiToSweden', {
     *   region: 'eu-north-1',
     *   name: '/parameter/path/message',
     *   description: 'Some message for the Swedes',
     *   value: 'Hej då!',
     * });
     */
    constructor(scope, name, props) {
        super(scope, name);
        this.uniqueTagKey = '@alma-cdk/cross-region-parameter:fromConstruct';
        const withDefaults = (0, ts_deepmerge_1.merge)(props, {
            tags: [{ key: this.uniqueTagKey, value: this.node.path }],
        });
        this.uniqueTagValue = this.node.path;
        this.validateRegion(withDefaults.region);
        const st = this.definePolicy(withDefaults);
        const policy = new iam.Policy(this, `${(0, change_case_1.pascalCase)(name)}CrPolicy`, { statements: [st] });
        const role = new iam.Role(this, 'ParameterCrRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
        });
        role.attachInlinePolicy(policy);
        const customResource = new cr.AwsCustomResource(this, 'AwsCustomResource', {
            onCreate: this.defineCreateUpdateSdkCall(OnEvent.ON_CREATE, withDefaults),
            onUpdate: this.defineCreateUpdateSdkCall(OnEvent.ON_UPDATE, withDefaults),
            onDelete: this.defineDeleteSdkCall(withDefaults),
            policy: cr.AwsCustomResourcePolicy.fromStatements([st]),
            role,
        });
        customResource.node.addDependency(role);
    }
    definePhysicalResourceId(props) {
        const { region, name } = props;
        return cr.PhysicalResourceId.of(`CrossRegionParameter${(0, change_case_1.pascalCase)(region)}${(0, change_case_1.pascalCase)(name)}`);
    }
    defineCreateUpdateSdkCall(eventType, props) {
        const { region, name, description, value, allowedPattern, keyId, parameterTier: tier = ssm.ParameterTier.STANDARD, parameterType: type = ssm.ParameterType.STRING, tags, policies, } = props;
        const parameters = {
            Name: name, /* required */
            Value: value, /* required */
            AllowedPattern: allowedPattern,
            Description: description,
            KeyId: keyId,
            Overwrite: eventType !== OnEvent.ON_CREATE,
            Policies: policies,
            Tags: this.tagPropsToTagParams(tags),
            Tier: tier,
            DataType: type,
        };
        return {
            physicalResourceId: this.definePhysicalResourceId(props),
            region,
            service: 'SSM',
            action: 'putParameter',
            parameters,
        };
    }
    /** Ensure Parameter target region is not the same as the current "source" region. */
    validateRegion(region) {
        const currentRegion = aws_cdk_lib_1.Stack.of(this).region;
        if (currentRegion === region) {
            (0, add_1.addError)(this, `Parameter target region ${region} can not be the same as source region ${currentRegion}`);
        }
    }
    /** Convert CDK/JSII compatible TagPropList to SDK compatible TagList. */
    tagPropsToTagParams(tags) {
        return tags?.map(t => ({
            Key: t.key,
            Value: t.value,
        }));
    }
    defineDeleteSdkCall(props) {
        const { region, name } = props;
        return {
            physicalResourceId: this.definePhysicalResourceId(props),
            region,
            service: 'ssm',
            action: 'DeleteParameter',
            parameters: {
                Name: name,
            },
        };
    }
    definePolicy(props) {
        const { region, name } = props;
        // Depending if path parameter or simple parameter we may or may not need to set a slash separator to resource ARN
        const separator = name.indexOf('/') === 0 ? '' : '/';
        return new iam.PolicyStatement({
            actions: ['ssm:PutParameter', 'ssm:DeleteParameter'],
            resources: [`arn:aws:ssm:${region}:${aws_cdk_lib_1.Stack.of(this).account}:parameter${separator}*`],
            effect: iam.Effect.ALLOW,
            conditions: {
                StringEquals: {
                    [`aws:ResourceTag/${this.uniqueTagKey}`]: this.uniqueTagValue,
                },
            },
        });
    }
}
exports.CrossRegionParameter = CrossRegionParameter;
_a = JSII_RTTI_SYMBOL_1;
CrossRegionParameter[_a] = { fqn: "@alma-cdk/cross-region-parameter.CrossRegionParameter", version: "1.0.0-beta.6" };
//# sourceMappingURL=data:application/json;base64,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