"""
Abstract base class for AI agents using LangChain.
This module provides a foundation for creating and managing AI agents
that interact with language models and retrieve information.
"""
from time import sleep
from abc import ABC, abstractmethod
from typing import List, Tuple, Callable, Iterable
import threading

import numpy as np
from langchain_core.language_models.llms import BaseLLM
from langchain_core.vectorstores.base import VectorStore


class BaseAgent(ABC):
    """
    Abstract base class for an AI agent that interacts with a language model
    and retrieves information to respond to queries.
    """

    base_llm: BaseLLM
    """
    The base language model used by the agent. This model is responsible for
    generating responses and performing language processing tasks.
    """

    vector_store: VectorStore
    """
    The vector store used by the agent for retrieving information.
    """

    q_and_a: List[Tuple[str, str | Callable]]
    """
    A list of tuples, where each tuple contains:
        - A question (str).
        - The expected response, which can either be:
            - A string (str), or
            - A callable function (Callable) that returns the expected
              response as a string.
    Used for evaluating the agent's performance.
    """

    def __init__(self):
        self._is_running = False
        self.nap_time = 30
        self.received_response = None
        self._thread = None

    @property
    @abstractmethod
    def version(self) -> str:
        """
        Abstract property to return the version of the agent.

        Returns:
            str: The version of the agent.
        """
        return ""

    @abstractmethod
    def listen(self, context: str) -> bool:
        """
        Other agents can call this function to pass contextual information.

        Args:
            context (str): The contextual information in natural language.

        Returns:
            bool: Return True if the knowledge has been accepted
        """
        return True

    @abstractmethod
    def _retrieve(self, q: str) -> List[Tuple[str, dict]]:
        """
        Retrieve relevant information for a given query.

        Args:
            q (str): The query string for which relevant data needs to be
                     retrieved.

        Returns:
            List[Tuple[str, dict]]: A list of tuples where each tuple contains
            a relevant piece of information (e.g., a document or text snippet)
            as a string, and its associated metadata as a dictionary.
        """
        return []

    @abstractmethod
    def _prompt(self, q: str) -> str:
        """
        Generate a formatted prompt based on the given query.

        Args:
            q (str): The query string for which the prompt is generated.

        Returns:
            str: The formatted prompt that will be used for generating a
            response.
        """
        return ""

    @abstractmethod
    def respond(self, q: str) -> Iterable[str]:
        """
        Generate a response for the given query by retrieving information and
        interacting with the language model.

        Args:
            q (str): The query string for which the response is generated.

        Yields:
            Iterable[str]: An iterable of response strings generated by the
            agent.
        """
        while False:
            yield ""

    def ask(self, agent: "BaseAgent", q: str) -> Iterable[str]:
        """
        Asynchronously send a query to another agent and process its response.

        Args:
            agent (Agent): Another agent to which the query is sent. q (str):
            The query string to be sent.

        Yields:
            Iterable[str]: An iterable of response strings generated by the
            other agent.
        """
        self.received_response = ""
        for chunk in agent.respond(q):
            self.received_response += chunk
            yield chunk

    def receive(self, generator: Iterable) -> str:
        """
        Concatenates the output of an iterable generator into a single string.

        Args:
            generator (Iterable): An iterable object (e.g., a generator) that
            yields chunks of text or other string-like elements.

        Returns:
            str: The concatenated string formed by joining all elements from the
            generator.
        """
        return ''.join(list(generator))

    def tell(self, agent: "BaseAgent", context: str) -> bool:
        """
        Send information to another agent.

        Args:
            agent (Agent): The recipient agent to which the information is sent.
            context (str): The information or context to be shared, framed as a
            query.

        """
        return agent.listen(context)

    @abstractmethod
    def run(self) -> None:
        """
        Run the agent. This method should contain the logic that needs to be
        executed in the loop.
        """

    def _run(self) -> None:
        """
        Internal method to run the agent in a loop.
        """
        while self._is_running:
            self.run()
            if self.nap_time is not None:
                sleep(self.nap_time)
            else:
                sleep(30)

    def start(self) -> None:
        """
        Start the agent by creating and starting the thread.
        """
        if not self._is_running:
            self._is_running = True
            self._thread = threading.Thread(target=self._run, daemon=True)
            self._thread.start()

    def stop(self) -> None:
        """
        Stop the agent and wait for the thread to finish.
        """
        if self._is_running:
            self._is_running = False
            if self._thread is not None:
                self._thread.join()

    def evaluate(self) -> Tuple[float, np.ndarray]:
        """
        Evaluate the agent's performance by comparing its responses to a list of
        predefined question-answer pairs (`q_and_a`).

        This method iterates through the list of questions and their
        corresponding expected answers. For each question, the agent generates a
        response using the `respond` method. The response is compared to the
        expected answer (either a string or the output of a callable function)
        to determine correctness.

        Returns:
            Tuple[float, np.ndarray]: A tuple containing:
                - The accuracy as a float (correct answers / total questions),
                  formatted to three decimal places.
                - A NumPy array (`np.ndarray`) where each element is 1 if the
                  response matched the expected answer, or 0 otherwise.

        Example:
            If `q_and_a` contains 10 questions and the agent answers 7
            correctly, and their correctness is stored in a NumPy array, this
            method will return `(0.700, array([1, 1, 0, ...]))`.
        """
        matches = np.zeros(len(self.q_and_a))
        i = 0
        for q, expected in self.q_and_a:
            if isinstance(expected, str):
                if ''.join(list(self.respond(q))).strip() == expected.strip():
                    matches[i] = 1
            else:
                if ''.join(list(self.respond(q))).strip() == expected.strip():
                    matches[i] = 1
            i += 1

        return f'{matches.sum() / len(matches):.03f}', matches
