"""PyWebDAV Server - WebDAV server with pluggable storage backends.

This package provides a WebDAV server that supports multiple storage backends:
- Local filesystem (default)
- Drime Cloud (optional, requires pydrime)

Example usage:
    # Using local filesystem backend
    from pywebdavserver.providers.local import LocalStorageProvider
    from pywebdavserver.server import run_webdav_server

    provider = LocalStorageProvider("/tmp/webdav", readonly=False)
    run_webdav_server(provider, host="0.0.0.0", port=8080)

    # Using Drime Cloud backend (requires pydrime)
    from pydrime.api import DrimeClient
    from pywebdavserver.providers.drime import DrimeDAVProvider
    from pywebdavserver.server import run_webdav_server

    client = DrimeClient(email="user@example.com", password="password")
    provider = DrimeDAVProvider(client, workspace_id=0)
    run_webdav_server(provider, host="0.0.0.0", port=8080)
"""

from typing import Any

from .constants import (
    DEFAULT_CACHE_TTL,
    DEFAULT_HOST,
    DEFAULT_MAX_FILE_SIZE,
    DEFAULT_PATH,
    DEFAULT_PORT,
)

# Version will be auto-generated by setuptools_scm
try:
    from ._version import __version__, __version_tuple__
except ImportError:
    __version__ = "unknown"
    __version_tuple__ = (0, 0, "unknown", "unknown")

__all__ = [
    "__version__",
    "__version_tuple__",
    "StorageProvider",
    "LocalStorageProvider",
    "DrimeStorageProvider",
    "create_webdav_app",
    "run_webdav_server",
    "DEFAULT_HOST",
    "DEFAULT_PORT",
    "DEFAULT_PATH",
    "DEFAULT_CACHE_TTL",
    "DEFAULT_MAX_FILE_SIZE",
]


def __getattr__(name: str) -> Any:
    """Lazy import to avoid loading heavy dependencies."""
    if name == "StorageProvider":
        from .provider import StorageProvider

        return StorageProvider
    elif name == "LocalStorageProvider":
        from .providers.local import LocalStorageProvider

        return LocalStorageProvider
    elif name == "DrimeStorageProvider":
        from .providers.drime import DrimeDAVProvider

        return DrimeDAVProvider
    elif name == "create_webdav_app":
        from .server import create_webdav_app

        return create_webdav_app
    elif name == "run_webdav_server":
        from .server import run_webdav_server

        return run_webdav_server
    raise AttributeError(f"module {__name__!r} has no attribute {name!r}")
