"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.fingerprint = void 0;
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const ignore_1 = require("./ignore");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const exclude = options.exclude || [];
    if (exclude.length) {
        _hashField(hash, 'options.exclude', JSON.stringify(exclude));
    }
    const ignoreMode = options.ignoreMode || options_1.IgnoreMode.GLOB;
    if (ignoreMode != options_1.IgnoreMode.GLOB) {
        _hashField(hash, 'options.ignoreMode', ignoreMode);
    }
    const ignoreStrategy = ignore_1.IgnoreStrategy.fromCopyOptions(options, fileOrDirectory);
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (!isRootDir && ignoreStrategy.ignores(symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // eslint-disable-next-line no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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