import { CopyOptions } from './options';
/**
 * Represents file path ignoring behavior.
 *
 * @stability stable
 */
export declare abstract class IgnoreStrategy {
    /**
     * Ignores file paths based on simple glob patterns.
     *
     * @param absoluteRootPath the absolute path to the root directory of the paths to be considered.
     * @returns `GlobIgnorePattern` associated with the given patterns.
     * @stability stable
     */
    static glob(absoluteRootPath: string, patterns: string[]): GlobIgnoreStrategy;
    /**
     * Ignores file paths based on the [`.gitignore specification`](https://git-scm.com/docs/gitignore).
     *
     * @param absoluteRootPath the absolute path to the root directory of the paths to be considered.
     * @returns `GitIgnorePattern` associated with the given patterns.
     * @stability stable
     */
    static git(absoluteRootPath: string, patterns: string[]): GitIgnoreStrategy;
    /**
     * Ignores file paths based on the [`.dockerignore specification`](https://docs.docker.com/engine/reference/builder/#dockerignore-file).
     *
     * @param absoluteRootPath the absolute path to the root directory of the paths to be considered.
     * @returns `DockerIgnorePattern` associated with the given patterns.
     * @stability stable
     */
    static docker(absoluteRootPath: string, patterns: string[]): DockerIgnoreStrategy;
    /**
     * Creates an IgnoreStrategy based on the `ignoreMode` and `exclude` in a `CopyOptions`.
     *
     * @param options the `CopyOptions` to create the `IgnoreStrategy` from.
     * @param absoluteRootPath the absolute path to the root directory of the paths to be considered.
     * @returns `IgnoreStrategy` based on the `CopyOptions`
     * @stability stable
     */
    static fromCopyOptions(options: CopyOptions, absoluteRootPath: string): IgnoreStrategy;
    /**
     * Adds another pattern.
     *
     * @stability stable
     * @params pattern the pattern to add
     */
    abstract add(pattern: string): void;
    /**
     * Determines whether a given file path should be ignored or not.
     *
     * @param absoluteFilePath absolute file path to be assessed against the pattern.
     * @returns `true` if the file should be ignored
     * @stability stable
     */
    abstract ignores(absoluteFilePath: string): boolean;
}
/**
 * Ignores file paths based on simple glob patterns.
 *
 * @stability stable
 */
export declare class GlobIgnoreStrategy extends IgnoreStrategy {
    private readonly absoluteRootPath;
    private readonly patterns;
    /**
     * @stability stable
     */
    constructor(absoluteRootPath: string, patterns: string[]);
    /**
     * Adds another pattern.
     *
     * @stability stable
     * @params pattern the pattern to add
     */
    add(pattern: string): void;
    /**
     * Determines whether a given file path should be ignored or not.
     *
     * @param absoluteFilePath absolute file path to be assessed against the pattern.
     * @returns `true` if the file should be ignored
     * @stability stable
     */
    ignores(absoluteFilePath: string): boolean;
}
/**
 * Ignores file paths based on the [`.gitignore specification`](https://git-scm.com/docs/gitignore).
 *
 * @stability stable
 */
export declare class GitIgnoreStrategy extends IgnoreStrategy {
    private readonly absoluteRootPath;
    private readonly ignore;
    /**
     * @stability stable
     */
    constructor(absoluteRootPath: string, patterns: string[]);
    /**
     * Adds another pattern.
     *
     * @stability stable
     * @params pattern the pattern to add
     */
    add(pattern: string): void;
    /**
     * Determines whether a given file path should be ignored or not.
     *
     * @param absoluteFilePath absolute file path to be assessed against the pattern.
     * @returns `true` if the file should be ignored
     * @stability stable
     */
    ignores(absoluteFilePath: string): boolean;
}
/**
 * Ignores file paths based on the [`.dockerignore specification`](https://docs.docker.com/engine/reference/builder/#dockerignore-file).
 *
 * @stability stable
 */
export declare class DockerIgnoreStrategy extends IgnoreStrategy {
    private readonly absoluteRootPath;
    private readonly ignore;
    /**
     * @stability stable
     */
    constructor(absoluteRootPath: string, patterns: string[]);
    /**
     * Adds another pattern.
     *
     * @stability stable
     * @params pattern the pattern to add
     */
    add(pattern: string): void;
    /**
     * Determines whether a given file path should be ignored or not.
     *
     * @param absoluteFilePath absolute file path to be assessed against the pattern.
     * @returns `true` if the file should be ignored
     * @stability stable
     */
    ignores(absoluteFilePath: string): boolean;
}
