"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RememberingTokenResolver = exports.findTokens = exports.resolve = void 0;
const resolvable_1 = require("../resolvable");
const encoding_1 = require("./encoding");
const token_map_1 = require("./token-map");
// This file should not be exported to consumers, resolving should happen through Construct.resolve()
const tokenMap = token_map_1.TokenMap.instance();
/**
 * Resolves an object by evaluating all tokens and removing any undefined or empty objects or arrays.
 * Values can only be primitives, arrays or tokens. Other objects (i.e. with methods) will be rejected.
 *
 * @param obj The object to resolve.
 * @param prefix Prefix key path components for diagnostics.
 */
function resolve(obj, options) {
    const prefix = options.prefix || [];
    const pathName = '/' + prefix.join('/');
    /**
     * Make a new resolution context
     */
    function makeContext(appendPath) {
        const newPrefix = appendPath !== undefined ? prefix.concat([appendPath]) : options.prefix;
        let postProcessor;
        const context = {
            preparing: options.preparing,
            scope: options.scope,
            registerPostProcessor(pp) { postProcessor = pp; },
            resolve(x) { return resolve(x, { ...options, prefix: newPrefix }); },
        };
        return [context, { postProcess(x) { return postProcessor ? postProcessor.postProcess(x, context) : x; } }];
    }
    // protect against cyclic references by limiting depth.
    if (prefix.length > 200) {
        throw new Error('Unable to resolve object tree with circular reference. Path: ' + pathName);
    }
    //
    // undefined
    //
    if (typeof (obj) === 'undefined') {
        return undefined;
    }
    //
    // null
    //
    if (obj === null) {
        return null;
    }
    //
    // functions - not supported (only tokens are supported)
    //
    if (typeof (obj) === 'function') {
        throw new Error(`Trying to resolve a non-data object. Only token are supported for lazy evaluation. Path: ${pathName}. Object: ${obj}`);
    }
    //
    // string - potentially replace all stringified Tokens
    //
    if (typeof (obj) === 'string') {
        // If this is a "list element" Token, it should never occur by itself in string context
        if (encoding_1.TokenString.forListToken(obj).test()) {
            throw new Error('Found an encoded list token string in a scalar string context. Use \'Fn.select(0, list)\' (not \'list[0]\') to extract elements from token lists.');
        }
        // Otherwise look for a stringified Token in this object
        const str = encoding_1.TokenString.forString(obj);
        if (str.test()) {
            const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
            return options.resolver.resolveString(fragments, makeContext()[0]);
        }
        return obj;
    }
    //
    // number - potentially decode Tokenized number
    //
    if (typeof (obj) === 'number') {
        return resolveNumberToken(obj, makeContext()[0]);
    }
    //
    // primitives - as-is
    //
    if (typeof (obj) !== 'object' || obj instanceof Date) {
        return obj;
    }
    //
    // arrays - resolve all values, remove undefined and remove empty arrays
    //
    if (Array.isArray(obj)) {
        if (encoding_1.containsListTokenElement(obj)) {
            return options.resolver.resolveList(obj, makeContext()[0]);
        }
        const arr = obj
            .map((x, i) => makeContext(`${i}`)[0].resolve(x))
            .filter(x => typeof (x) !== 'undefined');
        return arr;
    }
    //
    // tokens - invoke 'resolve' and continue to resolve recursively
    //
    if (encoding_1.unresolved(obj)) {
        const [context, postProcessor] = makeContext();
        return options.resolver.resolveToken(obj, context, postProcessor);
    }
    //
    // objects - deep-resolve all values
    //
    // Must not be a Construct at this point, otherwise you probably made a typo
    // mistake somewhere and resolve will get into an infinite loop recursing into
    // child.parent <---> parent.children
    if (isConstruct(obj)) {
        throw new Error('Trying to resolve() a Construct at ' + pathName);
    }
    const result = {};
    for (const key of Object.keys(obj)) {
        const resolvedKey = makeContext()[0].resolve(key);
        if (typeof (resolvedKey) !== 'string') {
            // eslint-disable-next-line max-len
            throw new Error(`"${key}" is used as the key in a map so must resolve to a string, but it resolves to: ${JSON.stringify(resolvedKey)}. Consider using "CfnJson" to delay resolution to deployment-time`);
        }
        const value = makeContext(key)[0].resolve(obj[key]);
        // skip undefined
        if (typeof (value) === 'undefined') {
            continue;
        }
        result[resolvedKey] = value;
    }
    return result;
}
exports.resolve = resolve;
/**
 * Find all Tokens that are used in the given structure
 */
function findTokens(scope, fn) {
    const resolver = new RememberingTokenResolver(new resolvable_1.StringConcat());
    resolve(fn(), { scope, prefix: [], resolver, preparing: true });
    return resolver.tokens;
}
exports.findTokens = findTokens;
/**
 * Remember all Tokens encountered while resolving
 */
class RememberingTokenResolver extends resolvable_1.DefaultTokenResolver {
    constructor() {
        super(...arguments);
        this.tokensSeen = new Set();
    }
    resolveToken(t, context, postProcessor) {
        this.tokensSeen.add(t);
        return super.resolveToken(t, context, postProcessor);
    }
    resolveString(s, context) {
        const ret = super.resolveString(s, context);
        return ret;
    }
    get tokens() {
        return Array.from(this.tokensSeen);
    }
}
exports.RememberingTokenResolver = RememberingTokenResolver;
/**
 * Determine whether an object is a Construct
 *
 * Not in 'construct.ts' because that would lead to a dependency cycle via 'uniqueid.ts',
 * and this is a best-effort protection against a common programming mistake anyway.
 */
function isConstruct(x) {
    return x._children !== undefined && x._metadata !== undefined;
}
function resolveNumberToken(x, context) {
    const token = token_map_1.TokenMap.instance().lookupNumberToken(x);
    if (token === undefined) {
        return x;
    }
    return context.resolve(token);
}
//# sourceMappingURL=data:application/json;base64,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