"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Resource = void 0;
// v2 - leave this as a separate section so it reduces merge conflicts when compat is removed
// eslint-disable-next-line import/order
const construct_compat_1 = require("./construct-compat");
const lazy_1 = require("./lazy");
const physical_name_generator_1 = require("./private/physical-name-generator");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * A construct which represents an AWS resource.
 *
 * @stability stable
 */
class Resource extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id);
        this.stack = stack_1.Stack.of(this);
        this.env = {
            account: (_a = props.account) !== null && _a !== void 0 ? _a : this.stack.account,
            region: (_b = props.region) !== null && _b !== void 0 ? _b : this.stack.region,
        };
        let physicalName = props.physicalName;
        if (props.physicalName && physical_name_generator_1.isGeneratedWhenNeededMarker(props.physicalName)) {
            // auto-generate only if cross-env is required
            this._physicalName = undefined;
            this._allowCrossEnvironment = true;
            physicalName = lazy_1.Lazy.string({ produce: () => this._physicalName });
        }
        else if (props.physicalName && !token_1.Token.isUnresolved(props.physicalName)) {
            // concrete value specified by the user
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = true;
        }
        else {
            // either undefined (deploy-time) or has tokens, which means we can't use for cross-env
            this._physicalName = props.physicalName;
            this._allowCrossEnvironment = false;
        }
        if (physicalName === undefined) {
            physicalName = token_1.Token.asString(undefined);
        }
        this.physicalName = physicalName;
    }
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment() {
        if (!this._allowCrossEnvironment) {
            // error out - a deploy-time name cannot be used across environments
            throw new Error(`Cannot use resource '${this.node.path}' in a cross-environment fashion, ` +
                "the resource's physical name must be explicit set or use `PhysicalName.GENERATE_IF_NEEDED`");
        }
        if (!this._physicalName) {
            this._physicalName = this.generatePhysicalName();
        }
    }
    /**
     * @stability stable
     */
    generatePhysicalName() {
        return physical_name_generator_1.generatePhysicalName(this);
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * @experimental
     */
    getResourceNameAttribute(nameAttr) {
        return lazy_1.Lazy.uncachedString({
            produce: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.physicalName;
                }
                else {
                    return nameAttr;
                }
            },
        });
    }
    /**
     * (experimental) Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * @param arnComponents The format of the ARN of this resource.
     * @experimental
     */
    getResourceArnAttribute(arnAttr, arnComponents) {
        return token_1.Token.asString({
            resolve: (context) => {
                const consumingStack = stack_1.Stack.of(context.scope);
                if (this.stack.environment !== consumingStack.environment) {
                    this._enableCrossEnvironment();
                    return this.stack.formatArn(arnComponents);
                }
                else {
                    return arnAttr;
                }
            },
        });
    }
}
exports.Resource = Resource;
//# sourceMappingURL=data:application/json;base64,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