"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DefaultStackSynthesizer = exports.BOOTSTRAP_QUALIFIER_CONTEXT = void 0;
const fs = require("fs");
const path = require("path");
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const assets_1 = require("../assets");
const cfn_fn_1 = require("../cfn-fn");
const cfn_parameter_1 = require("../cfn-parameter");
const cfn_rule_1 = require("../cfn-rule");
const token_1 = require("../token");
const _shared_1 = require("./_shared");
const stack_synthesizer_1 = require("./stack-synthesizer");
exports.BOOTSTRAP_QUALIFIER_CONTEXT = '@aws-cdk/core:bootstrapQualifier';
/* eslint-disable max-len */
/**
 * The minimum bootstrap stack version required by this app.
 */
const MIN_BOOTSTRAP_STACK_VERSION = 4;
/**
 * Uses conventionally named roles and reify asset storage locations.
 *
 * This synthesizer is the only StackSynthesizer that generates
 * an asset manifest, and is required to deploy CDK applications using the
 * `@aws-cdk/app-delivery` CI/CD library.
 *
 * Requires the environment to have been bootstrapped with Bootstrap Stack V2.
 *
 * @stability stable
 */
class DefaultStackSynthesizer extends stack_synthesizer_1.StackSynthesizer {
    /**
     * @stability stable
     */
    constructor(props = {}) {
        super();
        this.props = props;
        this.files = {};
        this.dockerImages = {};
    }
    /**
     * Bind to the stack this environment is going to be used on.
     *
     * Must be called before any of the other methods are called.
     *
     * @stability stable
     */
    bind(stack) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j;
        if (this._stack !== undefined) {
            throw new Error('A StackSynthesizer can only be used for one Stack: create a new instance to use with a different Stack');
        }
        this._stack = stack;
        const qualifier = (_b = (_a = this.props.qualifier) !== null && _a !== void 0 ? _a : stack.node.tryGetContext(exports.BOOTSTRAP_QUALIFIER_CONTEXT)) !== null && _b !== void 0 ? _b : DefaultStackSynthesizer.DEFAULT_QUALIFIER;
        this.qualifier = qualifier;
        // Function to replace placeholders in the input string as much as possible
        //
        // We replace:
        // - ${Qualifier}: always
        // - ${AWS::AccountId}, ${AWS::Region}: only if we have the actual values available
        // - ${AWS::Partition}: never, since we never have the actual partition value.
        const specialize = (s) => {
            s = replaceAll(s, '${Qualifier}', qualifier);
            return cxapi.EnvironmentPlaceholders.replace(s, {
                region: resolvedOr(stack.region, cxapi.EnvironmentPlaceholders.CURRENT_REGION),
                accountId: resolvedOr(stack.account, cxapi.EnvironmentPlaceholders.CURRENT_ACCOUNT),
                partition: cxapi.EnvironmentPlaceholders.CURRENT_PARTITION,
            });
        };
        /* eslint-disable max-len */
        this.bucketName = specialize((_c = this.props.fileAssetsBucketName) !== null && _c !== void 0 ? _c : DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME);
        this.repositoryName = specialize((_d = this.props.imageAssetsRepositoryName) !== null && _d !== void 0 ? _d : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME);
        this._deployRoleArn = specialize((_e = this.props.deployRoleArn) !== null && _e !== void 0 ? _e : DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN);
        this._cloudFormationExecutionRoleArn = specialize((_f = this.props.cloudFormationExecutionRole) !== null && _f !== void 0 ? _f : DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN);
        this.fileAssetPublishingRoleArn = specialize((_g = this.props.fileAssetPublishingRoleArn) !== null && _g !== void 0 ? _g : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN);
        this.imageAssetPublishingRoleArn = specialize((_h = this.props.imageAssetPublishingRoleArn) !== null && _h !== void 0 ? _h : DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN);
        this.bucketPrefix = specialize((_j = this.props.bucketPrefix) !== null && _j !== void 0 ? _j : DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX);
        /* eslint-enable max-len */
    }
    /**
     * Register a File Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addFileAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.bucketName);
        const objectKey = this.bucketPrefix + asset.sourceHash + (asset.packaging === assets_1.FileAssetPackaging.ZIP_DIRECTORY ? '.zip' : '');
        // Add to manifest
        this.files[asset.sourceHash] = {
            source: {
                path: asset.fileName,
                packaging: asset.packaging,
            },
            destinations: {
                [this.manifestEnvName]: {
                    bucketName: this.bucketName,
                    objectKey,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.fileAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.fileAssetPublishingExternalId,
                },
            },
        };
        const { region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        const httpUrl = cfnify(`https://s3.${region}.${urlSuffix}/${this.bucketName}/${objectKey}`);
        const s3ObjectUrl = cfnify(`s3://${this.bucketName}/${objectKey}`);
        // Return CFN expression
        return {
            bucketName: cfnify(this.bucketName),
            objectKey,
            httpUrl,
            s3ObjectUrl,
            s3Url: httpUrl,
        };
    }
    /**
     * Register a Docker Image Asset.
     *
     * Returns the parameters that can be used to refer to the asset inside the template.
     *
     * @stability stable
     */
    addDockerImageAsset(asset) {
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.repositoryName);
        const imageTag = asset.sourceHash;
        // Add to manifest
        this.dockerImages[asset.sourceHash] = {
            source: {
                directory: asset.directoryName,
                dockerBuildArgs: asset.dockerBuildArgs,
                dockerBuildTarget: asset.dockerBuildTarget,
                dockerFile: asset.dockerFile,
            },
            destinations: {
                [this.manifestEnvName]: {
                    repositoryName: this.repositoryName,
                    imageTag,
                    region: resolvedOr(this.stack.region, undefined),
                    assumeRoleArn: this.imageAssetPublishingRoleArn,
                    assumeRoleExternalId: this.props.imageAssetPublishingExternalId,
                },
            },
        };
        const { account, region, urlSuffix } = stackLocationOrInstrinsics(this.stack);
        // Return CFN expression
        return {
            repositoryName: cfnify(this.repositoryName),
            imageUri: cfnify(`${account}.dkr.ecr.${region}.${urlSuffix}/${this.repositoryName}:${imageTag}`),
        };
    }
    /**
     * Synthesize the associated stack to the session.
     *
     * @stability stable
     */
    synthesize(session) {
        var _a;
        _shared_1.assertBound(this.stack);
        _shared_1.assertBound(this.qualifier);
        // Must be done here -- if it's done in bind() (called in the Stack's constructor)
        // then it will become impossible to set context after that.
        //
        // If it's done AFTER _synthesizeTemplate(), then the template won't contain the
        // right constructs.
        if ((_a = this.props.generateBootstrapVersionRule) !== null && _a !== void 0 ? _a : true) {
            addBootstrapVersionRule(this.stack, MIN_BOOTSTRAP_STACK_VERSION, this.qualifier);
        }
        this.synthesizeStackTemplate(this.stack, session);
        // Add the stack's template to the artifact manifest
        const templateManifestUrl = this.addStackTemplateToAssetManifest(session);
        const artifactId = this.writeAssetManifest(session);
        this.emitStackArtifact(this.stack, session, {
            assumeRoleArn: this._deployRoleArn,
            cloudFormationExecutionRoleArn: this._cloudFormationExecutionRoleArn,
            stackTemplateAssetObjectUrl: templateManifestUrl,
            requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
            additionalDependencies: [artifactId],
        });
    }
    /**
     * Returns the ARN of the deploy Role.
     *
     * @stability stable
     */
    get deployRoleArn() {
        if (!this._deployRoleArn) {
            throw new Error('deployRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._deployRoleArn;
    }
    /**
     * Returns the ARN of the CFN execution Role.
     *
     * @stability stable
     */
    get cloudFormationExecutionRoleArn() {
        if (!this._cloudFormationExecutionRoleArn) {
            throw new Error('cloudFormationExecutionRoleArn getter can only be called after the synthesizer has been bound to a Stack');
        }
        return this._cloudFormationExecutionRoleArn;
    }
    /**
     * @stability stable
     */
    get stack() {
        return this._stack;
    }
    /**
     * Add the stack's template as one of the manifest assets
     *
     * This will make it get uploaded to S3 automatically by S3-assets. Return
     * the manifest URL.
     *
     * (We can't return the location returned from `addFileAsset`, as that
     * contains CloudFormation intrinsics which can't go into the manifest).
     */
    addStackTemplateToAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const templatePath = path.join(session.assembly.outdir, this.stack.templateFile);
        const template = fs.readFileSync(templatePath, { encoding: 'utf-8' });
        const sourceHash = _shared_1.contentHash(template);
        this.addFileAsset({
            fileName: this.stack.templateFile,
            packaging: assets_1.FileAssetPackaging.FILE,
            sourceHash,
        });
        // We should technically return an 'https://s3.REGION.amazonaws.com[.cn]/name/hash' URL here,
        // because that is what CloudFormation expects to see.
        //
        // However, there's no way for us to actually know the UrlSuffix a priori, so we can't construct it here.
        //
        // Instead, we'll have a protocol with the CLI that we put an 's3://.../...' URL here, and the CLI
        // is going to resolve it to the correct 'https://.../' URL before it gives it to CloudFormation.
        return `s3://${this.bucketName}/${sourceHash}`;
    }
    /**
     * Write an asset manifest to the Cloud Assembly, return the artifact IDs written
     */
    writeAssetManifest(session) {
        _shared_1.assertBound(this.stack);
        const artifactId = `${this.stack.artifactId}.assets`;
        const manifestFile = `${artifactId}.json`;
        const outPath = path.join(session.assembly.outdir, manifestFile);
        const manifest = {
            version: cxschema.Manifest.version(),
            files: this.files,
            dockerImages: this.dockerImages,
        };
        fs.writeFileSync(outPath, JSON.stringify(manifest, undefined, 2));
        session.assembly.addArtifact(artifactId, {
            type: cxschema.ArtifactType.ASSET_MANIFEST,
            properties: {
                file: manifestFile,
                requiresBootstrapStackVersion: MIN_BOOTSTRAP_STACK_VERSION,
            },
        });
        return artifactId;
    }
    get manifestEnvName() {
        _shared_1.assertBound(this.stack);
        return [
            resolvedOr(this.stack.account, 'current_account'),
            resolvedOr(this.stack.region, 'current_region'),
        ].join('-');
    }
}
exports.DefaultStackSynthesizer = DefaultStackSynthesizer;
/**
 * Default ARN qualifier.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_QUALIFIER = 'hnb659fds';
/**
 * Default CloudFormation role ARN.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_CLOUDFORMATION_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-cfn-exec-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default deploy role ARN.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_DEPLOY_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-deploy-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for file (S3) assets.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-file-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default asset publishing role ARN for image (ECR) assets.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSET_PUBLISHING_ROLE_ARN = 'arn:${AWS::Partition}:iam::${AWS::AccountId}:role/cdk-${Qualifier}-image-publishing-role-${AWS::AccountId}-${AWS::Region}';
/**
 * Default image assets repository name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_IMAGE_ASSETS_REPOSITORY_NAME = 'cdk-${Qualifier}-container-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Default file assets bucket name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSETS_BUCKET_NAME = 'cdk-${Qualifier}-assets-${AWS::AccountId}-${AWS::Region}';
/**
 * Name of the CloudFormation Export with the asset key name.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_KEY_ARN_EXPORT_NAME = 'CdkBootstrap-${Qualifier}-FileAssetKeyArn';
/**
 * Default file asset prefix.
 *
 * @stability stable
 */
DefaultStackSynthesizer.DEFAULT_FILE_ASSET_PREFIX = '';
/**
 * Return the given value if resolved or fall back to a default
 */
function resolvedOr(x, def) {
    return token_1.Token.isUnresolved(x) ? def : x;
}
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
/**
 * If the string still contains placeholders, wrap it in a Fn::Sub so they will be substituted at CFN deployment time
 *
 * (This happens to work because the placeholders we picked map directly onto CFN
 * placeholders. If they didn't we'd have to do a transformation here).
 */
function cfnify(s) {
    return s.indexOf('${') > -1 ? cfn_fn_1.Fn.sub(s) : s;
}
/**
 * Return the stack locations if they're concrete, or the original CFN intrisics otherwise
 *
 * We need to return these instead of the tokenized versions of the strings,
 * since we must accept those same ${AWS::AccountId}/${AWS::Region} placeholders
 * in bucket names and role names (in order to allow environment-agnostic stacks).
 *
 * We'll wrap a single {Fn::Sub} around the final string in order to replace everything,
 * but we can't have the token system render part of the string to {Fn::Join} because
 * the CFN specification doesn't allow the {Fn::Sub} template string to be an arbitrary
 * expression--it must be a string literal.
 */
function stackLocationOrInstrinsics(stack) {
    return {
        account: resolvedOr(stack.account, '${AWS::AccountId}'),
        region: resolvedOr(stack.region, '${AWS::Region}'),
        urlSuffix: resolvedOr(stack.urlSuffix, '${AWS::URLSuffix}'),
    };
}
/**
 * Add a CfnRule to the Stack which checks the current version of the bootstrap stack this template is targeting
 *
 * The CLI normally checks this, but in a pipeline the CLI is not involved
 * so we encode this rule into the template in a way that CloudFormation will check it.
 */
function addBootstrapVersionRule(stack, requiredVersion, qualifier) {
    // Because of https://github.com/aws/aws-cdk/blob/master/packages/@aws-cdk/assert/lib/synth-utils.ts#L74
    // synthesize() may be called more than once on a stack in unit tests, and the below would break
    // if we execute it a second time. Guard against the constructs already existing.
    if (stack.node.tryFindChild('BootstrapVersion')) {
        return;
    }
    const param = new cfn_parameter_1.CfnParameter(stack, 'BootstrapVersion', {
        type: 'AWS::SSM::Parameter::Value<String>',
        description: 'Version of the CDK Bootstrap resources in this environment, automatically retrieved from SSM Parameter Store.',
        default: `/cdk-bootstrap/${qualifier}/version`,
    });
    // There is no >= check in CloudFormation, so we have to check the number
    // is NOT in [1, 2, 3, ... <required> - 1]
    const oldVersions = range(1, requiredVersion).map(n => `${n}`);
    new cfn_rule_1.CfnRule(stack, 'CheckBootstrapVersion', {
        assertions: [
            {
                assert: cfn_fn_1.Fn.conditionNot(cfn_fn_1.Fn.conditionContains(oldVersions, param.valueAsString)),
                assertDescription: `CDK bootstrap stack version ${requiredVersion} required. Please run 'cdk bootstrap' with a recent version of the CDK CLI.`,
            },
        ],
    });
}
function range(startIncl, endExcl) {
    const ret = new Array();
    for (let i = startIncl; i < endExcl; i++) {
        ret.push(i);
    }
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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