"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const cxapi = require("@aws-cdk/cx-api");
const constructs_1 = require("constructs");
const synthesis_1 = require("./private/synthesis");
// v2 - keep this import as a separate section to reduce merge conflict when forward merging with the v2 branch.
// eslint-disable-next-line
const construct_compat_1 = require("./construct-compat");
const STAGE_SYMBOL = Symbol.for('@aws-cdk/core.Stage');
/**
 * An abstract application modeling unit consisting of Stacks that should be deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 *
 * @stability stable
 */
class Stage extends construct_compat_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        Object.defineProperty(this, STAGE_SYMBOL, { value: true });
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * (experimental) Return the stage this construct is contained with, if available.
     *
     * If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return constructs_1.Node.of(construct).scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * (experimental) Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && typeof (x) === 'object' && STAGE_SYMBOL in x;
    }
    /**
     * The cloud assembly output directory.
     *
     * @stability stable
     */
    get outdir() {
        return this._assemblyBuilder.outdir;
    }
    /**
     * The cloud assembly asset output directory.
     *
     * @stability stable
     */
    get assetOutdir() {
        return this._assemblyBuilder.assetOutdir;
    }
    /**
     * (experimental) Artifact ID of the assembly if it is a nested stage. The root stage (app) will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     *
     * @stability stable
     */
    synth(options = {}) {
        if (!this.assembly || options.force) {
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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