from typing import Dict, List

import git

from safa.config import SafaConfig
from safa.data.artifact_json import ArtifactJson
from safa.data.file_change import FileChange
from safa.api.safa_client import SafaClient
from safa.tools.utils.generate import generate_summary
from safa.tools.utils.git_helpers import get_file_content_before, get_staged_diffs, stage_files
from safa.utils.markdown import list_formatter
from safa.utils.menu import input_option
from safa.utils.printers import print_title


def run_committer(config: SafaConfig, client: SafaClient):
    """
    Reads staged changes and generates commit details (i.e. title, changes). Allows user to edit afterwards.
    :param config: The configuration of the tool.
    :return:
    """
    print_title("Committer Tool")
    project_data = get_safa_project(config, client)
    artifact_map = create_artifact_name_lookup(project_data["artifacts"])

    repo = git.Repo(config.repo_path)
    stage_files(repo)
    file2diff = get_staged_diffs(repo)
    if len(file2diff) == 0:
        print("No changes staged for commit.")
        return

    changes = create_file_changes(file2diff, artifact_map, repo)
    title, changes = generate_summary(changes, project_data["specification"])

    run_commit_menu(repo, title, changes)


def create_file_changes(file2diff, artifact_map: Dict[str, ArtifactJson], repo) -> List[FileChange]:
    """
    Augments file diffs with artifact summary and file before changes.
    :param file2diff: Map of file to its diffs for those to convert.
    :param artifact_map: Artifact name lookup table.
    :param repo: Git repository to use to get file state before change.
    :return: List of file changes.
    """
    changes: List[FileChange] = []
    for file, diff in file2diff.items():
        file_artifact = artifact_map.get(file, None)
        content_before = get_file_content_before(repo, file)
        changes.append(FileChange(
            file=file,
            diff=diff,
            content_before=content_before,
            summary=file_artifact["summary"] if file_artifact else None
        ))
    return changes


def run_commit_menu(repo: git.Repo, title: str, changes: List[str]) -> None:
    """
    Runs commit management menu.
    :param repo: Repository that commit is being applied to.
    :param title: The current title of the commit.
    :param changes: The current changes to the commit.
    :return: None
    """
    print_title("Commit Menu")
    menu_options = ["Edit Title", "Edit Change", "Remove Change", "Add Change", "Commit"]
    running = True
    while running:
        print_commit_message(title, changes, format_type="numbered")
        selected_option = input_option(menu_options)
        option_num = menu_options.index(selected_option)

        if option_num == 0:
            title = input("New Title:")
        elif option_num == 1:
            change_num = int(input("Change ID:"))
            changes[change_num - 1] = input("New Change:")
        elif option_num == 2:
            change_num = int(input("Change ID:"))
            changes.pop(change_num - 1)
        elif option_num == 3:
            changes.append(input("New Change:"))
        elif option_num == 4:
            repo.index.commit(to_commit_message(title, changes))
            running = False
        else:
            raise Exception("Invalid option")


def get_safa_project(config: SafaConfig, client: SafaClient):
    """
    Reads SAFA project.
    :param config: Configuration detailing account details and project.
    :return: The project data.
    """
    print("...retrieving safa project...")
    project_data = client.get_version(config.version_id)
    print("Project Name: ", project_data["name"])
    return project_data


def create_artifact_name_lookup(artifacts: List[Dict]) -> Dict:
    """
    Creates lookup map for artifact by names.
    :param artifacts: The artifacts to include in the map.
    :return: Artifact Map.
    """
    artifact_map = {}
    for artifact in artifacts:
        artifact_map[artifact["name"]] = artifact
    return artifact_map


def print_commit_message(title, changes, **kwargs) -> None:
    """
    Prints commit message to console.
    :param title: Title of the message.
    :param changes: The changes in the commit.
    :return: None
    """
    print(to_commit_message(title, changes, **kwargs))


def to_commit_message(title: str, changes: List[str], **kwargs) -> str:
    """
    Creates commit message with title and list of changes.
    :param title: Title of commit.
    :param changes: List of changes in commit.
    :return: Commit message.
    """
    return f"{title}\n\n{list_formatter(changes, **kwargs)}"
