"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.listValue({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg.role, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,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