"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_reference_1 = require("./private/cfn-reference");
const token_1 = require("./token");
const AWS_ACCOUNTID = 'AWS::AccountId';
const AWS_URLSUFFIX = 'AWS::URLSuffix';
const AWS_NOTIFICATIONARNS = 'AWS::NotificationARNs';
const AWS_PARTITION = 'AWS::Partition';
const AWS_REGION = 'AWS::Region';
const AWS_STACKID = 'AWS::StackId';
const AWS_STACKNAME = 'AWS::StackName';
const AWS_NOVALUE = 'AWS::NoValue';
/**
 * Accessor for pseudo parameters
 *
 * Since pseudo parameters need to be anchored to a stack somewhere in the
 * construct tree, this class takes an scope parameter; the pseudo parameter
 * values can be obtained as properties from an scoped object.
 */
class Aws {
    constructor() {
    }
}
exports.Aws = Aws;
Aws.ACCOUNT_ID = pseudoString(AWS_ACCOUNTID);
Aws.URL_SUFFIX = pseudoString(AWS_URLSUFFIX);
Aws.NOTIFICATION_ARNS = token_1.Token.asList({ Ref: AWS_NOTIFICATIONARNS }, { displayHint: AWS_NOTIFICATIONARNS });
Aws.PARTITION = pseudoString(AWS_PARTITION);
Aws.REGION = pseudoString(AWS_REGION);
Aws.STACK_ID = pseudoString(AWS_STACKID);
Aws.STACK_NAME = pseudoString(AWS_STACKNAME);
Aws.NO_VALUE = pseudoString(AWS_NOVALUE);
/**
 * Accessor for scoped pseudo parameters
 *
 * These pseudo parameters are anchored to a stack somewhere in the construct
 * tree, and their values will be exported automatically.
 */
class ScopedAws {
    constructor(scope) {
        this.scope = scope;
    }
    get accountId() {
        return this.asString(AWS_ACCOUNTID);
    }
    get urlSuffix() {
        return this.asString(AWS_URLSUFFIX);
    }
    get notificationArns() {
        return token_1.Token.asList(cfn_reference_1.CfnReference.forPseudo(AWS_NOTIFICATIONARNS, this.scope), {
            displayHint: AWS_NOTIFICATIONARNS
        });
    }
    get partition() {
        return this.asString(AWS_PARTITION);
    }
    get region() {
        return this.asString(AWS_REGION);
    }
    get stackId() {
        return this.asString(AWS_STACKID);
    }
    get stackName() {
        return this.asString(AWS_STACKNAME);
    }
    asString(name) {
        return token_1.Token.asString(cfn_reference_1.CfnReference.forPseudo(name, this.scope), { displayHint: name });
    }
}
exports.ScopedAws = ScopedAws;
function pseudoString(name) {
    return token_1.Token.asString({ Ref: name }, { displayHint: name });
}
//# sourceMappingURL=data:application/json;base64,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