"use strict";
const nodeunit = require("nodeunit");
const lib_1 = require("../lib");
function floatEqual(test, actual, expected) {
    test.ok(
    // Floats are subject to rounding errors up to Number.ESPILON
    actual >= expected - Number.EPSILON && actual <= expected + Number.EPSILON, `${actual} == ${expected}`);
}
module.exports = nodeunit.testCase({
    'negative amount'(test) {
        test.throws(() => lib_1.Duration.seconds(-1), /negative/);
        test.done();
    },
    'unresolved amount'(test) {
        const stack = new lib_1.Stack();
        const lazyDuration = lib_1.Duration.seconds(lib_1.Token.asNumber({ resolve: () => 1337 }));
        test.equals(stack.resolve(lazyDuration.toSeconds()), 1337);
        test.throws(() => stack.resolve(lazyDuration.toMinutes()), /Unable to perform time unit conversion on un-resolved token/);
        test.done();
    },
    'Duration in seconds'(test) {
        const duration = lib_1.Duration.seconds(300);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'300 seconds' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.seconds(60 * 60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in minutes'(test) {
        const duration = lib_1.Duration.minutes(5);
        test.equal(duration.toSeconds(), 300);
        test.equal(duration.toMinutes(), 5);
        test.throws(() => duration.toDays(), /'5 minutes' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 300 / 86400);
        test.equal(lib_1.Duration.minutes(60 * 24).toDays(), 1);
        test.done();
    },
    'Duration in hours'(test) {
        const duration = lib_1.Duration.hours(5);
        test.equal(duration.toSeconds(), 18000);
        test.equal(duration.toMinutes(), 300);
        test.throws(() => duration.toDays(), /'5 hours' cannot be converted into a whole number of days/);
        floatEqual(test, duration.toDays({ integral: false }), 5 / 24);
        test.equal(lib_1.Duration.hours(24).toDays(), 1);
        test.done();
    },
    'seconds to milliseconds'(test) {
        const duration = lib_1.Duration.seconds(5);
        test.equal(duration.toMilliseconds(), 5000);
        test.done();
    },
    'Duration in days'(test) {
        const duration = lib_1.Duration.days(1);
        test.equal(duration.toSeconds(), 86400);
        test.equal(duration.toMinutes(), 1440);
        test.equal(duration.toDays(), 1);
        test.done();
    },
    'toISOString'(test) {
        test.equal(lib_1.Duration.seconds(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.minutes(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.hours(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.days(0).toISOString(), 'PT0S');
        test.equal(lib_1.Duration.seconds(5).toISOString(), 'PT5S');
        test.equal(lib_1.Duration.minutes(5).toISOString(), 'PT5M');
        test.equal(lib_1.Duration.hours(5).toISOString(), 'PT5H');
        test.equal(lib_1.Duration.days(5).toISOString(), 'PT5D');
        test.equal(lib_1.Duration.seconds(1 + 60 * (1 + 60 * (1 + 24))).toISOString(), 'PT1D1H1M1S');
        test.done();
    },
    'parse'(test) {
        test.equal(lib_1.Duration.parse('PT0S').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0M').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0H').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT0D').toSeconds(), 0);
        test.equal(lib_1.Duration.parse('PT5S').toSeconds(), 5);
        test.equal(lib_1.Duration.parse('PT5M').toSeconds(), 300);
        test.equal(lib_1.Duration.parse('PT5H').toSeconds(), 18000);
        test.equal(lib_1.Duration.parse('PT5D').toSeconds(), 432000);
        test.equal(lib_1.Duration.parse('PT1D1H1M1S').toSeconds(), 1 + 60 * (1 + 60 * (1 + 24)));
        test.done();
    }
});
//# sourceMappingURL=data:application/json;base64,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