"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const lib_1 = require("../../lib");
const evaluate_cfn_1 = require("../evaluate-cfn");
const CFN_CONTEXT = {
    'AWS::Region': 'the_region',
    'AWS::AccountId': 'the_account',
    'AWS::URLSuffix': 'domain.aws',
};
let app;
let stack;
/**
 * Evaluate a possibly string-containing value the same way CFN would do
 *
 * (Be invariant to the specific Fn::Sub or Fn::Join we would output)
 */
function evalCFN(value) {
    return evaluate_cfn_1.evaluateCFN(stack.resolve(value), CFN_CONTEXT);
}
function isAssetManifest(x) {
    return x instanceof cxapi.AssetManifestArtifact;
}
module.exports = {
    'setUp'(cb) {
        app = new lib_1.App({
            context: {
                [cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT]: 'true',
            },
        });
        stack = new lib_1.Stack(app, 'Stack');
        cb();
    },
    'add file asset'(test) {
        // WHEN
        const location = stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.bucketName), 'cdk-bootstrap-hnb659fds-assets-the_account-the_region');
        test.equals(evalCFN(location.s3Url), 'https://s3.the_region.domain.aws/cdk-bootstrap-hnb659fds-assets-the_account-the_region/abcdef');
        // THEN - object key contains source hash somewhere
        test.ok(location.objectKey.indexOf('abcdef') > -1);
        test.done();
    },
    'add docker image asset'(test) {
        // WHEN
        const location = stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // THEN - we have a fixed asset location with region placeholders
        test.equals(evalCFN(location.repositoryName), 'cdk-bootstrap-hnb659fds-container-assets-the_account-the_region');
        test.equals(evalCFN(location.imageUri), 'the_account.dkr.ecr.the_region.domain.aws/cdk-bootstrap-hnb659fds-container-assets-the_account-the_region:abcdef');
        test.done();
    },
    'synthesis'(test) {
        // GIVEN
        stack.synthesizer.addFileAsset({
            fileName: __filename,
            packaging: lib_1.FileAssetPackaging.FILE,
            sourceHash: 'abcdef',
        });
        stack.synthesizer.addDockerImageAsset({
            directoryName: '.',
            sourceHash: 'abcdef',
        });
        // WHEN
        const asm = app.synth();
        // THEN - we have an asset manifest with both assets and the stack template in there
        const manifestArtifact = asm.artifacts.filter(isAssetManifest)[0];
        test.ok(manifestArtifact);
        const manifest = JSON.parse(fs.readFileSync(manifestArtifact.file, { encoding: 'utf-8' }));
        test.equals(Object.keys(manifest.files || {}).length, 2);
        test.equals(Object.keys(manifest.dockerImages || {}).length, 1);
        // THEN - every artifact has an assumeRoleArn
        for (const file of Object.values({ ...manifest.files, ...manifest.dockerImages })) {
            for (const destination of Object.values(file.destinations)) {
                test.ok(destination.assumeRoleArn);
            }
        }
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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