#!/usr/bin/env python3
"""
Command Line Interface for TMIN Package
"""

from .core import PIPE
import sys
import textwrap

def main():
    print("\nTMIN CLI - Pipe Thickness Analysis\n" + "="*40)
    # Professional disclaimer
    disclaimer = ("\033[1mDISCLAIMER\033[0m\n\n"
        "TMIN is an engineering analysis tool intended to assist with pipe thickness "
        "evaluation and retirement planning in petrochemical, processing, and refinery "
        "environments. \033[1mIt is not a stand-alone indicator of piping integrity.\033[0m "
        "All results and recommendations generated by TMIN must be interpreted and "
        "applied by qualified professionals with appropriate engineering judgment and field "
        "inspection expertise.\n\n"
        "This software does \033[1mnot\033[0m account for all possible factors affecting piping "
        "integrity, and its analysis may be limited by assumptions, data quality, or unmodeled "
        "conditions. A detailed design document is provided with this package, outlining the "
        "scope, assumptions, and limitations of TMIN. \033[1mUsers are strongly encouraged "
        "to review this document before relying on the results.\033[0m\n\n"
        "By using TMIN, you acknowledge that:\n"
        "- The software is a decision-support tool, not a substitute for professional engineering assessment.\n"
        "- Final decisions regarding piping integrity and retirement must be made by qualified personnel, considering all relevant data and site-specific conditions.\n"
        "- The developers and distributors of TMIN accept no liability for any damages or losses resulting from the use or misuse of this software.\n"
    )
    print(textwrap.fill(disclaimer, width=80, replace_whitespace=False, subsequent_indent='    '))
    
    try:
        # Get pipe parameters from user
        schedule = input("\nEnter pipe schedule (10, 40, 80, 120, 160): ").strip()
        nps = input("Enter nominal pipe size (e.g., '2', '3/4'): ").strip()
        pressure = float(input("Enter design pressure (psi): "))
        pressure_class = int(input("Enter pressure class (150, 300, 600, 900, 1500, 2500): "))
        metallurgy = input("Enter metallurgy (CS A106 GR B, SS 316/316S, SS 304, Inconel 625): ").strip()
        
        # Optional parameters
        corrosion_rate_input = input("Enter corrosion rate in MPY (optional, press Enter to skip): ").strip()
        corrosion_rate = float(corrosion_rate_input) if corrosion_rate_input else None
        
        default_retirement_limit_input = input("If there is a company issued default retirement limit for this pipe, enter here in inches (optional, press Enter to skip): ").strip()
        default_retirement_limit = float(default_retirement_limit_input) if default_retirement_limit_input else None
        
        measured_thickness = float(input("Enter measured thickness during inspection (inches): "))

        # Get inspection year for time-based corrosion calculation
        year_inspected_input = input("Enter year when thickness was measured (e.g., 2020, optional, press Enter to skip, the present year is assumed if skipped): ").strip()
        year_inspected = int(year_inspected_input) if year_inspected_input else None
        
        API_table = input("Enter which ASME B31.3 edition you would like to use (e.g. 2024 (latest), 2009):").strip()

        # Create pipe instance
        pipe = PIPE(
            schedule=schedule,
            nps=nps,
            pressure=pressure,
            pressure_class=pressure_class,
            metallurgy=metallurgy,
            corrosion_rate=corrosion_rate,
            default_retirement_limit=default_retirement_limit,
            API_table=API_table
        )
        
        # Generate full report
        print("\nGenerating analysis and reports...")
        report_files = pipe.generate_full_report(measured_thickness, year_inspected)
        
        print("\nAnalysis complete! Generated files:")
        for file_type, file_path in report_files.items():
            if file_type != "analysis_results":
                print(f"  {file_type}: {file_path}")
        
    except ValueError as e:
        print(f"Error: Invalid input - {e}")
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)

if __name__ == "__main__":
    main() 