"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NodejsFunction = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const bundling_1 = require("./bundling");
const util_1 = require("./util");
/**
 * A Node.js Lambda function bundled using esbuild.
 *
 * @stability stable
 */
class NodejsFunction extends lambda.Function {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c, _d, _e, _f;
        if (props.runtime && props.runtime.family !== lambda.RuntimeFamily.NODEJS) {
            throw new Error('Only `NODEJS` runtimes are supported.');
        }
        // Find lock file
        let depsLockFilePath;
        if (props.depsLockFilePath) {
            if (!fs.existsSync(props.depsLockFilePath)) {
                throw new Error(`Lock file at ${props.depsLockFilePath} doesn't exist`);
            }
            if (!fs.statSync(props.depsLockFilePath).isFile()) {
                throw new Error('`depsLockFilePath` should point to a file');
            }
            depsLockFilePath = path.resolve(props.depsLockFilePath);
        }
        else {
            const lockFile = (_b = util_1.findUp(util_1.LockFile.YARN)) !== null && _b !== void 0 ? _b : util_1.findUp(util_1.LockFile.NPM);
            if (!lockFile) {
                throw new Error('Cannot find a package lock file (`yarn.lock` or `package-lock.json`). Please specify it with `depsFileLockPath`.');
            }
            depsLockFilePath = lockFile;
        }
        // Entry and defaults
        const entry = path.resolve(findEntry(id, props.entry));
        const handler = (_c = props.handler) !== null && _c !== void 0 ? _c : 'handler';
        const defaultRunTime = util_1.nodeMajorVersion() >= 14
            ? lambda.Runtime.NODEJS_14_X
            : lambda.Runtime.NODEJS_12_X;
        const runtime = (_d = props.runtime) !== null && _d !== void 0 ? _d : defaultRunTime;
        super(scope, id, {
            ...props,
            runtime,
            code: bundling_1.Bundling.bundle({
                ...(_e = props.bundling) !== null && _e !== void 0 ? _e : {},
                entry,
                runtime,
                depsLockFilePath,
            }),
            handler: `index.${handler}`,
        });
        // Enable connection reuse for aws-sdk
        if ((_f = props.awsSdkConnectionReuse) !== null && _f !== void 0 ? _f : true) {
            this.addEnvironment('AWS_NODEJS_CONNECTION_REUSE_ENABLED', '1', { removeInEdge: true });
        }
    }
}
exports.NodejsFunction = NodejsFunction;
_a = JSII_RTTI_SYMBOL_1;
NodejsFunction[_a] = { fqn: "@aws-cdk/aws-lambda-nodejs.NodejsFunction", version: "1.99.0" };
/**
 * Searches for an entry file. Preference order is the following:
 * 1. Given entry file
 * 2. A .ts file named as the defining file with id as suffix (defining-file.id.ts)
 * 3. A .js file name as the defining file with id as suffix (defining-file.id.js)
 */
function findEntry(id, entry) {
    if (entry) {
        if (!/\.(jsx?|tsx?)$/.test(entry)) {
            throw new Error('Only JavaScript or TypeScript entry files are supported.');
        }
        if (!fs.existsSync(entry)) {
            throw new Error(`Cannot find entry file at ${entry}`);
        }
        return entry;
    }
    const definingFile = findDefiningFile();
    const extname = path.extname(definingFile);
    const tsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.ts`);
    if (fs.existsSync(tsHandlerFile)) {
        return tsHandlerFile;
    }
    const jsHandlerFile = definingFile.replace(new RegExp(`${extname}$`), `.${id}.js`);
    if (fs.existsSync(jsHandlerFile)) {
        return jsHandlerFile;
    }
    throw new Error('Cannot find entry file.');
}
/**
 * Finds the name of the file where the `NodejsFunction` is defined
 */
function findDefiningFile() {
    let definingIndex;
    const sites = util_1.callsites();
    for (const [index, site] of sites.entries()) {
        if (site.getFunctionName() === 'NodejsFunction') {
            // The next site is the site where the NodejsFunction was created
            definingIndex = index + 1;
            break;
        }
    }
    if (!definingIndex || !sites[definingIndex]) {
        throw new Error('Cannot find defining file.');
    }
    return sites[definingIndex].getFileName();
}
//# sourceMappingURL=data:application/json;base64,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