"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const route53 = require("@aws-cdk/aws-route53");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const targets = require("../lib");
const [recordName, zoneName] = ['foo', 'test.public'];
const bucketName = [recordName, zoneName].join('.');
test('use S3 bucket website as record target', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    const bucketWebsite = new s3.Bucket(stack, 'Bucket', { bucketName });
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName });
    new route53.ARecord(zone, 'Alias', {
        zone,
        recordName,
        target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
    });
    // THEN
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        AliasTarget: {
            DNSName: "s3-website-us-east-1.amazonaws.com",
            HostedZoneId: "Z3AQBSTGFYJSTF"
        },
    });
});
test('use S3 bucket website as record target (fromBucketName)', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'us-east-1' } });
    const bucketWebsite = s3.Bucket.fromBucketName(stack, 'Bucket', bucketName);
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName });
    new route53.ARecord(zone, 'Alias', {
        zone,
        recordName,
        target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
    });
    // THEN
    expect(stack).toHaveResource('AWS::Route53::RecordSet', {
        AliasTarget: {
            DNSName: "s3-website-us-east-1.amazonaws.com",
            HostedZoneId: "Z3AQBSTGFYJSTF"
        },
    });
});
test('throws if region agnostic', () => {
    // GIVEN
    const stack = new core_1.Stack();
    const bucketWebsite = new s3.Bucket(stack, 'Bucket', { bucketName });
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName });
    // THEN
    expect(() => {
        new route53.ARecord(zone, 'Alias', {
            zone,
            recordName,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
        });
    }).toThrow(/Cannot use an S3 record alias in region-agnostic stacks/);
});
test('throws if bucket website hosting is unavailable (cn-northwest-1)', () => {
    // GIVEN
    const app = new core_1.App();
    const stack = new core_1.Stack(app, 'test', { env: { region: 'cn-northwest-1' } });
    const bucketWebsite = new s3.Bucket(stack, 'Bucket');
    // WHEN
    const zone = new route53.PublicHostedZone(stack, 'HostedZone', { zoneName });
    // THEN
    expect(() => {
        new route53.ARecord(zone, 'Alias', {
            zone,
            recordName,
            target: route53.RecordTarget.fromAlias(new targets.BucketWebsiteTarget(bucketWebsite))
        });
    }).toThrow(/Bucket website target is not supported/);
});
//# sourceMappingURL=data:application/json;base64,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