import re
from itertools import zip_longest, chain
from docutils import nodes

class cjkspacer(nodes.General, nodes.Element):
    def __init__(self, spacer_str):
        self._spacer_str = spacer_str
        nodes.General.__init__(self)
        nodes.Element.__init__(self)

class CJKBoundary(object):
    CJK_CHARACTERS = r'\u2460-\u24FF\u2E80-\u2FFF\u3003-\u3007\u3012\u3013\u301C-\u30FA\u30FC-\u9FFF\uF900-\uFAFF\uFF00\uFF03-\uFF06\uFF0A\uFF0B\uFF0D\uFF10-\uFF19\uFF1C\uFF1D\uFF1E\uFF20-\uFF3A\uFF3E-\uFF5A\uFF5E\uFFE0-\uFFE6\U00020000-\U0002A6DF\U0002A700-\U0002EBEF\U0002F800-\U0002FA1F\U00030000-\U0003134F'
    BEFORE_EXCEPTIONS = ' \t\f\v\n\r' + r'({\[\u3000\u3001\u3002\u3008-\u3011\u3014-\u301B\u30FB\uFF01\uFF02\uFF07\uFF08\uFF09\uFF0C\uFF0E\uFF0F\uFF1A\uFF1B\uFF1F\uFF3B\uFF3C\uFF3D\uFF5B\uFF5C\uFF5D\uFF5F\uFF60'
    AFTER_EXCEPTIONS  = ' \t\f\v\n\r' + r')}\],.:;!?\u3000\u3001\u3002\u3008-\u3011\u3014-\u301B\u30FB\uFF01\uFF02\uFF07\uFF08\uFF09\uFF0C\uFF0E\uFF0F\uFF1A\uFF1B\uFF1F\uFF3B\uFF3C\uFF3D\uFF5B\uFF5C\uFF5D\uFF5F\uFF60'

    # The above expressions are generated by the following emacs-lisp code (See also README.md)
    # (setf cjk-list
    #       (concat (number-sequence  #x2E80  #x2FEF)
    #               (number-sequence  #x2F00  #x2FDF)
    #               (number-sequence  #x2FF0  #x2FFF)
    #               (number-sequence  #x3000  #x303F)
    #               (number-sequence  #x3040  #x309F)
    #               (number-sequence  #x30A0  #x30FF)
    #               (number-sequence  #x3100  #x312F)
    #               (number-sequence  #x3130  #x318F)
    #               (number-sequence  #x3190  #x319F)
    #               (number-sequence  #x31A0  #x31BF)
    #               (number-sequence  #x31C0  #x31EF)
    #               (number-sequence  #x31F0  #x31FF)
    #               (number-sequence  #x3200  #x32FF)
    #               (number-sequence  #x3300  #x33FF)
    #               (number-sequence  #x3400  #x4DBF)
    #               (number-sequence  #x4DC0  #x4DFF)
    #               (number-sequence  #x4E00  #x9FFF)
    #               (number-sequence  #xF900  #xFAFF)
    #               (number-sequence  #xFF00  #xFF60)
    #               (number-sequence  #xFFE0  #xFFE6)
    #               (number-sequence #x20000 #x2A6DF)
    #               (number-sequence #x2A700 #x2B73F)
    #               (number-sequence #x2B740 #x2B81F)
    #               (number-sequence #x2B820 #x2CEAF)
    #               (number-sequence #x2CEB0 #x2EBEF)
    #               (number-sequence #x2F800 #x2FA1F)
    #               (number-sequence #x30000 #x3134F)
    #               (number-sequence  #x2460  #x24FF)))
    # (setf cjk-exception-list
    #       (list #x3000 #x3001 #x3002 #x3008 #x3009 #x300A #x300B #x300C
    #             #x300D #x300E #x300F #x3010 #x3011 #x3014 #x3015 #x3016
    #             #x3017 #x3018 #x3019 #x301A #x301B
    #             #x30FB
    #             #xFF01 #xFF02 #xFF07 #xFF08 #xFF09 #xFF0C #xFF0E #xFF0F
    #             #xFF1A #xFF1B #xFF1F #xFF3B #xFF3C #xFF3D #xFF5B #xFF5C
    #             #xFF5D #xFF5F #xFF60))
    # (setf regexp-cjk-spacer
    #       (regexp-opt-charset (seq-difference cjk-list cjk-exception-list)))
    # (setf regexp-cjk-spacer-py
    #       (apply #'concat
    #              (apply #'append (seq-map (lambda (x)
    #                                         (cond ((<= 0 x #x7F)
    #                                                (char-to-string x))
    #                                               ((<= x #xFFFF)
    #                                                (format "\\u%X" x))
    #                                               (:else
    #                                                (format "\\U%08X" x))))
    #                                       regexp-cjk-spacer)
    #                     '())))
    # (setf regexp-cjk-spacer-exception
    #       (regexp-opt-charset cjk-exception-list))
    # (setf regexp-cjk-spacer-exception-py
    #       (apply #'concat
    #              (apply #'append (seq-map (lambda (x)
    #                                         (cond ((<= 0 x #x7F)
    #                                                (char-to-string x))
    #                                               ((<= x #xFFFF)
    #                                                (format "\\u%X" x))
    #                                               (:else
    #                                                (format "\\U%08X" x))))
    #                                       regexp-cjk-spacer-exception)
    #                     '())))
    
    def __init__(self, cjk_pattern, b_except_pattern, a_except_pattern):
        self._border_pattern_pairs = (
            (re.compile('(?:[^{b_except}])$'.format(b_except=b_except_pattern + cjk_pattern)),
             re.compile('^(?:[{cjk}])'.format(cjk=cjk_pattern))),
            (re.compile('(?:[{cjk}])$'.format(cjk=cjk_pattern)),
             re.compile('^(?:[^{a_except}])'.format(a_except=a_except_pattern + cjk_pattern)))
        )
        self._pattern = re.compile('(?<![{b_except}])(?=[{cjk}])|(?<=[{cjk}])(?![{a_except}])'.format(
            cjk=cjk_pattern,
            b_except=b_except_pattern + cjk_pattern,
            a_except=a_except_pattern + cjk_pattern)
        )

    def split(self, txt):
        return self._pattern.split(txt)        

    def check_boundary(self, leading_txt, following_txt):
        return any((pat[0].search(leading_txt) and pat[1].match(following_txt))
                   for pat in self._border_pattern_pairs)

class CJKSpacerVisitor(nodes.GenericNodeVisitor):
    EXCLUDED_ELEMENTS = (
        nodes.FixedTextElement, nodes.Inline,
        nodes.Invisible, nodes.Bibliographic
    )

    def __init__(self, document, spacer, cjk_boundary):
        super(CJKSpacerVisitor, self).__init__(document)
        self._spacer = spacer
        self._cjk_boundary = cjk_boundary

    def default_visit(self, node):
        if not isinstance(node, nodes.TextElement):
            return
        if isinstance(node, CJKSpacerVisitor.EXCLUDED_ELEMENTS):
            return
        self._insert_cjkspacer(node)
        raise nodes.SkipChildren

    def unknown_visit(self, node):
        self.default_visit(node)

    def _insert_cjkspacer(self, node):
        target_inline_elems = (nodes.emphasis, nodes.strong, nodes.inline, nodes.reference)
        num_children = len(node.children)
        new_children = []
        for idx, child in enumerate(node.children):
            if not isinstance(child, nodes.Text):
                if isinstance(child, target_inline_elems):
                    self._insert_cjkspacer(child)
                new_children.append(child)
                continue
            new_txt = child.astext()
            new_child = list(chain.from_iterable(zip_longest((nodes.Text(sep)
                                                              for sep in self._cjk_boundary.split(new_txt) if sep),
                                                             [], fillvalue=self._spacer)))[:-1]
            if len(new_children) > 0:
                prev_txt = new_children[-1].astext()
                if (not prev_txt
                    and isinstance(new_children[-1], nodes.target)
                    and len(new_children) > 1):
                    prev_txt = new_children[-2].astext()
                if prev_txt and self._cjk_boundary.check_boundary(prev_txt, new_txt):
                    new_child = [self._spacer] + new_child
            if idx + 1 < num_children:
                next_txt = node.children[idx + 1].astext()
                if next_txt and self._cjk_boundary.check_boundary(new_txt, next_txt):
                    new_child = new_child + [self._spacer]
            new_children += new_child
        node.children = new_children

def get_bool_value(config, builder_name):
    if not isinstance(config, (list, tuple)):
        return config
    return builder_name in config

def insert_cjkspacer(app, doctree, _docname):
    if app.builder.format not in app.config.cjkspacer_spacer:
        return
    spacer_str = app.config.cjkspacer_spacer[app.builder.format]
    visitor = CJKSpacerVisitor(doctree,
                               cjkspacer(spacer_str),
                               CJKBoundary(app.config.cjkspacer_cjk_characters,
                                           app.config.cjkspacer_before_exceptions,
                                           app.config.cjkspacer_after_exceptions))
    doctree.walk(visitor)

def visit_cjkspacer(self, node):
    self.body.append(node._spacer_str)
    
def depart_cjkspacer(self, node):
    pass

def cjkspacer_init(app):
    app.add_node(cjkspacer,
                 **{format:(visit_cjkspacer, depart_cjkspacer)
                     for format in app.config.cjkspacer_spacer.keys()})

def setup(app):
    app.add_config_value('cjkspacer_spacer', {'html':'<span class="cjkspacer"></span>'}, 'env', dict)
    app.add_config_value('cjkspacer_cjk_characters',   CJKBoundary.CJK_CHARACTERS,   'env', str)
    app.add_config_value('cjkspacer_before_exceptions', CJKBoundary.BEFORE_EXCEPTIONS, 'env', str)
    app.add_config_value('cjkspacer_after_exceptions',  CJKBoundary.AFTER_EXCEPTIONS,  'env', str)
    # formats, dict
    
    app.connect('builder-inited', cjkspacer_init)
    app.connect("doctree-resolved", insert_cjkspacer)
    
    return {'parallel_read_safe': True, 'parallel_write_safe': True}

